package com.rajat.pdfviewer.compose

import android.content.Context
import android.net.Uri
import android.os.ParcelFileDescriptor
import androidx.compose.runtime.*
import androidx.compose.ui.platform.LocalContext
import com.rajat.pdfviewer.HeaderData
import com.rajat.pdfviewer.PdfDownloader
import com.rajat.pdfviewer.PdfRendererCore
import com.rajat.pdfviewer.util.CacheStrategy
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withContext
import java.io.File
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

@Composable
fun rememberPdfRendererCore(
    file: File,
    cacheStrategy: CacheStrategy = CacheStrategy.MAXIMIZE_PERFORMANCE
): PdfRendererCore {
    val context = LocalContext.current
    val fileDescriptor = remember(file) {
        PdfRendererCore.getFileDescriptor(file)
    }
    return remember(file.absolutePath) {
        PdfRendererCore(
            context,
            fileDescriptor,
            PdfRendererCore.getCacheIdentifierFromFile(file),
            cacheStrategy
        )
    }
}

@Composable
fun rememberPdfRendererCoreFromUri(
    uri: Uri,
    cacheStrategy: CacheStrategy = CacheStrategy.MAXIMIZE_PERFORMANCE
): PdfRendererCore? {
    val context = LocalContext.current
    return remember(uri) {
        val fileDescriptor =
            context.contentResolver.openFileDescriptor(uri, "r") ?: return@remember null
        val cacheIdentifier = uri.toString().hashCode().toString()
        PdfRendererCore(context, fileDescriptor, cacheIdentifier, cacheStrategy)
    }
}

@Composable
fun rememberPdfRendererCoreFromUrl(
    url: String,
    headers: HeaderData = HeaderData(),
    cacheStrategy: CacheStrategy = CacheStrategy.MAXIMIZE_PERFORMANCE,
    onError: (Throwable) -> Unit = {},
    onProgress: (downloaded: Long, total: Long?) -> Unit = { _, _ -> }
): State<PdfRendererCore?> {
    val context = LocalContext.current
    return produceState<PdfRendererCore?>(initialValue = null, key1 = url) {
        try {
            val file = downloadPdfFile(
                context = context,
                url = url,
                headers = headers,
                cacheStrategy = cacheStrategy,
                onProgress = onProgress
            )
            val fileDescriptor: ParcelFileDescriptor = PdfRendererCore.getFileDescriptor(file)
            value = PdfRendererCore(
                context,
                fileDescriptor,
                PdfRendererCore.getCacheIdentifierFromFile(file),
                cacheStrategy
            )
        } catch (e: Throwable) {
            onError(e)
        }
    }
}

suspend fun downloadPdfFile(
    context: Context,
    url: String,
    headers: HeaderData = HeaderData(),
    cacheStrategy: CacheStrategy = CacheStrategy.MAXIMIZE_PERFORMANCE,
    onProgress: (downloaded: Long, total: Long?) -> Unit = { _, _ -> }
): File = suspendCancellableCoroutine { cont ->

    PdfDownloader(
        coroutineScope = kotlinx.coroutines.CoroutineScope(Dispatchers.IO),
        headers = headers,
        url = url,
        cacheStrategy = cacheStrategy,
        listener = object : PdfDownloader.StatusListener {
            override fun getContext(): Context = context

            override fun onDownloadStart() {}

            override fun onDownloadProgress(currentBytes: Long, totalBytes: Long) {
                onProgress(currentBytes, totalBytes)
            }

            override fun onDownloadSuccess(downloadedFile: File) {
                cont.resume(downloadedFile)
            }

            override fun onError(error: Throwable) {
                cont.resumeWithException(error)
            }
        }
    )
}