package com.rajat.pdfviewer.compose

import androidx.compose.foundation.layout.*
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import com.rajat.pdfviewer.HeaderData
import com.rajat.pdfviewer.util.CacheStrategy

@Composable
fun PdfViewerScreen(
    modifier: Modifier = Modifier,
    url: String,
    headers: HeaderData = HeaderData()
) {
    var progress by remember { mutableStateOf<Pair<Long, Long?>>(0L to null) }
    var error by remember { mutableStateOf<Throwable?>(null) }

    val rendererState = rememberPdfRendererCoreFromUrl(
        url = url,
        headers = headers,
        cacheStrategy = CacheStrategy.MAXIMIZE_PERFORMANCE,
        onError = { error = it },
        onProgress = { current, total -> progress = current to total }
    )

    Box(modifier = modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
        when {
            error != null -> {
                Text(
                    text = "Error loading PDF: ${error?.localizedMessage ?: "Unknown error"}",
                    color = MaterialTheme.colorScheme.error,
                    textAlign = TextAlign.Center,
                    modifier = Modifier.padding(24.dp)
                )
            }
            rendererState.value == null -> {
                Column(horizontalAlignment = Alignment.CenterHorizontally) {
                    CircularProgressIndicator()
                    Spacer(modifier = Modifier.height(12.dp))
                    Text("Downloading PDF... ${'$'}{progress.first / 1024} KB")
                }
            }
            else -> {
                PdfDocumentViewer(renderer = rendererState.value!!)
            }
        }
    }
}
