/*
 * Decompiled with CFR 0.152.
 */
package io.fusionauth.jwks;

import io.fusionauth.der.DerInputStream;
import io.fusionauth.der.DerValue;
import io.fusionauth.jwks.JSONWebKeyBuilderException;
import io.fusionauth.jwks.domain.JSONWebKey;
import io.fusionauth.jwt.JWTUtils;
import io.fusionauth.jwt.domain.Algorithm;
import io.fusionauth.jwt.domain.KeyType;
import io.fusionauth.pem.domain.PEM;
import java.io.IOException;
import java.math.BigInteger;
import java.security.Key;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECPoint;
import java.util.Arrays;
import java.util.Base64;
import java.util.Objects;

public class JSONWebKeyBuilder {
    public JSONWebKey build(String encodedPEM) {
        Objects.requireNonNull(encodedPEM);
        PEM pem = PEM.decode(encodedPEM);
        if (pem.privateKey != null) {
            return this.build(pem.privateKey);
        }
        if (pem.publicKey != null) {
            return this.build(pem.publicKey);
        }
        if (pem.certificate != null) {
            return this.build(pem.certificate);
        }
        throw new JSONWebKeyBuilderException("The provided PEM did not contain a public or private key.");
    }

    public JSONWebKey build(PrivateKey privateKey) {
        RSAPrivateKey rsaPrivateKey;
        Objects.requireNonNull(privateKey);
        JSONWebKey key = new JSONWebKey();
        key.kty = this.getKeyType(privateKey);
        key.use = "sig";
        if (privateKey instanceof RSAPrivateKey) {
            rsaPrivateKey = (RSAPrivateKey)privateKey;
            key.n = this.base64EncodeUint(rsaPrivateKey.getModulus());
            key.d = this.base64EncodeUint(rsaPrivateKey.getPrivateExponent());
        }
        if (privateKey instanceof RSAPrivateCrtKey) {
            rsaPrivateKey = (RSAPrivateCrtKey)privateKey;
            key.e = this.base64EncodeUint(rsaPrivateKey.getPublicExponent());
            key.p = this.base64EncodeUint(rsaPrivateKey.getPrimeP());
            key.q = this.base64EncodeUint(rsaPrivateKey.getPrimeQ());
            key.qi = this.base64EncodeUint(rsaPrivateKey.getCrtCoefficient());
            Object dp = rsaPrivateKey.getPrivateExponent().mod(rsaPrivateKey.getPrimeP().subtract(BigInteger.valueOf(1L)));
            BigInteger dq = rsaPrivateKey.getPrivateExponent().mod(rsaPrivateKey.getPrimeQ().subtract(BigInteger.valueOf(1L)));
            key.dp = this.base64EncodeUint((BigInteger)dp);
            key.dq = this.base64EncodeUint(dq);
        }
        if (privateKey instanceof ECPrivateKey) {
            ECPrivateKey ecPrivateKey = (ECPrivateKey)privateKey;
            key.crv = this.getCurveOID(privateKey);
            if (key.crv != null) {
                switch (key.crv) {
                    case "P-256": {
                        key.alg = Algorithm.ES256;
                        break;
                    }
                    case "P-384": {
                        key.alg = Algorithm.ES384;
                        break;
                    }
                    case "P-521": {
                        key.alg = Algorithm.ES512;
                    }
                }
            }
            int byteLength = this.getCoordinateLength(ecPrivateKey);
            key.d = this.base64EncodeUint(ecPrivateKey.getS(), byteLength);
            key.x = this.base64EncodeUint(ecPrivateKey.getParams().getGenerator().getAffineX(), byteLength);
            key.y = this.base64EncodeUint(ecPrivateKey.getParams().getGenerator().getAffineY(), byteLength);
        }
        return key;
    }

    public JSONWebKey build(PublicKey publicKey) {
        Objects.requireNonNull(publicKey);
        JSONWebKey key = new JSONWebKey();
        key.kty = this.getKeyType(publicKey);
        key.use = "sig";
        if (publicKey instanceof RSAPublicKey) {
            RSAPublicKey rsaPublicKey = (RSAPublicKey)publicKey;
            key.e = this.base64EncodeUint(rsaPublicKey.getPublicExponent());
            key.n = this.base64EncodeUint(rsaPublicKey.getModulus());
        }
        if (key.kty == KeyType.EC) {
            ECPublicKey ecPublicKey = (ECPublicKey)publicKey;
            key.crv = this.getCurveOID(ecPublicKey);
            ECPoint point = ecPublicKey.getW();
            int length = point.getAffineX().toByteArray().length;
            key.alg = Algorithm.ES256;
            if (length >= 63) {
                key.alg = Algorithm.ES512;
            } else if (length >= 47) {
                key.alg = Algorithm.ES384;
            }
            int byteLength = this.getCoordinateLength(ecPublicKey);
            key.x = this.base64EncodeUint(ecPublicKey.getW().getAffineX(), byteLength);
            key.y = this.base64EncodeUint(ecPublicKey.getW().getAffineY(), byteLength);
        }
        return key;
    }

    public JSONWebKey build(Certificate certificate) {
        Objects.requireNonNull(certificate);
        JSONWebKey key = this.build(certificate.getPublicKey());
        if (certificate instanceof X509Certificate) {
            key.alg = Algorithm.fromName(((X509Certificate)certificate).getSigAlgName());
            try {
                String encodedCertificate = new String(Base64.getEncoder().encode(certificate.getEncoded()));
                key.x5t = JWTUtils.generateJWS_x5t(encodedCertificate);
                key.x5t_256 = JWTUtils.generateJWS_x5t("SHA-256", encodedCertificate);
            }
            catch (CertificateEncodingException e) {
                throw new JSONWebKeyBuilderException("Failed to decode X.509 certificate", e);
            }
        }
        return key;
    }

    private String base64EncodeUint(BigInteger value, int minimumLength) {
        if (value.signum() < 0) {
            throw new JSONWebKeyBuilderException("Illegal parameter, cannot encode a negative number.", new IllegalArgumentException());
        }
        byte[] bytes = value.toByteArray();
        if (value.bitLength() % 8 == 0 && bytes[0] == 0 && bytes.length > 1) {
            bytes = Arrays.copyOfRange(bytes, 1, bytes.length);
        }
        if (minimumLength != -1 && bytes.length < minimumLength) {
            byte[] buf = new byte[minimumLength];
            System.arraycopy(bytes, 0, buf, minimumLength - bytes.length, bytes.length);
            bytes = buf;
        }
        return Base64.getUrlEncoder().withoutPadding().encodeToString(bytes);
    }

    private String base64EncodeUint(BigInteger value) {
        return this.base64EncodeUint(value, -1);
    }

    private int getCoordinateLength(ECKey key) {
        return (int)Math.ceil((double)key.getParams().getCurve().getField().getFieldSize() / 8.0);
    }

    private KeyType getKeyType(Key key) {
        if (key.getAlgorithm().equals("RSA")) {
            return KeyType.RSA;
        }
        if (key.getAlgorithm().equals("EC")) {
            return KeyType.EC;
        }
        return null;
    }

    private String readCurveObjectIdentifier(Key key) {
        try {
            if (key instanceof PrivateKey) {
                DerValue[] sequence = new DerInputStream(key.getEncoded()).getSequence();
                sequence[1].getOID();
                return sequence[1].getOID().decode();
            }
            DerValue[] sequence = new DerInputStream(key.getEncoded()).getSequence();
            sequence[0].getOID();
            return sequence[0].getOID().decode();
        }
        catch (IOException e) {
            throw new JSONWebKeyBuilderException("Unable to read the Object Identifier of the public key.", e);
        }
    }

    String getCurveOID(Key key) {
        String oid;
        switch (oid = this.readCurveObjectIdentifier(key)) {
            case "1.2.840.10045.3.1.7": {
                return "P-256";
            }
            case "1.3.132.0.34": {
                return "P-384";
            }
            case "1.3.132.0.35": {
                return "P-521";
            }
        }
        return null;
    }
}

