/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.dsl.internal;

import io.fabric8.kubernetes.api.model.HasMetadata;
import io.fabric8.kubernetes.api.model.KubernetesResource;
import io.fabric8.kubernetes.api.model.KubernetesResourceList;
import io.fabric8.kubernetes.api.model.ListOptions;
import io.fabric8.kubernetes.api.model.Status;
import io.fabric8.kubernetes.api.model.WatchEvent;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.WatcherException;
import io.fabric8.kubernetes.client.dsl.base.BaseOperation;
import io.fabric8.kubernetes.client.dsl.base.OperationSupport;
import io.fabric8.kubernetes.client.dsl.internal.AbstractWatchManager;
import io.fabric8.kubernetes.client.utils.HttpClientUtils;
import io.fabric8.kubernetes.client.utils.Serialization;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.logging.HttpLoggingInterceptor;
import okio.BufferedSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WatchHTTPManager<T extends HasMetadata, L extends KubernetesResourceList<T>>
extends AbstractWatchManager<T> {
    private static final Logger logger = LoggerFactory.getLogger(WatchHTTPManager.class);
    private final BaseOperation<T, L, ?> baseOperation;
    private final AtomicBoolean reconnectPending = new AtomicBoolean(false);
    private final URL requestUrl;

    public WatchHTTPManager(OkHttpClient client, BaseOperation<T, L, ?> baseOperation, ListOptions listOptions, Watcher<T> watcher, int reconnectInterval, int reconnectLimit, long connectTimeout) throws MalformedURLException {
        this(client, baseOperation, listOptions, watcher, reconnectInterval, reconnectLimit, connectTimeout, 5);
    }

    public WatchHTTPManager(OkHttpClient client, BaseOperation<T, L, ?> baseOperation, ListOptions listOptions, Watcher<T> watcher, int reconnectInterval, int reconnectLimit, long connectTimeout, int maxIntervalExponent) throws MalformedURLException {
        super(watcher, listOptions, reconnectLimit, reconnectInterval, maxIntervalExponent, client.newBuilder().connectTimeout(connectTimeout, TimeUnit.MILLISECONDS).readTimeout(0L, TimeUnit.MILLISECONDS).cache(null).build());
        this.baseOperation = baseOperation;
        for (Interceptor i : this.clonedClient.networkInterceptors()) {
            if (!(i instanceof HttpLoggingInterceptor)) continue;
            HttpLoggingInterceptor interceptor = (HttpLoggingInterceptor)i;
            interceptor.setLevel(HttpLoggingInterceptor.Level.BASIC);
        }
        this.requestUrl = baseOperation.getNamespacedUrl();
        this.runWatch();
    }

    private void runWatch() {
        logger.debug("Watching via HTTP GET ... {}", (Object)this);
        HttpUrl.Builder httpUrlBuilder = HttpUrl.get((URL)this.requestUrl).newBuilder();
        String labelQueryParam = this.baseOperation.getLabelQueryParam();
        if (Utils.isNotNullOrEmpty(labelQueryParam)) {
            httpUrlBuilder.addQueryParameter("labelSelector", labelQueryParam);
        }
        String fieldQueryString = this.baseOperation.getFieldQueryParam();
        String name = this.baseOperation.getName();
        if (name != null && name.length() > 0) {
            if (this.baseOperation.isApiGroup()) {
                httpUrlBuilder.addPathSegment(name);
            } else {
                if (fieldQueryString.length() > 0) {
                    fieldQueryString = fieldQueryString + ",";
                }
                fieldQueryString = fieldQueryString + "metadata.name=" + name;
            }
        }
        if (Utils.isNotNullOrEmpty(fieldQueryString)) {
            httpUrlBuilder.addQueryParameter("fieldSelector", fieldQueryString);
        }
        this.listOptions.setResourceVersion((String)this.resourceVersion.get());
        HttpClientUtils.appendListOptionParams(httpUrlBuilder, this.listOptions);
        String origin = this.requestUrl.getProtocol() + "://" + this.requestUrl.getHost();
        if (this.requestUrl.getPort() != -1) {
            origin = origin + ":" + this.requestUrl.getPort();
        }
        final Request request = new Request.Builder().get().url(httpUrlBuilder.build()).addHeader("Origin", origin).build();
        this.clonedClient.newCall(request).enqueue(new Callback(){

            public void onFailure(Call call, IOException e) {
                logger.info("Watch connection failed. reason: {}", (Object)e.getMessage());
                WatchHTTPManager.this.scheduleReconnect();
            }

            public void onResponse(Call call, Response response) throws IOException {
                if (!response.isSuccessful()) {
                    throw OperationSupport.requestFailure(request, OperationSupport.createStatus(response.code(), response.message()));
                }
                try {
                    BufferedSource source = response.body().source();
                    while (!source.exhausted()) {
                        String message = source.readUtf8LineStrict();
                        WatchHTTPManager.this.onMessage(message);
                    }
                }
                catch (Exception e) {
                    logger.info("Watch terminated unexpectedly. reason: {}", (Object)e.getMessage());
                }
                if (response != null) {
                    response.body().close();
                }
                WatchHTTPManager.this.scheduleReconnect();
            }
        });
    }

    private void scheduleReconnect() {
        if (this.forceClosed.get()) {
            logger.warn("Ignoring error for already closed/closing connection");
            return;
        }
        if (this.cannotReconnect()) {
            this.watcher.onClose(new WatcherException("Connection unexpectedly closed"));
            return;
        }
        logger.debug("Submitting reconnect task to the executor");
        this.submit(() -> {
            if (!this.reconnectPending.compareAndSet(false, true)) {
                logger.debug("Reconnect already scheduled");
                return;
            }
            try {
                logger.debug("Scheduling reconnect task");
                this.schedule(() -> {
                    try {
                        this.runWatch();
                        this.reconnectPending.set(false);
                    }
                    catch (Exception e) {
                        logger.error("Exception in reconnect", (Throwable)e);
                        this.close();
                        this.watcher.onClose(new WatcherException("Unhandled exception in reconnect attempt", e));
                    }
                }, this.nextReconnectInterval(), TimeUnit.MILLISECONDS);
            }
            catch (RejectedExecutionException e) {
                if (!this.forceClosed.get()) {
                    logger.error("Exception in reconnect", (Throwable)e);
                }
                this.reconnectPending.set(false);
            }
        });
    }

    public void onMessage(String messageSource) {
        try {
            WatchEvent event = WatchHTTPManager.readWatchEvent(messageSource);
            KubernetesResource object = event.getObject();
            if (object instanceof HasMetadata) {
                HasMetadata obj = (HasMetadata)object;
                this.resourceVersion.set(obj.getMetadata().getResourceVersion());
                Watcher.Action action = Watcher.Action.valueOf(event.getType());
                this.watcher.eventReceived(action, obj);
            } else if (object instanceof KubernetesResourceList) {
                KubernetesResourceList list = (KubernetesResourceList)((Object)object);
                this.resourceVersion.set(list.getMetadata().getResourceVersion());
                Watcher.Action action = Watcher.Action.valueOf(event.getType());
                List items = list.getItems();
                if (items != null) {
                    for (HasMetadata item : items) {
                        this.watcher.eventReceived(action, item);
                    }
                }
            } else if (object instanceof Status) {
                Status status = (Status)object;
                if (status.getCode() == 410) {
                    this.close();
                    this.watcher.onClose(new WatcherException(status.getMessage(), new KubernetesClientException(status)));
                    return;
                }
                this.watcher.eventReceived(Watcher.Action.ERROR, null);
                logger.error("Error received: {}", (Object)status);
            } else {
                logger.error("Unknown message received: {}", (Object)messageSource);
            }
        }
        catch (ClassCastException e) {
            logger.error("Received wrong type of object for watch", (Throwable)e);
        }
        catch (IllegalArgumentException e) {
            logger.error("Invalid event type", (Throwable)e);
        }
    }

    protected static WatchEvent readWatchEvent(String messageSource) {
        WatchEvent event = Serialization.unmarshal(messageSource, WatchEvent.class);
        KubernetesResource object = null;
        if (event != null) {
            object = event.getObject();
        }
        if (object == null) {
            object = Serialization.unmarshal(messageSource, KubernetesResource.class);
            if (event == null) {
                event = new WatchEvent(object, "MODIFIED");
            } else {
                event.setObject(object);
            }
        }
        if (event.getType() == null) {
            event.setType("MODIFIED");
        }
        return event;
    }

    @Override
    public void close() {
        logger.debug("Force closing the watch {}", (Object)this);
        this.closeEvent();
        this.closeExecutorService();
    }
}

