/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.internal;

import io.fabric8.kubernetes.client.internal.PKCS1Util;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertUtils {
    private static final Logger LOG = LoggerFactory.getLogger(CertUtils.class);
    private static final String TRUST_STORE_SYSTEM_PROPERTY = "javax.net.ssl.trustStore";
    private static final String TRUST_STORE_PASSWORD_SYSTEM_PROPERTY = "javax.net.ssl.trustStorePassword";
    private static final String TRUST_STORE_TYPE_SYSTEM_PROPERTY = "javax.net.ssl.trustStoreType";
    private static final String KEY_STORE_SYSTEM_PROPERTY = "javax.net.ssl.keyStore";
    private static final String KEY_STORE_PASSWORD_SYSTEM_PROPERTY = "javax.net.ssl.keyStorePassword";
    private static final String KEY_STORE_DEFAULT_PASSWORD = "changeit";

    private CertUtils() {
    }

    public static ByteArrayInputStream getInputStreamFromDataOrFile(String data, String file) throws IOException {
        if (data != null) {
            return CertUtils.createInputStreamFromBase64EncodedString(data);
        }
        return new ByteArrayInputStream(new String(Files.readAllBytes(Paths.get(file, new String[0]))).trim().getBytes());
    }

    public static KeyStore createTrustStore(String caCertData, String caCertFile, String trustStoreFile, String trustStorePassphrase) throws IOException, CertificateException, KeyStoreException, NoSuchAlgorithmException {
        ByteArrayInputStream pemInputStream = CertUtils.getInputStreamFromDataOrFile(caCertData, caCertFile);
        KeyStore trustStore = CertUtils.loadTrustStore(trustStoreFile, CertUtils.getPassphrase(TRUST_STORE_PASSWORD_SYSTEM_PROPERTY, trustStorePassphrase));
        return CertUtils.mergePemCertsIntoTrustStore(pemInputStream, trustStore, true);
    }

    static KeyStore loadTrustStore(String trustStoreFile, char[] trustStorePassphrase) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException, FileNotFoundException {
        String trustStoreType = System.getProperty(TRUST_STORE_TYPE_SYSTEM_PROPERTY, KeyStore.getDefaultType());
        KeyStore trustStore = KeyStore.getInstance(trustStoreType);
        if (Utils.isNotNullOrEmpty(trustStoreFile)) {
            try (FileInputStream fis = new FileInputStream(trustStoreFile);){
                trustStore.load(fis, trustStorePassphrase);
            }
        } else {
            CertUtils.loadDefaultTrustStoreFile(trustStore, trustStorePassphrase);
        }
        return trustStore;
    }

    static KeyStore mergePemCertsIntoTrustStore(ByteArrayInputStream pemInputStream, KeyStore trustStore, boolean first) throws CertificateException, KeyStoreException {
        CertificateFactory certFactory = CertificateFactory.getInstance("X509");
        while (pemInputStream.available() > 0) {
            X509Certificate cert;
            try {
                cert = (X509Certificate)certFactory.generateCertificate(pemInputStream);
            }
            catch (CertificateException e) {
                if (pemInputStream.available() > 0) {
                    throw e;
                }
                LOG.debug("The trailing entry generated a certificate exception.  More than likely the contents end with comments.", e);
                break;
            }
            try {
                String alias = cert.getSubjectX500Principal().getName() + "_" + cert.getSerialNumber().toString(16);
                trustStore.setCertificateEntry(alias, cert);
                first = false;
            }
            catch (KeyStoreException e) {
                if (first) {
                    pemInputStream.reset();
                    KeyStore writableStore = KeyStore.getInstance("PKCS12");
                    try {
                        writableStore.load(null, null);
                    }
                    catch (IOException | NoSuchAlgorithmException | CertificateException e1) {
                        throw e;
                    }
                    for (String alias : Collections.list(trustStore.aliases())) {
                        writableStore.setCertificateEntry(alias, trustStore.getCertificate(alias));
                    }
                    return CertUtils.mergePemCertsIntoTrustStore(pemInputStream, writableStore, false);
                }
                throw e;
            }
        }
        return trustStore;
    }

    public static KeyStore createKeyStore(InputStream certInputStream, InputStream keyInputStream, String clientKeyAlgo, char[] clientKeyPassphrase, String keyStoreFile, char[] keyStorePassphrase) throws IOException, CertificateException, NoSuchAlgorithmException, InvalidKeySpecException, KeyStoreException {
        CertificateFactory certFactory = CertificateFactory.getInstance("X509");
        Collection<? extends Certificate> certificates = certFactory.generateCertificates(certInputStream);
        PrivateKey privateKey = CertUtils.loadKey(keyInputStream, clientKeyAlgo);
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        if (Utils.isNotNullOrEmpty(keyStoreFile)) {
            try (FileInputStream fis = new FileInputStream(keyStoreFile);){
                keyStore.load(fis, keyStorePassphrase);
            }
        } else {
            CertUtils.loadDefaultKeyStoreFile(keyStore, keyStorePassphrase);
        }
        String alias = certificates.stream().map(cert -> ((X509Certificate)cert).getIssuerX500Principal().getName()).collect(Collectors.joining("_"));
        keyStore.setKeyEntry(alias, privateKey, clientKeyPassphrase, certificates.toArray(new Certificate[0]));
        return keyStore;
    }

    private static PrivateKey loadKey(InputStream keyInputStream, String clientKeyAlgo) throws IOException, InvalidKeySpecException, NoSuchAlgorithmException {
        if (clientKeyAlgo == null) {
            clientKeyAlgo = "RSA";
        }
        byte[] keyBytes = CertUtils.decodePem(keyInputStream);
        if (clientKeyAlgo.equals("EC") || clientKeyAlgo.equals("RSA")) {
            try {
                return CertUtils.handleOtherKeys(keyBytes, clientKeyAlgo);
            }
            catch (IOException e) {
                if (clientKeyAlgo.equals("EC")) {
                    return CertUtils.handleECKey(keyBytes);
                }
                throw e;
            }
        }
        throw new InvalidKeySpecException("Unknown type of PKCS8 Private Key, tried RSA and ECDSA");
    }

    private static PrivateKey handleECKey(byte[] keyBytes) throws IOException, InvalidKeySpecException, NoSuchAlgorithmException {
        return KeyFactory.getInstance("EC").generatePrivate(PKCS1Util.getECKeySpec(keyBytes));
    }

    private static PrivateKey handleOtherKeys(byte[] keyBytes, String clientKeyAlgo) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        try {
            return KeyFactory.getInstance(clientKeyAlgo).generatePrivate(new PKCS8EncodedKeySpec(keyBytes));
        }
        catch (InvalidKeySpecException e) {
            RSAPrivateCrtKeySpec keySpec = PKCS1Util.decodePKCS1(keyBytes);
            return KeyFactory.getInstance(clientKeyAlgo).generatePrivate(keySpec);
        }
    }

    private static void loadDefaultTrustStoreFile(KeyStore keyStore, char[] trustStorePassphrase) throws CertificateException, NoSuchAlgorithmException, IOException {
        File trustStoreFile = CertUtils.getDefaultTrustStoreFile();
        if (!CertUtils.loadDefaultStoreFile(keyStore, trustStoreFile, trustStorePassphrase)) {
            keyStore.load(null);
        }
    }

    private static File getDefaultTrustStoreFile() {
        String securityDirectory = System.getProperty("java.home") + File.separator + "lib" + File.separator + "security" + File.separator;
        String trustStorePath = System.getProperty(TRUST_STORE_SYSTEM_PROPERTY);
        if (Utils.isNotNullOrEmpty(trustStorePath)) {
            return new File(trustStorePath);
        }
        File jssecacertsFile = new File(securityDirectory + "jssecacerts");
        if (jssecacertsFile.exists() && jssecacertsFile.isFile()) {
            return jssecacertsFile;
        }
        return new File(securityDirectory + "cacerts");
    }

    private static void loadDefaultKeyStoreFile(KeyStore keyStore, char[] keyStorePassphrase) throws CertificateException, NoSuchAlgorithmException, IOException {
        File keyStoreFile;
        String keyStorePath = System.getProperty(KEY_STORE_SYSTEM_PROPERTY);
        if (Utils.isNotNullOrEmpty(keyStorePath) && CertUtils.loadDefaultStoreFile(keyStore, keyStoreFile = new File(keyStorePath), keyStorePassphrase)) {
            return;
        }
        keyStore.load(null);
    }

    private static boolean loadDefaultStoreFile(KeyStore keyStore, File fileToLoad, char[] passphrase) {
        boolean bl;
        if (!fileToLoad.exists() || !fileToLoad.isFile() || fileToLoad.length() == 0L) {
            return false;
        }
        Exception ex = null;
        FileInputStream fis = new FileInputStream(fileToLoad);
        try {
            keyStore.load(fis, passphrase);
            bl = true;
        }
        catch (Throwable throwable) {
            try {
                try {
                    fis.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                ex = e;
                if ((passphrase == null || passphrase.length == 0) && ex.getCause() instanceof UnrecoverableKeyException) {
                    boolean bl2;
                    FileInputStream fis1 = new FileInputStream(fileToLoad);
                    try {
                        keyStore.load(fis1, KEY_STORE_DEFAULT_PASSWORD.toCharArray());
                        bl2 = true;
                    }
                    catch (Throwable throwable3) {
                        try {
                            try {
                                fis1.close();
                            }
                            catch (Throwable throwable4) {
                                throwable3.addSuppressed(throwable4);
                            }
                            throw throwable3;
                        }
                        catch (Exception exception) {
                            // empty catch block
                        }
                    }
                    fis1.close();
                    return bl2;
                }
                LOG.info("There is a problem with reading default keystore/truststore file {} - the file won't be loaded. The reason is: {}", (Object)fileToLoad, (Object)ex.getMessage());
                return false;
            }
        }
        fis.close();
        return bl;
    }

    public static KeyStore createKeyStore(String clientCertData, String clientCertFile, String clientKeyData, String clientKeyFile, String clientKeyAlgo, String clientKeyPassphrase, String keyStoreFile, String keyStorePassphrase) throws IOException, CertificateException, NoSuchAlgorithmException, InvalidKeySpecException, KeyStoreException {
        ByteArrayInputStream certInputStream = CertUtils.getInputStreamFromDataOrFile(clientCertData, clientCertFile);
        ByteArrayInputStream keyInputStream = CertUtils.getInputStreamFromDataOrFile(clientKeyData, clientKeyFile);
        return CertUtils.createKeyStore(certInputStream, keyInputStream, clientKeyAlgo, clientKeyPassphrase.toCharArray(), keyStoreFile, CertUtils.getPassphrase(KEY_STORE_PASSWORD_SYSTEM_PROPERTY, keyStorePassphrase));
    }

    private static char[] getPassphrase(String property, String passphrase) {
        if (Utils.isNullOrEmpty(passphrase)) {
            passphrase = System.getProperty(property, passphrase);
        }
        if (passphrase != null) {
            return passphrase.toCharArray();
        }
        return null;
    }

    private static byte[] decodePem(InputStream keyInputStream) throws IOException {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(keyInputStream));){
            String line;
            while ((line = reader.readLine()) != null) {
                if (!line.contains("-----BEGIN ")) continue;
                byte[] byArray = CertUtils.readBytes(reader, line.trim().replace("BEGIN", "END"));
                return byArray;
            }
            throw new IOException("PEM is invalid: no begin marker");
        }
    }

    private static byte[] readBytes(BufferedReader reader, String endMarker) throws IOException {
        String line;
        StringBuffer buf = new StringBuffer();
        while ((line = reader.readLine()) != null) {
            if (line.indexOf(endMarker) != -1) {
                return Base64.getDecoder().decode(buf.toString());
            }
            buf.append(line.trim());
        }
        throw new IOException("PEM is invalid : No end marker");
    }

    private static ByteArrayInputStream createInputStreamFromBase64EncodedString(String data) {
        byte[] bytes;
        try {
            bytes = Base64.getDecoder().decode(data);
        }
        catch (IllegalArgumentException illegalArgumentException) {
            bytes = data.getBytes();
        }
        return new ByteArrayInputStream(bytes);
    }
}

