/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.crd.generator.collector;

import io.fabric8.crd.generator.collector.JandexException;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Stream;
import org.jboss.jandex.ClassSummary;
import org.jboss.jandex.Index;
import org.jboss.jandex.Indexer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class JandexIndexer {
    private static final Logger log = LoggerFactory.getLogger(JandexIndexer.class);
    private static final String CLASS_FILE_SUFFIX = ".class";
    private static final String JAR_FILE_SUFFIX = ".jar";
    private int maxJarEntries = 100000;
    private long maxBytesReadFromJar = 100000000L;
    private long maxClassFileSize = 1000000L;

    JandexIndexer() {
    }

    public JandexIndexer withMaxJarEntries(int maxJarEntries) {
        if (maxJarEntries < 1) {
            throw new IllegalArgumentException("maxJarEntries must be greater than 0");
        }
        this.maxJarEntries = maxJarEntries;
        return this;
    }

    public JandexIndexer withMaxClassFileSize(int maxClassFileSize) {
        if (maxClassFileSize < 10) {
            throw new IllegalArgumentException("maxClassFileSize must be greater than 10");
        }
        this.maxClassFileSize = maxClassFileSize;
        return this;
    }

    public JandexIndexer withMaxBytesReadFromJar(long maxBytesReadFromJar) {
        if (maxBytesReadFromJar < 10L) {
            throw new IllegalArgumentException("maxBytesReadFromJar must be greater than 10");
        }
        this.maxBytesReadFromJar = maxBytesReadFromJar;
        return this;
    }

    public Index createIndex(File ... files) {
        return this.createIndex(Arrays.asList(files));
    }

    public Index createIndex(Collection<File> files) {
        Indexer indexer = new Indexer();
        this.appendToIndex(files, indexer);
        return indexer.complete();
    }

    private void appendToIndex(Collection<File> files, Indexer indexer) {
        for (File file : files) {
            if (file.isDirectory()) {
                this.scanDirectoryAndAddToIndex(file, indexer);
                continue;
            }
            if (file.isFile() && file.getName().endsWith(CLASS_FILE_SUFFIX)) {
                this.addClassFileToIndex(file, indexer);
                continue;
            }
            if (file.isFile() && file.getName().endsWith(JAR_FILE_SUFFIX)) {
                this.scanJarFileAndAddToIndex(file, indexer);
                continue;
            }
            throw new JandexException("Not a class file, JAR file or directory: " + file);
        }
    }

    private void addClassFileToIndex(File file, Indexer indexer) {
        try (InputStream in = Files.newInputStream(file.toPath(), new OpenOption[0]);){
            this.addToIndex(in, indexer);
        }
        catch (IOException e) {
            throw new JandexException(e);
        }
    }

    private void scanJarFileAndAddToIndex(File file, Indexer indexer) {
        try (JarFile jar = new JarFile(file);){
            Enumeration<JarEntry> entries = jar.entries();
            int totalEntries = 0;
            long totalBytesRead = 0L;
            while (entries.hasMoreElements()) {
                ++totalEntries;
                JarEntry entry = entries.nextElement();
                if (!entry.isDirectory() && entry.getName().endsWith(CLASS_FILE_SUFFIX)) {
                    long bytesRead = this.addToIndex(jar, entry, indexer);
                    totalBytesRead += bytesRead;
                }
                if (totalEntries > this.maxJarEntries) {
                    throw new JandexException("Limit for total JAR file entries exceeded: " + totalEntries);
                }
                if (totalBytesRead <= this.maxBytesReadFromJar) continue;
                throw new JandexException("Limit for total bytes read from JAR file exceeded: " + totalBytesRead + " bytes");
            }
        }
        catch (IOException e) {
            throw new JandexException("Could not index JAR file " + file, e);
        }
    }

    private void scanDirectoryAndAddToIndex(File directory, Indexer indexer) {
        try (Stream<Path> stream = Files.walk(directory.toPath(), new FileVisitOption[0]);){
            stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(file -> file.toString().endsWith(CLASS_FILE_SUFFIX)).forEach(file -> this.addToIndex((Path)file, indexer));
        }
        catch (IOException e) {
            throw new JandexException(e);
        }
    }

    private long addToIndex(JarFile zip, JarEntry entry, Indexer indexer) {
        long l;
        block8: {
            InputStream in = zip.getInputStream(entry);
            try {
                l = this.addToIndex(in, indexer);
                if (in == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (in != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new JandexException("Could not index " + entry.getName() + " from JAR file " + zip.getName(), e);
                }
            }
            in.close();
        }
        return l;
    }

    private long addToIndex(Path file, Indexer indexer) {
        long l;
        block8: {
            InputStream in = Files.newInputStream(file, new OpenOption[0]);
            try {
                l = this.addToIndex(in, indexer);
                if (in == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (in != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new JandexException("Could not index " + file, e);
                }
            }
            in.close();
        }
        return l;
    }

    private long addToIndex(InputStream inputStream, Indexer indexer) throws IOException {
        try (BoundedInputStream boundedInputStream = new BoundedInputStream(inputStream, this.maxClassFileSize);){
            long l;
            try (BufferedInputStream bufferedInputStream = new BufferedInputStream(boundedInputStream);){
                ClassSummary info = indexer.indexWithSummary(bufferedInputStream);
                log.debug("Indexed: {} ({} Annotations)", (Object)info.name(), (Object)info.annotationsCount());
                l = boundedInputStream.getBytesRead();
            }
            return l;
        }
    }

    static class BoundedInputStream
    extends InputStream
    implements AutoCloseable {
        private final long maxBytes;
        private final InputStream inputStream;
        private long bytesRead = 0L;

        public BoundedInputStream(InputStream inputStream, long maxBytes) {
            this.inputStream = inputStream;
            this.maxBytes = maxBytes;
        }

        public long getBytesRead() {
            return this.bytesRead;
        }

        @Override
        public int read() throws IOException {
            if (this.bytesRead >= this.maxBytes) {
                throw new IOException("Read limit of " + this.maxBytes + " bytes exceeded");
            }
            int result = this.inputStream.read();
            if (result != -1) {
                ++this.bytesRead;
            }
            return result;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            int count;
            if (this.bytesRead >= this.maxBytes) {
                throw new IOException("Read limit of " + this.maxBytes + " bytes exceeded");
            }
            long bytesRemaining = this.maxBytes - this.bytesRead;
            if ((long)len > bytesRemaining) {
                len = (int)bytesRemaining;
            }
            if ((count = this.inputStream.read(b, off, len)) > 0) {
                this.bytesRead += (long)count;
            }
            return count;
        }

        @Override
        public int available() throws IOException {
            long bytesRemaining;
            long available = this.inputStream.available();
            if (available > (bytesRemaining = this.maxBytes - this.bytesRead)) {
                return (int)bytesRemaining;
            }
            return (int)available;
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
            super.close();
        }
    }
}

