/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.cockroachdb.connection;

import io.debezium.connector.cockroachdb.CockroachDBConnectorConfig;
import io.debezium.connector.cockroachdb.CockroachDBErrorHandler;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CockroachDBConnection
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(CockroachDBConnection.class);
    private final CockroachDBConnectorConfig config;
    private final CockroachDBErrorHandler errorHandler;
    private Connection connection;

    public CockroachDBConnection(CockroachDBConnectorConfig config) {
        this.config = config;
        this.errorHandler = new CockroachDBErrorHandler(config, null);
    }

    public void connect() throws SQLException {
        String url = this.buildConnectionUrl();
        Properties props = this.buildConnectionProperties();
        SQLException lastException = null;
        int maxRetries = this.config.getConnectionMaxRetries();
        for (int attempts = 0; attempts < maxRetries; ++attempts) {
            try {
                LOGGER.info("Attempting to connect to CockroachDB (attempt {}/{}): {}", new Object[]{attempts + 1, maxRetries, url});
                this.connection = DriverManager.getConnection(url, props);
                try (Statement stmt = this.connection.createStatement();){
                    stmt.execute("SELECT 1");
                }
                this.checkChangefeedPermissions();
                LOGGER.info("Successfully connected to CockroachDB with required permissions");
                return;
            }
            catch (SQLException e) {
                lastException = e;
                try {
                    long retryDelay = this.config.getConnectionRetryDelayMs() * (long)attempts;
                    if (this.errorHandler.handleConnectionError(e, attempts, maxRetries, retryDelay)) continue;
                    break;
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new SQLException("Connection interrupted", ie);
                }
            }
        }
        LOGGER.error("Failed to connect to CockroachDB after {} attempts", (Object)maxRetries);
        throw lastException != null ? lastException : new SQLException("Failed to connect to CockroachDB");
    }

    private String buildConnectionUrl() {
        StringBuilder url = new StringBuilder();
        url.append("jdbc:postgresql://");
        url.append(this.config.getHostname());
        url.append(":").append(this.config.getPort());
        url.append("/").append(this.config.getDatabaseName());
        CockroachDBConnectorConfig.SecureConnectionMode sslMode = CockroachDBConnectorConfig.SecureConnectionMode.parse(this.config.getSslMode());
        if (sslMode != CockroachDBConnectorConfig.SecureConnectionMode.DISABLED) {
            url.append("?sslmode=").append(sslMode.getValue());
        }
        return url.toString();
    }

    private Properties buildConnectionProperties() {
        Properties props = new Properties();
        props.setProperty("user", this.config.getUser());
        String password = this.config.getPassword();
        if (password != null) {
            props.setProperty("password", password);
        }
        props.setProperty("connectTimeout", String.valueOf(this.config.getConnectionTimeoutMs() / 1000L));
        CockroachDBConnectorConfig.SecureConnectionMode sslMode = CockroachDBConnectorConfig.SecureConnectionMode.parse(this.config.getSslMode());
        if (sslMode != CockroachDBConnectorConfig.SecureConnectionMode.DISABLED) {
            if (this.config.getSslRootCert() != null) {
                props.setProperty("sslrootcert", this.config.getSslRootCert());
            }
            if (this.config.getSslClientCert() != null) {
                props.setProperty("sslcert", this.config.getSslClientCert());
            }
            if (this.config.getSslClientKey() != null) {
                props.setProperty("sslkey", this.config.getSslClientKey());
            }
            if (this.config.getSslClientKeyPassword() != null) {
                props.setProperty("sslpassword", this.config.getSslClientKeyPassword());
            }
        }
        if (this.config.isTcpKeepAlive()) {
            props.setProperty("tcpKeepAlive", "true");
        }
        return props;
    }

    public Connection connection() {
        return this.connection;
    }

    public boolean isValid() {
        try {
            return this.connection != null && !this.connection.isClosed() && this.connection.isValid(5);
        }
        catch (SQLException e) {
            LOGGER.debug("Error checking connection validity: {}", (Object)e.getMessage());
            return false;
        }
    }

    private void checkChangefeedPermissions() throws SQLException {
        LOGGER.debug("Checking changefeed permissions for user: {}", (Object)this.config.getUser());
        try (Statement stmt = this.connection.createStatement();){
            boolean rangefeedEnabled = false;
            try {
                stmt.execute("SHOW CLUSTER SETTING kv.rangefeed.enabled");
                ResultSet rs = stmt.getResultSet();
                if (rs != null && rs.next()) {
                    String rangefeedSetting = rs.getString(1);
                    rangefeedEnabled = "true".equalsIgnoreCase(rangefeedSetting);
                    LOGGER.debug("Rangefeed setting: {}", (Object)rangefeedSetting);
                }
            }
            catch (SQLException e) {
                if (e.getMessage().contains("VIEWCLUSTERSETTING") || e.getMessage().contains("MODIFYCLUSTERSETTING")) {
                    LOGGER.warn("Cannot check rangefeed cluster setting due to insufficient privileges. Assuming rangefeed is enabled. If changefeeds fail, ensure 'kv.rangefeed.enabled = true' and grant VIEWCLUSTERSETTING to user '{}': GRANT VIEWCLUSTERSETTING TO {}", (Object)this.config.getUser(), (Object)this.config.getUser());
                    rangefeedEnabled = true;
                }
                throw e;
            }
            if (!rangefeedEnabled) {
                throw new SQLException("Rangefeed is disabled. Enable rangefeed by setting 'kv.rangefeed.enabled = true' in your CockroachDB cluster configuration. This is required for changefeeds to work.");
            }
            String dbName = this.config.getDatabaseName();
            String schemaName = this.config.getSchemaName();
            if (schemaName == null || schemaName.trim().isEmpty()) {
                schemaName = "public";
                LOGGER.debug("Schema name was null or empty, using default: {}", (Object)schemaName);
            }
            LOGGER.debug("Checking CHANGEFEED privileges for schema: {}", (Object)schemaName);
            stmt.execute("SHOW GRANTS ON TABLE " + schemaName + ".*");
            ResultSet rs = stmt.getResultSet();
            boolean hasChangefeedPrivilege = false;
            if (rs != null) {
                while (rs.next()) {
                    String grantee = rs.getString("grantee");
                    String privilegeType = rs.getString("privilege_type");
                    if (!this.config.getUser().equals(grantee) || !"CHANGEFEED".equals(privilegeType)) continue;
                    hasChangefeedPrivilege = true;
                    break;
                }
            }
            if (!hasChangefeedPrivilege) {
                throw new SQLException("User '" + this.config.getUser() + "' lacks CHANGEFEED privilege on any table in database '" + dbName + "' schema '" + schemaName + "'. Grant the privilege with: GRANT CHANGEFEED ON TABLE " + schemaName + ".table_name TO " + this.config.getUser());
            }
            LOGGER.debug("User has CHANGEFEED privilege on at least one table in database: {} schema: {}", (Object)dbName, (Object)schemaName);
            stmt.execute("SELECT 1 WHERE EXISTS (SELECT 1 FROM information_schema.tables WHERE table_schema = '" + schemaName + "' LIMIT 1)");
            rs = stmt.getResultSet();
            if (rs != null && !rs.next()) {
                throw new SQLException("No accessible tables found in database '" + dbName + "' schema '" + schemaName + "'. Ensure the user has SELECT privilege on at least one table.");
            }
        }
        catch (SQLException e) {
            LOGGER.error("Permission check failed: {}", (Object)e.getMessage());
            throw e;
        }
    }

    @Override
    public void close() {
        if (this.connection != null) {
            try {
                this.connection.close();
                LOGGER.debug("Closed CockroachDB connection");
            }
            catch (SQLException e) {
                LOGGER.warn("Error closing CockroachDB connection: {}", (Object)e.getMessage());
            }
            finally {
                this.connection = null;
            }
        }
    }
}

