/*
 * Decompiled with CFR 0.152.
 */
package io.continual.util.collections;

import io.continual.util.time.Clock;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ShardedExpiringCache<K, V> {
    private final String fName;
    private final long fDurMs;
    private final int fShardCount;
    private final ArrayList<Shard> fShards;
    private final Monitor fMonitor;
    private static final int kDefaultShardCount = 1024;
    private static final long skWarnOnLockDurationMs = 10000L;
    private static final Logger log = LoggerFactory.getLogger(ShardedExpiringCache.class);

    public V read(K key) {
        return this.read(key, null);
    }

    public V read(K key, Validator<V> validator) {
        try {
            return this.read(key, validator, null);
        }
        catch (Fetcher.FetchException e) {
            throw new RuntimeException("Null fetcher caused fetch exception?", e);
        }
    }

    public V read(K key, Validator<V> validator, Fetcher<K, V> fetcher) throws Fetcher.FetchException {
        try (ShardCallWrap lw = new ShardCallWrap("for read");){
            Object v = this.getShard(key).read(key, validator, fetcher);
            return v;
        }
    }

    public void write(K key, V val) {
        this.write(key, val, this.fDurMs);
    }

    public void write(K key, V val, long cacheDurationMs) {
        try (ShardCallWrap lw = new ShardCallWrap("for write");){
            this.getShard(key).write(key, val, cacheDurationMs);
        }
    }

    public void remove(K key) {
        try (ShardCallWrap lw = new ShardCallWrap("for object removal");){
            this.getShard(key).remove(key);
        }
    }

    public void empty() {
        for (int i = 0; i < this.fShardCount; ++i) {
            this.fShards.get(i).empty();
        }
    }

    private Shard getShard(K key) {
        int hash = Math.abs(key.hashCode());
        int index = hash % this.fShardCount;
        return this.fShards.get(index);
    }

    private ShardedExpiringCache(Builder<K, V> b) {
        this.fName = ((Builder)b).fName;
        this.fDurMs = ((Builder)b).fDurMs;
        this.fShardCount = ((Builder)b).fShardCount;
        this.fShards = new ArrayList(this.fShardCount);
        for (int i = 0; i < this.fShardCount; ++i) {
            this.fShards.add(new Shard(i));
        }
        this.fMonitor = ((Builder)b).fMonitor;
    }

    static long now() {
        return Clock.now();
    }

    private static class ShardCallWrap
    implements AutoCloseable {
        private final long fStartTimeMs = ShardedExpiringCache.now();

        public ShardCallWrap(String msg) {
            log.debug("locking shard " + msg);
        }

        @Override
        public void close() {
            long stopTimeMs = ShardedExpiringCache.now();
            long durationMs = stopTimeMs - this.fStartTimeMs;
            log.debug("shard released, {} ms", (Object)durationMs);
            if (durationMs > 10000L) {
                log.warn("Shard call wrap took {} ms", (Object)durationMs);
            }
        }
    }

    private class CacheEntry {
        public final K fKey;
        public final V fVal;
        public final long fExpiresAtMs;

        public CacheEntry(K key, V val, long expiresAtMs) {
            this.fKey = key;
            this.fVal = val;
            this.fExpiresAtMs = expiresAtMs;
        }

        public String toString() {
            return this.fKey.toString();
        }
    }

    private class Shard {
        private final int fId;
        private final HashMap<K, CacheEntry> fItemCache;
        private final LinkedList<CacheEntry> fItemCleanups;

        public Shard(int id) {
            this.fId = id;
            this.fItemCache = new HashMap();
            this.fItemCleanups = new LinkedList();
        }

        public synchronized V read(K key, Validator<V> validator, Fetcher<K, V> fetcher) throws Fetcher.FetchException {
            this.cleanupCache();
            CacheEntry e = this.fItemCache.get(key);
            if (e != null) {
                if (validator == null || validator.isValid(e.fVal)) {
                    ShardedExpiringCache.this.fMonitor.onCacheHit();
                    log.info("Read/returned {} from cache {}/{}.", new Object[]{key.toString(), ShardedExpiringCache.this.fName, this.fId});
                    return e.fVal;
                }
                log.debug("Read {} from cache {}/{}, but it's not valid.", new Object[]{key.toString(), ShardedExpiringCache.this.fName, this.fId});
                this.fItemCache.remove(key);
            }
            log.debug("No valid entry for {} in cache {}/{}.", new Object[]{key.toString(), ShardedExpiringCache.this.fName, this.fId});
            ShardedExpiringCache.this.fMonitor.onCacheMiss();
            if (fetcher != null) {
                log.info("Cache fetching {} from backing store in shard {}/{}", new Object[]{key, ShardedExpiringCache.this.fName, this.fId});
                Object fetched = fetcher.fetch(key);
                if (fetched != null) {
                    this.write(key, fetched);
                    return fetched;
                }
            }
            return null;
        }

        public synchronized void write(K key, V val) {
            this.write(key, val, ShardedExpiringCache.this.fDurMs);
        }

        public synchronized void write(K key, V val, long cacheDurationMs) {
            if (key == null) {
                log.warn("Ignoring null key insert in cache {}.", (Object)ShardedExpiringCache.this.fName);
                return;
            }
            if (cacheDurationMs <= 0L) {
                return;
            }
            CacheEntry ce = new CacheEntry(key, val, ShardedExpiringCache.now() + cacheDurationMs);
            this.fItemCache.put(key, ce);
            this.fItemCleanups.add(ce);
            log.debug("Wrote {} to cache {}.", (Object)key.toString(), (Object)ShardedExpiringCache.this.fName);
        }

        public synchronized void remove(K key) {
            this.fItemCache.remove(key);
        }

        public synchronized void empty() {
            this.fItemCache.clear();
            this.fItemCleanups.clear();
        }

        private void cleanupCache() {
            long now = ShardedExpiringCache.now();
            while (this.fItemCleanups.size() > 0 && this.fItemCleanups.get((int)0).fExpiresAtMs < now) {
                CacheEntry cleanupEntry = this.fItemCleanups.remove();
                CacheEntry cacheEntry = this.fItemCache.get(cleanupEntry.fKey);
                if (cacheEntry != null && cacheEntry == cleanupEntry) {
                    this.fItemCache.remove(cleanupEntry.fKey);
                    log.info("Removed cache entry for \"{}\" in cache {}.", (Object)cleanupEntry.toString(), (Object)ShardedExpiringCache.this.fName);
                    continue;
                }
                log.debug("Removed cleanup entry for \"{}\" in cache {}, but no match in item cache.", (Object)cleanupEntry.toString(), (Object)ShardedExpiringCache.this.fName);
            }
        }
    }

    public static class Builder<K, V> {
        private int fShardCount = 1024;
        private Monitor fMonitor = new Monitor(){};
        private String fName = "(unnamed)";
        private long fDurMs = 900000L;

        public Builder<K, V> named(String name) {
            this.fName = name;
            return this;
        }

        public Builder<K, V> cachingFor(long duration, TimeUnit timeUnit) {
            this.fDurMs = TimeUnit.MILLISECONDS.convert(duration, timeUnit);
            return this;
        }

        public Builder<K, V> withShardCount(int shardCount) {
            this.fShardCount = shardCount;
            return this;
        }

        public Builder<K, V> notificationsTo(Monitor m) {
            this.fMonitor = m;
            return this;
        }

        public ShardedExpiringCache<K, V> build() {
            return new ShardedExpiringCache(this);
        }
    }

    public static interface Monitor {
        default public void onCacheHit() {
        }

        default public void onCacheMiss() {
        }
    }

    public static interface Fetcher<K, V> {
        public V fetch(K var1) throws FetchException;

        public static class FetchException
        extends Exception {
            private static final long serialVersionUID = 1L;

            public FetchException(String msg) {
                super(msg);
            }

            public FetchException(Throwable t) {
                super(t);
            }
        }
    }

    public static interface Validator<V> {
        public boolean isValid(V var1);
    }
}

