@file:Suppress("unused")

package coil.decode

import android.content.Context
import android.graphics.ImageDecoder
import android.graphics.drawable.AnimatedImageDrawable
import android.os.Build.VERSION.SDK_INT
import androidx.annotation.RequiresApi
import androidx.core.graphics.decodeDrawable
import androidx.core.util.component1
import androidx.core.util.component2
import coil.bitmap.BitmapPool
import coil.drawable.ScaleDrawable
import coil.request.animatedTransformation
import coil.request.animationEndCallback
import coil.request.animationStartCallback
import coil.request.repeatCount
import coil.size.PixelSize
import coil.size.Size
import coil.util.animatable2CallbackOf
import coil.util.asPostProcessor
import coil.util.isHardware
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okio.BufferedSource
import okio.buffer
import okio.sink
import java.io.File
import java.nio.ByteBuffer
import kotlin.math.roundToInt

/**
 * A [Decoder] that uses [ImageDecoder] to decode GIFs, animated WebPs, and animated HEIFs.
 *
 * NOTE: Animated HEIF files are only supported on API 30 and above.
 *
 * @param context An Android context.
 * @param enforceMinimumFrameDelay If true, rewrite a GIF's frame delay to a default value if
 *  it is below a threshold. See https://github.com/coil-kt/coil/issues/540 for more info.
 */
@RequiresApi(28)
class ImageDecoderDecoder private constructor(
    // Reverse parameter order to avoid platform declaration clash.
    private val enforceMinimumFrameDelay: Boolean,
    private val context: Context?
) : Decoder {

    @Deprecated(
        message = "Migrate to the constructor that accepts a Context.",
        replaceWith = ReplaceWith("ImageDecoderDecoder(context)")
    )
    constructor() : this(false, null)

    constructor(context: Context) : this(false, context)

    constructor(context: Context, enforceMinimumFrameDelay: Boolean) : this(enforceMinimumFrameDelay, context)

    override fun handles(source: BufferedSource, mimeType: String?): Boolean {
        return DecodeUtils.isGif(source) ||
            DecodeUtils.isAnimatedWebP(source) ||
            (SDK_INT >= 30 && DecodeUtils.isAnimatedHeif(source))
    }

    override suspend fun decode(
        pool: BitmapPool,
        source: BufferedSource,
        size: Size,
        options: Options
    ): DecodeResult {
        var isSampled = false
        val baseDrawable = withInterruptibleSource(source) { interruptibleSource ->
            var tempFile: File? = null
            try {
                val bufferedInterruptibleSource = interruptibleSource.buffer()
                val bufferedSource = if (enforceMinimumFrameDelay && DecodeUtils.isGif(bufferedInterruptibleSource)) {
                    FrameDelayRewritingSource(bufferedInterruptibleSource).buffer()
                } else {
                    bufferedInterruptibleSource
                }
                val decoderSource = if (SDK_INT >= 30) {
                    // Buffer the source into memory.
                    ImageDecoder.createSource(ByteBuffer.wrap(bufferedSource.use { it.readByteArray() }))
                } else {
                    // Work around https://issuetracker.google.com/issues/139371066 by copying the source to a temp file.
                    tempFile = File.createTempFile("tmp", null, context?.cacheDir?.apply { mkdirs() })
                    bufferedSource.use { tempFile.sink().use(it::readAll) }
                    ImageDecoder.createSource(tempFile)
                }

                decoderSource.decodeDrawable { info, _ ->
                    // It's safe to delete the temp file here.
                    tempFile?.delete()

                    if (size is PixelSize) {
                        val (srcWidth, srcHeight) = info.size
                        val multiplier = DecodeUtils.computeSizeMultiplier(
                            srcWidth = srcWidth,
                            srcHeight = srcHeight,
                            dstWidth = size.width,
                            dstHeight = size.height,
                            scale = options.scale
                        )

                        // Set the target size if the image is larger than the requested dimensions
                        // or the request requires exact dimensions.
                        isSampled = multiplier < 1
                        if (isSampled || !options.allowInexactSize) {
                            val targetWidth = (multiplier * srcWidth).roundToInt()
                            val targetHeight = (multiplier * srcHeight).roundToInt()
                            setTargetSize(targetWidth, targetHeight)
                        }
                    }

                    allocator = if (options.config.isHardware) {
                        ImageDecoder.ALLOCATOR_HARDWARE
                    } else {
                        ImageDecoder.ALLOCATOR_SOFTWARE
                    }

                    memorySizePolicy = if (options.allowRgb565) {
                        ImageDecoder.MEMORY_POLICY_LOW_RAM
                    } else {
                        ImageDecoder.MEMORY_POLICY_DEFAULT
                    }

                    if (options.colorSpace != null) {
                        setTargetColorSpace(options.colorSpace)
                    }

                    isUnpremultipliedRequired = !options.premultipliedAlpha

                    postProcessor = options.parameters.animatedTransformation()?.asPostProcessor()
                }
            } finally {
                tempFile?.delete()
            }
        }

        val drawable = if (baseDrawable is AnimatedImageDrawable) {
            baseDrawable.repeatCount = options.parameters.repeatCount() ?: AnimatedImageDrawable.REPEAT_INFINITE

            // Set the start and end animation callbacks if any one is supplied through the request.
            val onStart = options.parameters.animationStartCallback()
            val onEnd = options.parameters.animationEndCallback()
            if (onStart != null || onEnd != null) {
                // Animation callbacks must be set on the main thread.
                withContext(Dispatchers.Main.immediate) {
                    baseDrawable.registerAnimationCallback(animatable2CallbackOf(onStart, onEnd))
                }
            }

            // Wrap AnimatedImageDrawable in a ScaleDrawable so it always scales to fill its bounds.
            ScaleDrawable(baseDrawable, options.scale)
        } else {
            baseDrawable
        }

        return DecodeResult(
            drawable = drawable,
            isSampled = isSampled
        )
    }

    companion object {
        const val REPEAT_COUNT_KEY = GifDecoder.REPEAT_COUNT_KEY
        const val ANIMATED_TRANSFORMATION_KEY = GifDecoder.ANIMATED_TRANSFORMATION_KEY
        const val ANIMATION_START_CALLBACK_KEY = GifDecoder.ANIMATION_START_CALLBACK_KEY
        const val ANIMATION_END_CALLBACK_KEY = GifDecoder.ANIMATION_END_CALLBACK_KEY
    }
}
