/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.future.CompletableActorFuture;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.ReceivedSnapshot;
import io.camunda.zeebe.snapshots.SnapshotChunk;
import io.camunda.zeebe.snapshots.SnapshotId;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotId;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotMetadata;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotStoreImpl;
import io.camunda.zeebe.snapshots.impl.SfvChecksumImpl;
import io.camunda.zeebe.snapshots.impl.SnapshotChunkUtil;
import io.camunda.zeebe.snapshots.impl.SnapshotWriteException;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedReceivedSnapshot
implements ReceivedSnapshot {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileBasedReceivedSnapshot.class);
    private static final int BLOCK_SIZE = 524288;
    private final Path directory;
    private final ConcurrencyControl actor;
    private final FileBasedSnapshotStoreImpl snapshotStore;
    private final FileBasedSnapshotId snapshotId;
    private int expectedTotalCount;
    private FileBasedSnapshotMetadata metadata;
    private ByteBuffer metadataBuffer;
    private long writtenMetadataBytes;
    private SfvChecksumImpl checksumCollection;

    FileBasedReceivedSnapshot(FileBasedSnapshotId snapshotId, Path directory, FileBasedSnapshotStoreImpl snapshotStore, ConcurrencyControl actor) {
        this.snapshotId = snapshotId;
        this.snapshotStore = snapshotStore;
        this.directory = directory;
        this.actor = actor;
        this.expectedTotalCount = Integer.MIN_VALUE;
        this.writtenMetadataBytes = 0L;
    }

    @Override
    public long index() {
        return this.snapshotId.getIndex();
    }

    @Override
    public ActorFuture<Void> apply(SnapshotChunk snapshotChunk) {
        return this.actor.call(() -> {
            this.applyInternal(snapshotChunk);
            return null;
        });
    }

    private void applyInternal(SnapshotChunk snapshotChunk) throws SnapshotWriteException {
        this.checkSnapshotIdIsValid(snapshotChunk.getSnapshotId());
        int currentTotalCount = snapshotChunk.getTotalCount();
        this.checkTotalCountIsValid(currentTotalCount);
        String snapshotId = snapshotChunk.getSnapshotId();
        String chunkName = snapshotChunk.getChunkName();
        if (this.snapshotStore.hasSnapshotId(snapshotId)) {
            LOGGER.debug("Ignore snapshot snapshotChunk {}, because snapshot {} already exists.", (Object)chunkName, (Object)snapshotId);
            return;
        }
        this.checkChunkChecksumIsValid(snapshotChunk, snapshotId, chunkName);
        Path tmpSnapshotDirectory = this.directory;
        try {
            FileUtil.ensureDirectoryExists((Path)tmpSnapshotDirectory);
        }
        catch (IOException e) {
            throw new SnapshotWriteException(String.format("Failed to ensure that directory %s exists.", tmpSnapshotDirectory), e);
        }
        Path snapshotFile = tmpSnapshotDirectory.resolve(chunkName);
        LOGGER.trace("Consume snapshot snapshotChunk {} of snapshot {}", (Object)chunkName, (Object)snapshotId);
        this.writeReceivedSnapshotChunk(snapshotChunk, snapshotFile);
        if (this.checksumCollection == null) {
            this.checksumCollection = new SfvChecksumImpl();
        }
        this.checksumCollection.updateFromBytes(snapshotFile.getFileName().toString(), snapshotChunk.getContent());
        if (snapshotChunk.getChunkName().equals("zeebe.metadata")) {
            try {
                this.collectMetadata(snapshotChunk);
            }
            catch (IOException e) {
                throw new SnapshotWriteException("Cannot decode snapshot metadata");
            }
        }
    }

    private void collectMetadata(SnapshotChunk chunk) throws IOException {
        if (this.metadataBuffer == null) {
            this.metadataBuffer = ByteBuffer.allocate(Math.toIntExact(chunk.getTotalFileSize()));
        }
        this.metadataBuffer.put(Math.toIntExact(chunk.getFileBlockPosition()), chunk.getContent());
        this.writtenMetadataBytes += (long)chunk.getContent().length;
        if (this.writtenMetadataBytes == chunk.getTotalFileSize()) {
            this.metadata = FileBasedSnapshotMetadata.decode(this.metadataBuffer.array());
        }
    }

    private void checkChunkChecksumIsValid(SnapshotChunk snapshotChunk, String snapshotId, String chunkName) throws SnapshotWriteException {
        long actualChecksum;
        long expectedChecksum = snapshotChunk.getChecksum();
        if (expectedChecksum != (actualChecksum = SnapshotChunkUtil.createChecksum(snapshotChunk.getContent()))) {
            throw new SnapshotWriteException(String.format("Expected to have checksum %d for snapshot chunk %s (%s), but calculated %d", expectedChecksum, chunkName, snapshotId, actualChecksum));
        }
    }

    private void checkTotalCountIsValid(int currentTotalCount) throws SnapshotWriteException {
        if (this.expectedTotalCount == Integer.MIN_VALUE) {
            this.expectedTotalCount = currentTotalCount;
        }
        if (this.expectedTotalCount != currentTotalCount) {
            throw new SnapshotWriteException(String.format("Expected snapshot chunk with equal snapshot total count %d, but got chunk with total count %d.", this.expectedTotalCount, currentTotalCount));
        }
    }

    private void checkSnapshotIdIsValid(String snapshotId) throws SnapshotWriteException {
        Optional<FileBasedSnapshotId> receivedSnapshotId = FileBasedSnapshotId.ofFileName(snapshotId);
        if (receivedSnapshotId.isEmpty()) {
            throw new SnapshotWriteException(String.format("Snapshot file name '%s' has unexpected format", snapshotId));
        }
        FileBasedSnapshotId chunkSnapshotId = receivedSnapshotId.get();
        if (this.snapshotId.compareTo(chunkSnapshotId) != 0) {
            throw new SnapshotWriteException(String.format("Expected snapshot id in chunk to be '%s' but was '%s' instead", this.snapshotId, chunkSnapshotId));
        }
    }

    private void writeReceivedSnapshotChunk(SnapshotChunk snapshotChunk, Path snapshotFile) throws SnapshotWriteException {
        try (FileChannel channel = FileChannel.open(snapshotFile, StandardOpenOption.WRITE, StandardOpenOption.CREATE);){
            ByteBuffer buffer = ByteBuffer.wrap(snapshotChunk.getContent());
            while (buffer.hasRemaining()) {
                int newLimit = Math.min(buffer.capacity(), buffer.position() + 524288);
                channel.position(snapshotChunk.getFileBlockPosition() + (long)buffer.position());
                channel.write(buffer.limit(newLimit));
                buffer.limit(buffer.capacity());
            }
            channel.force(true);
        }
        catch (IOException e) {
            throw new SnapshotWriteException(String.format("Failed to write snapshot chunk %s", snapshotChunk), e);
        }
        LOGGER.trace("Wrote replicated snapshot chunk to file {}", (Object)snapshotFile);
    }

    @Override
    public ActorFuture<Void> abort() {
        CompletableActorFuture abortFuture = new CompletableActorFuture();
        this.actor.run(() -> {
            this.abortInternal();
            abortFuture.complete(null);
        });
        return abortFuture;
    }

    @Override
    public ActorFuture<PersistedSnapshot> persist() {
        CompletableActorFuture future = new CompletableActorFuture();
        this.actor.call(() -> {
            this.persistInternal((CompletableActorFuture<PersistedSnapshot>)future);
            return null;
        });
        return future;
    }

    @Override
    public SnapshotId snapshotId() {
        return this.snapshotId;
    }

    @Override
    public Path getPath() {
        return this.directory;
    }

    private void abortInternal() {
        try {
            LOGGER.debug("Aborting received snapshot in dir {}", (Object)this.directory);
            FileUtil.deleteFolderIfExists((Path)this.directory);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to delete pending snapshot {}", (Object)this, (Object)e);
        }
        finally {
            this.snapshotStore.removePendingSnapshot(this);
        }
    }

    private void persistInternal(CompletableActorFuture<PersistedSnapshot> future) {
        if (this.snapshotStore.hasSnapshotId(this.snapshotId.getSnapshotIdAsString())) {
            this.abortInternal();
            future.complete((Object)this.snapshotStore.getLatestSnapshot().orElseThrow());
            return;
        }
        Object[] files = this.directory.toFile().listFiles();
        try {
            Objects.requireNonNull(files, "No chunks have been applied yet");
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
            return;
        }
        if (files.length != this.expectedTotalCount) {
            future.completeExceptionally((Throwable)new IllegalStateException(String.format("Expected '%d' chunk files for this snapshot, but found '%d'. Files are: %s.", this.expectedTotalCount, files.length, Arrays.toString(files))));
            return;
        }
        try {
            if (this.metadata == null) {
                this.metadata = new FileBasedSnapshotMetadata(1, this.snapshotId.getProcessedPosition(), this.snapshotId.getExportedPosition(), Long.MAX_VALUE);
            }
            FileBasedSnapshot value = this.snapshotStore.persistNewSnapshot(this.snapshotId, this.checksumCollection, this.metadata);
            future.complete((Object)value);
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
        }
        this.snapshotStore.removePendingSnapshot(this);
    }

    public String toString() {
        return "FileBasedReceivedSnapshot{directory=" + String.valueOf(this.directory) + ", metadata=" + String.valueOf(this.snapshotId) + "}";
    }
}

