/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.operate.webapp.elasticsearch.backup;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.operate.conditions.ElasticsearchCondition;
import io.camunda.operate.exceptions.OperateElasticsearchConnectionException;
import io.camunda.operate.exceptions.OperateRuntimeException;
import io.camunda.operate.property.OperateProperties;
import io.camunda.operate.util.ThreadUtil;
import io.camunda.operate.webapp.api.v1.exceptions.ResourceNotFoundException;
import io.camunda.operate.webapp.backup.BackupRepository;
import io.camunda.operate.webapp.backup.BackupService;
import io.camunda.operate.webapp.backup.Metadata;
import io.camunda.operate.webapp.management.dto.BackupStateDto;
import io.camunda.operate.webapp.management.dto.GetBackupStateResponseDetailDto;
import io.camunda.operate.webapp.management.dto.GetBackupStateResponseDto;
import io.camunda.operate.webapp.rest.exception.InvalidRequestException;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.repositories.get.GetRepositoriesRequest;
import org.elasticsearch.action.admin.cluster.repositories.get.GetRepositoriesResponse;
import org.elasticsearch.action.admin.cluster.snapshots.create.CreateSnapshotRequest;
import org.elasticsearch.action.admin.cluster.snapshots.create.CreateSnapshotResponse;
import org.elasticsearch.action.admin.cluster.snapshots.delete.DeleteSnapshotRequest;
import org.elasticsearch.action.admin.cluster.snapshots.get.GetSnapshotsRequest;
import org.elasticsearch.action.admin.cluster.snapshots.get.GetSnapshotsResponse;
import org.elasticsearch.action.support.IndicesOptions;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.search.sort.SortOrder;
import org.elasticsearch.snapshots.SnapshotInfo;
import org.elasticsearch.snapshots.SnapshotShardFailure;
import org.elasticsearch.snapshots.SnapshotState;
import org.elasticsearch.transport.TransportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Conditional;
import org.springframework.stereotype.Component;

@Conditional(value={ElasticsearchCondition.class})
@Component
public class ElasticsearchBackupRepository
implements BackupRepository {
    public static final String SNAPSHOT_MISSING_EXCEPTION_TYPE = "type=snapshot_missing_exception";
    private static final String REPOSITORY_MISSING_EXCEPTION_TYPE = "type=repository_missing_exception";
    private static final Logger LOGGER = LoggerFactory.getLogger(ElasticsearchBackupRepository.class);
    @Autowired
    private RestHighLevelClient esClient;
    @Autowired
    @Qualifier(value="operateObjectMapper")
    private ObjectMapper objectMapper;
    @Autowired
    private OperateProperties operateProperties;

    @Override
    public void deleteSnapshot(String repositoryName, String snapshotName) {
        DeleteSnapshotRequest request = new DeleteSnapshotRequest(repositoryName);
        request.snapshots(new String[]{snapshotName});
        this.esClient.snapshot().deleteAsync(request, RequestOptions.DEFAULT, this.getDeleteListener());
    }

    @Override
    public void validateRepositoryExists(String repositoryName) {
        GetRepositoriesRequest getRepositoriesRequest = new GetRepositoriesRequest().repositories(new String[]{repositoryName});
        try {
            GetRepositoriesResponse getRepositoriesResponse = this.esClient.snapshot().getRepository(getRepositoriesRequest, RequestOptions.DEFAULT);
        }
        catch (IOException | TransportException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while retrieving repository with name [%s].", repositoryName);
            throw new OperateElasticsearchConnectionException(reason, ex);
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", repositoryName);
                throw new OperateRuntimeException(reason);
            }
            String reason = String.format("Exception occurred when validating existence of repository with name [%s].", repositoryName);
            throw new OperateRuntimeException(reason, (Throwable)e);
        }
    }

    @Override
    public void validateNoDuplicateBackupId(String repositoryName, Long backupId) {
        GetSnapshotsResponse response;
        GetSnapshotsRequest snapshotsStatusRequest = new GetSnapshotsRequest().repository(repositoryName).snapshots(new String[]{Metadata.buildSnapshotNamePrefix(backupId) + "*"});
        try {
            response = this.esClient.snapshot().get(snapshotsStatusRequest, RequestOptions.DEFAULT);
        }
        catch (IOException | TransportException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for duplicate backup. Repository name: [%s].", repositoryName);
            throw new OperateElasticsearchConnectionException(reason, ex);
        }
        catch (Exception e) {
            if (this.isSnapshotMissingException(e)) {
                return;
            }
            String reason = String.format("Exception occurred when validating whether backup with ID [%s] already exists.", backupId);
            throw new OperateRuntimeException(reason, (Throwable)e);
        }
        if (!response.getSnapshots().isEmpty()) {
            String reason = String.format("A backup with ID [%s] already exists. Found snapshots: [%s]", backupId, response.getSnapshots().stream().map(snapshotInfo -> snapshotInfo.snapshotId().toString()).collect(Collectors.joining(", ")));
            throw new InvalidRequestException(reason);
        }
    }

    @Override
    public GetBackupStateResponseDto getBackupState(String repositoryName, Long backupId) {
        List<SnapshotInfo> snapshots = this.findSnapshots(repositoryName, backupId);
        GetBackupStateResponseDto response = this.getBackupResponse(backupId, snapshots);
        return response;
    }

    @Override
    public List<GetBackupStateResponseDto> getBackups(String repositoryName) {
        GetSnapshotsRequest snapshotsStatusRequest = new GetSnapshotsRequest().repository(repositoryName).snapshots(new String[]{"camunda_operate_*"}).sort(GetSnapshotsRequest.SortBy.START_TIME).order(SortOrder.DESC);
        try {
            GetSnapshotsResponse response = this.esClient.snapshot().get(snapshotsStatusRequest, RequestOptions.DEFAULT);
            List snapshots = response.getSnapshots().stream().sorted(Comparator.comparing(SnapshotInfo::startTime).reversed()).collect(Collectors.toList());
            LinkedHashMap groupedSnapshotInfos = snapshots.stream().collect(Collectors.groupingBy(si -> {
                Metadata metadata = (Metadata)this.objectMapper.convertValue((Object)si.userMetadata(), Metadata.class);
                Long backupId = metadata.getBackupId();
                if (backupId == null) {
                    backupId = Metadata.extractBackupIdFromSnapshotName(si.snapshotId().getName());
                }
                return backupId;
            }, LinkedHashMap::new, Collectors.toList()));
            List<GetBackupStateResponseDto> responses = groupedSnapshotInfos.entrySet().stream().map(entry -> this.getBackupResponse((Long)entry.getKey(), (List)entry.getValue())).collect(Collectors.toList());
            return responses;
        }
        catch (IOException | TransportException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for snapshots. Repository name: [%s].", repositoryName);
            throw new OperateElasticsearchConnectionException(reason, ex);
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", repositoryName);
                throw new OperateRuntimeException(reason);
            }
            if (this.isSnapshotMissingException(e)) {
                return new ArrayList<GetBackupStateResponseDto>();
            }
            String reason = String.format("Exception occurred when searching for backups: %s", e.getMessage());
            throw new OperateRuntimeException(reason, (Throwable)e);
        }
    }

    @Override
    public void executeSnapshotting(BackupService.SnapshotRequest snapshotRequest, Runnable onSuccess, Runnable onFailure) {
        CreateSnapshotRequest request = new CreateSnapshotRequest().repository(snapshotRequest.repositoryName()).snapshot(snapshotRequest.snapshotName()).indices(snapshotRequest.indices()).indicesOptions(IndicesOptions.fromOptions((boolean)false, (boolean)true, (boolean)true, (boolean)true)).userMetadata((Map)this.objectMapper.convertValue((Object)snapshotRequest.metadata(), (TypeReference)new TypeReference<Map<String, Object>>(this){})).featureStates(new String[]{"none"}).waitForCompletion(true);
        CreateSnapshotListener listener = new CreateSnapshotListener(snapshotRequest, onSuccess, onFailure);
        this.esClient.snapshot().createAsync(request, RequestOptions.DEFAULT, (ActionListener)listener);
    }

    private ActionListener<AcknowledgedResponse> getDeleteListener() {
        return new ActionListener<AcknowledgedResponse>(){

            public void onResponse(AcknowledgedResponse response) {
                LOGGER.debug("Delete snapshot was acknowledged by Elasticsearch node: " + response.isAcknowledged());
            }

            public void onFailure(Exception e) {
                if (ElasticsearchBackupRepository.this.isSnapshotMissingException(e)) {
                    LOGGER.warn("No snapshot found for snapshot deletion: " + e.getMessage());
                } else {
                    LOGGER.error("Exception occurred while deleting the snapshot: " + e.getMessage(), (Throwable)e);
                }
            }
        };
    }

    private boolean isSnapshotMissingException(Exception e) {
        return e instanceof ElasticsearchStatusException && ((ElasticsearchStatusException)e).getDetailedMessage().contains(SNAPSHOT_MISSING_EXCEPTION_TYPE);
    }

    private boolean isRepositoryMissingException(Exception e) {
        return e instanceof ElasticsearchStatusException && ((ElasticsearchStatusException)e).getDetailedMessage().contains(REPOSITORY_MISSING_EXCEPTION_TYPE);
    }

    protected List<SnapshotInfo> findSnapshots(String repositoryName, Long backupId) {
        GetSnapshotsRequest snapshotsStatusRequest = new GetSnapshotsRequest().repository(repositoryName).snapshots(new String[]{Metadata.buildSnapshotNamePrefix(backupId) + "*"});
        try {
            GetSnapshotsResponse response = this.esClient.snapshot().get(snapshotsStatusRequest, RequestOptions.DEFAULT);
            return response.getSnapshots();
        }
        catch (IOException | TransportException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for snapshots. Repository name: [%s].", repositoryName);
            throw new OperateElasticsearchConnectionException(reason, ex);
        }
        catch (Exception e) {
            if (this.isSnapshotMissingException(e)) {
                throw new ResourceNotFoundException(String.format("No backup with id [%s] found.", backupId));
            }
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", repositoryName);
                throw new OperateRuntimeException(reason);
            }
            String reason = String.format("Exception occurred when searching for backup with ID [%s].", backupId);
            throw new OperateRuntimeException(reason, (Throwable)e);
        }
    }

    protected boolean isSnapshotFinishedWithinTimeout(String repositoryName, String snapshotName) {
        int count = 0;
        long startTime = System.currentTimeMillis();
        int snapshotTimeout = this.operateProperties.getBackup().getSnapshotTimeout();
        long backupId = Metadata.extractBackupIdFromSnapshotName(snapshotName);
        while (snapshotTimeout == 0 || System.currentTimeMillis() - startTime <= (long)(snapshotTimeout * 1000)) {
            List<SnapshotInfo> snapshotInfos = this.findSnapshots(repositoryName, backupId);
            SnapshotInfo currentSnapshot = snapshotInfos.stream().filter(x -> Objects.equals(x.snapshotId().getName(), snapshotName)).findFirst().orElse(null);
            if (currentSnapshot == null) {
                LOGGER.error(String.format("Expected (but not found) snapshot [%s] for backupId [%d].", snapshotName, backupId));
                return false;
            }
            if (currentSnapshot.state() == SnapshotState.IN_PROGRESS) {
                ThreadUtil.sleepFor((long)100L);
                if (++count % 600 != 0) continue;
                LOGGER.info(String.format("Waiting for snapshot [%s] to finish.", snapshotName));
                continue;
            }
            return this.snapshotWentWell(currentSnapshot);
        }
        LOGGER.error(String.format("Snapshot [%s] did not finish after configured timeout. Snapshot process won't continue.", snapshotName));
        return false;
    }

    private boolean snapshotWentWell(SnapshotInfo snapshotInfo) {
        if (snapshotInfo.state() == SnapshotState.SUCCESS) {
            LOGGER.info("Snapshot done: " + String.valueOf(snapshotInfo.snapshotId()));
            return true;
        }
        if (snapshotInfo.state() == SnapshotState.FAILED) {
            LOGGER.error(String.format("Snapshot taking failed for %s, reason %s", snapshotInfo.snapshotId(), snapshotInfo.reason()));
            return false;
        }
        LOGGER.warn(String.format("Snapshot state is %s for snapshot %s", snapshotInfo.state(), snapshotInfo.snapshotId()));
        return false;
    }

    private GetBackupStateResponseDto getBackupResponse(Long backupId, List<SnapshotInfo> snapshots) {
        GetBackupStateResponseDto response = new GetBackupStateResponseDto(backupId);
        Metadata metadata = (Metadata)this.objectMapper.convertValue((Object)snapshots.get(0).userMetadata(), Metadata.class);
        Integer expectedSnapshotsCount = metadata.getPartCount();
        if (snapshots.size() == expectedSnapshotsCount.intValue() && snapshots.stream().map(SnapshotInfo::state).allMatch(s -> SnapshotState.SUCCESS.equals(s))) {
            response.setState(BackupStateDto.COMPLETED);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch(s -> SnapshotState.FAILED.equals(s) || SnapshotState.PARTIAL.equals(s))) {
            response.setState(BackupStateDto.FAILED);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch(s -> SnapshotState.INCOMPATIBLE.equals(s))) {
            response.setState(BackupStateDto.INCOMPATIBLE);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch(s -> SnapshotState.IN_PROGRESS.equals(s))) {
            response.setState(BackupStateDto.IN_PROGRESS);
        } else if (snapshots.size() < expectedSnapshotsCount) {
            response.setState(BackupStateDto.INCOMPLETE);
        } else {
            response.setState(BackupStateDto.FAILED);
        }
        ArrayList<GetBackupStateResponseDetailDto> details = new ArrayList<GetBackupStateResponseDetailDto>();
        for (SnapshotInfo snapshot : snapshots) {
            GetBackupStateResponseDetailDto detail = new GetBackupStateResponseDetailDto();
            detail.setSnapshotName(snapshot.snapshotId().getName());
            detail.setStartTime(OffsetDateTime.ofInstant(Instant.ofEpochMilli(snapshot.startTime()), ZoneId.systemDefault()));
            if (snapshot.shardFailures() != null) {
                detail.setFailures((String[])snapshot.shardFailures().stream().map(SnapshotShardFailure::toString).toArray(String[]::new));
            }
            detail.setState(snapshot.state().name());
            details.add(detail);
        }
        response.setDetails(details);
        if (response.getState().equals((Object)BackupStateDto.FAILED)) {
            String failureReason = null;
            String failedSnapshots = snapshots.stream().filter(s -> s.state().equals((Object)SnapshotState.FAILED)).map(s -> s.snapshotId().getName()).collect(Collectors.joining(", "));
            if (!failedSnapshots.isEmpty()) {
                failureReason = String.format("There were failures with the following snapshots: %s", failedSnapshots);
            } else {
                String partialSnapshot = snapshots.stream().filter(s -> s.state().equals((Object)SnapshotState.PARTIAL)).map(s -> s.snapshotId().getName()).collect(Collectors.joining(", "));
                if (!partialSnapshot.isEmpty()) {
                    failureReason = String.format("Some of the snapshots are partial: %s", partialSnapshot);
                } else if (snapshots.size() > expectedSnapshotsCount) {
                    failureReason = "More snapshots found than expected.";
                }
            }
            if (failureReason != null) {
                response.setFailureReason(failureReason);
            }
        }
        return response;
    }

    public class CreateSnapshotListener
    implements ActionListener<CreateSnapshotResponse> {
        private final BackupService.SnapshotRequest snapshotRequest;
        private final long backupId;
        private final Runnable onSuccess;
        private final Runnable onFailure;

        public CreateSnapshotListener(BackupService.SnapshotRequest snapshotRequest, Runnable onSuccess, Runnable onFailure) {
            this.snapshotRequest = snapshotRequest;
            this.backupId = Metadata.extractBackupIdFromSnapshotName(snapshotRequest.snapshotName());
            this.onSuccess = onSuccess;
            this.onFailure = onFailure;
        }

        public void onResponse(CreateSnapshotResponse response) {
            if (ElasticsearchBackupRepository.this.snapshotWentWell(response.getSnapshotInfo())) {
                this.onSuccess.run();
            } else {
                this.onFailure.run();
            }
        }

        public void onFailure(Exception ex) {
            if (ex instanceof SocketTimeoutException) {
                int snapshotTimeout = ElasticsearchBackupRepository.this.operateProperties.getBackup().getSnapshotTimeout();
                LOGGER.warn(String.format("Socket timeout while creating snapshot [%s] for backup id [%d]. Start waiting with polling timeout, %s", this.snapshotRequest.snapshotName(), this.backupId, snapshotTimeout == 0 ? "until completion." : "at most " + snapshotTimeout + " seconds."));
                if (ElasticsearchBackupRepository.this.isSnapshotFinishedWithinTimeout(this.snapshotRequest.snapshotName(), this.snapshotRequest.repositoryName())) {
                    this.onSuccess.run();
                } else {
                    this.onFailure.run();
                }
            } else {
                LOGGER.error(String.format("Exception while creating snapshot [%s] for backup id [%d].", this.snapshotRequest.snapshotName(), this.backupId), (Throwable)ex);
                this.onFailure.run();
            }
        }
    }
}

