/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.storage.journal;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.atomix.storage.StorageException;
import io.atomix.storage.StorageLevel;
import io.atomix.storage.buffer.FileBuffer;
import io.atomix.storage.buffer.HeapBuffer;
import io.atomix.storage.buffer.MappedBuffer;
import io.atomix.storage.journal.Journal;
import io.atomix.storage.journal.JournalSegment;
import io.atomix.storage.journal.JournalSegmentDescriptor;
import io.atomix.storage.journal.JournalSegmentFile;
import io.atomix.storage.journal.SegmentedJournalReader;
import io.atomix.storage.journal.SegmentedJournalWriter;
import io.atomix.utils.serializer.Serializer;
import java.io.File;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentedJournal<E>
implements Journal<E> {
    private static final int DEFAULT_BUFFER_SIZE = 65536;
    private static final int SEGMENT_BUFFER_FACTOR = 3;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final String name;
    private final StorageLevel storageLevel;
    private final File directory;
    private final Serializer serializer;
    private final int maxSegmentSize;
    private final int maxEntriesPerSegment;
    private final double indexDensity;
    private final int cacheSize;
    private final NavigableMap<Long, JournalSegment<E>> segments = new ConcurrentSkipListMap<Long, JournalSegment<E>>();
    private final Collection<SegmentedJournalReader<E>> readers = Sets.newConcurrentHashSet();
    private JournalSegment<E> currentSegment;
    private final SegmentedJournalWriter<E> writer;
    private volatile boolean open = true;

    public static <E> Builder<E> builder() {
        return new Builder();
    }

    public SegmentedJournal(String name, StorageLevel storageLevel, File directory, Serializer serializer, int maxSegmentSize, int maxEntriesPerSegment, double indexDensity, int cacheSize) {
        this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name cannot be null");
        this.storageLevel = (StorageLevel)((Object)Preconditions.checkNotNull((Object)((Object)storageLevel), (Object)"storageLevel cannot be null"));
        this.directory = (File)Preconditions.checkNotNull((Object)directory, (Object)"directory cannot be null");
        this.serializer = (Serializer)Preconditions.checkNotNull((Object)serializer, (Object)"serializer cannot be null");
        this.maxSegmentSize = maxSegmentSize;
        this.maxEntriesPerSegment = maxEntriesPerSegment;
        this.indexDensity = indexDensity;
        this.cacheSize = cacheSize;
        this.open();
        this.writer = this.openWriter();
    }

    public String name() {
        return this.name;
    }

    public File directory() {
        return this.directory;
    }

    public StorageLevel storageLevel() {
        return this.storageLevel;
    }

    public int maxSegmentSize() {
        return this.maxSegmentSize;
    }

    public int maxEntriesPerSegment() {
        return this.maxEntriesPerSegment;
    }

    protected SegmentedJournalWriter<E> openWriter() {
        return new SegmentedJournalWriter(this);
    }

    private void open() {
        for (JournalSegment<E> segment : this.loadSegments()) {
            this.segments.put(segment.descriptor().index(), segment);
        }
        if (!this.segments.isEmpty()) {
            this.currentSegment = this.segments.lastEntry().getValue();
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).withMaxEntries(this.maxEntriesPerSegment).build();
            this.currentSegment = this.createSegment(descriptor);
            this.currentSegment.descriptor().update(System.currentTimeMillis());
            this.segments.put(1L, this.currentSegment);
        }
    }

    private void assertOpen() {
        Preconditions.checkState((this.currentSegment != null ? 1 : 0) != 0, (Object)"journal not open");
    }

    private void assertDiskSpace() {
        if (this.directory().getUsableSpace() < (long)(this.maxSegmentSize() * 3)) {
            throw new StorageException.OutOfDiskSpace("Not enough space to allocate a new journal segment");
        }
    }

    private synchronized void resetCurrentSegment() {
        JournalSegment<E> lastSegment = this.getLastSegment();
        if (lastSegment != null) {
            this.currentSegment = lastSegment;
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).withMaxEntries(this.maxEntriesPerSegment).build();
            this.currentSegment = this.createSegment(descriptor);
            this.segments.put(1L, this.currentSegment);
        }
    }

    JournalSegment<E> resetSegments(long index) {
        this.assertOpen();
        JournalSegment<E> firstSegment = this.getFirstSegment();
        if (index == firstSegment.index()) {
            return firstSegment;
        }
        for (JournalSegment segment : this.segments.values()) {
            segment.close();
            segment.delete();
        }
        this.segments.clear();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(index).withMaxSegmentSize(this.maxSegmentSize).withMaxEntries(this.maxEntriesPerSegment).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(index, this.currentSegment);
        return this.currentSegment;
    }

    JournalSegment<E> getFirstSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment<E>> segment = this.segments.firstEntry();
        return segment != null ? segment.getValue() : null;
    }

    JournalSegment<E> getLastSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment<E>> segment = this.segments.lastEntry();
        return segment != null ? segment.getValue() : null;
    }

    synchronized JournalSegment<E> getNextSegment() {
        this.assertOpen();
        this.assertDiskSpace();
        JournalSegment<E> lastSegment = this.getLastSegment();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(lastSegment != null ? lastSegment.descriptor().id() + 1L : 1L).withIndex(this.currentSegment.lastIndex() + 1L).withMaxSegmentSize(this.maxSegmentSize).withMaxEntries(this.maxEntriesPerSegment).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(descriptor.index(), this.currentSegment);
        return this.currentSegment;
    }

    JournalSegment<E> getNextSegment(long index) {
        Map.Entry<Long, JournalSegment<E>> nextSegment = this.segments.higherEntry(index);
        return nextSegment != null ? nextSegment.getValue() : null;
    }

    synchronized JournalSegment<E> getSegment(long index) {
        this.assertOpen();
        if (this.currentSegment != null && index > this.currentSegment.index()) {
            return this.currentSegment;
        }
        Map.Entry<Long, JournalSegment<E>> segment = this.segments.floorEntry(index);
        if (segment != null) {
            return segment.getValue();
        }
        return this.getFirstSegment();
    }

    synchronized void removeSegment(JournalSegment segment) {
        this.segments.remove(segment.index());
        segment.close();
        segment.delete();
        this.resetCurrentSegment();
    }

    JournalSegment<E> createSegment(JournalSegmentDescriptor descriptor) {
        switch (this.storageLevel) {
            case MEMORY: {
                return this.createMemorySegment(descriptor);
            }
            case MAPPED: {
                return this.createMappedSegment(descriptor);
            }
            case DISK: {
                return this.createDiskSegment(descriptor);
            }
        }
        throw new AssertionError();
    }

    protected JournalSegment<E> newSegment(JournalSegmentFile segmentFile, JournalSegmentDescriptor descriptor) {
        return new JournalSegment(segmentFile, descriptor, this.indexDensity, this.cacheSize, this.serializer);
    }

    private JournalSegment<E> createDiskSegment(JournalSegmentDescriptor descriptor) {
        File segmentFile = JournalSegmentFile.createSegmentFile(this.name, this.directory, descriptor.id());
        FileBuffer buffer = FileBuffer.allocate(segmentFile, Math.min(65536, descriptor.maxSegmentSize()), Integer.MAX_VALUE);
        descriptor.copyTo(buffer);
        JournalSegment<E> segment = this.newSegment(new JournalSegmentFile(segmentFile), descriptor);
        this.log.debug("Created disk segment: {}", segment);
        return segment;
    }

    private JournalSegment<E> createMappedSegment(JournalSegmentDescriptor descriptor) {
        File segmentFile = JournalSegmentFile.createSegmentFile(this.name, this.directory, descriptor.id());
        MappedBuffer buffer = MappedBuffer.allocate(segmentFile, Math.min(65536, descriptor.maxSegmentSize()), Integer.MAX_VALUE);
        descriptor.copyTo(buffer);
        JournalSegment<E> segment = this.newSegment(new JournalSegmentFile(segmentFile), descriptor);
        this.log.debug("Created memory mapped segment: {}", segment);
        return segment;
    }

    private JournalSegment<E> createMemorySegment(JournalSegmentDescriptor descriptor) {
        File segmentFile = JournalSegmentFile.createSegmentFile(this.name, this.directory, descriptor.id());
        HeapBuffer buffer = HeapBuffer.allocate(Math.min(65536, descriptor.maxSegmentSize()), Integer.MAX_VALUE);
        descriptor.copyTo(buffer);
        JournalSegment<E> segment = this.newSegment(new JournalSegmentFile(segmentFile), descriptor);
        this.log.debug("Created memory segment: {}", segment);
        return segment;
    }

    private JournalSegment<E> loadSegment(long segmentId) {
        switch (this.storageLevel) {
            case MEMORY: {
                return this.loadMemorySegment(segmentId);
            }
            case MAPPED: {
                return this.loadMappedSegment(segmentId);
            }
            case DISK: {
                return this.loadDiskSegment(segmentId);
            }
        }
        throw new AssertionError();
    }

    private JournalSegment<E> loadDiskSegment(long segmentId) {
        File file = JournalSegmentFile.createSegmentFile(this.name, this.directory, segmentId);
        FileBuffer buffer = FileBuffer.allocate(file, Math.min(65536, this.maxSegmentSize), Integer.MAX_VALUE);
        JournalSegmentDescriptor descriptor = new JournalSegmentDescriptor(buffer);
        JournalSegment<E> segment = this.newSegment(new JournalSegmentFile(file), descriptor);
        this.log.debug("Loaded disk segment: {} ({})", (Object)descriptor.id(), (Object)file.getName());
        return segment;
    }

    private JournalSegment<E> loadMappedSegment(long segmentId) {
        File file = JournalSegmentFile.createSegmentFile(this.name, this.directory, segmentId);
        MappedBuffer buffer = MappedBuffer.allocate(file, Math.min(65536, this.maxSegmentSize), Integer.MAX_VALUE);
        JournalSegmentDescriptor descriptor = new JournalSegmentDescriptor(buffer);
        JournalSegment<E> segment = this.newSegment(new JournalSegmentFile(file), descriptor);
        this.log.debug("Loaded memory mapped segment: {} ({})", (Object)descriptor.id(), (Object)file.getName());
        return segment;
    }

    private JournalSegment<E> loadMemorySegment(long segmentId) {
        File file = JournalSegmentFile.createSegmentFile(this.name, this.directory, segmentId);
        HeapBuffer buffer = HeapBuffer.allocate(Math.min(65536, this.maxSegmentSize), Integer.MAX_VALUE);
        JournalSegmentDescriptor descriptor = new JournalSegmentDescriptor(buffer);
        JournalSegment<E> segment = this.newSegment(new JournalSegmentFile(file), descriptor);
        this.log.debug("Loaded memory segment: {}", (Object)descriptor.id());
        return segment;
    }

    /*
     * Enabled aggressive block sorting
     */
    protected Collection<JournalSegment<E>> loadSegments() {
        this.directory.mkdirs();
        TreeMap<Long, JournalSegment<E>> segments = new TreeMap<Long, JournalSegment<E>>();
        for (File file : this.directory.listFiles(File::isFile)) {
            JournalSegment<E> segment;
            JournalSegmentDescriptor descriptor;
            JournalSegmentFile segmentFile;
            block9: {
                if (!JournalSegmentFile.isSegmentFile(this.name, file)) continue;
                segmentFile = new JournalSegmentFile(file);
                descriptor = new JournalSegmentDescriptor(FileBuffer.allocate(file, 64));
                segment = this.loadSegment(descriptor.id());
                Map.Entry previousEntry = segments.floorEntry(segment.index());
                if (previousEntry != null) {
                    JournalSegment previousSegment = (JournalSegment)previousEntry.getValue();
                    if (previousSegment.index() == segment.index()) {
                        if (segment.descriptor().version() > previousSegment.descriptor().version()) {
                            this.log.debug("Replaced segment {} with newer version: {} ({})", new Object[]{previousSegment.descriptor().id(), segment.descriptor().version(), segmentFile.file().getName()});
                            segments.remove(previousEntry.getKey());
                            previousSegment.close();
                            previousSegment.delete();
                            break block9;
                        } else {
                            segment.close();
                            segment.delete();
                            continue;
                        }
                    }
                    if (previousSegment.index() + previousSegment.length() > segment.index()) {
                        segment.close();
                        segment.delete();
                        continue;
                    }
                }
            }
            this.log.debug("Found segment: {} ({})", (Object)segment.descriptor().id(), (Object)segmentFile.file().getName());
            segments.put(segment.index(), segment);
            Map.Entry nextEntry = segments.higherEntry(segment.index());
            while (nextEntry != null && ((JournalSegment)nextEntry.getValue()).index() < segment.index() + segment.length()) {
                segments.remove(nextEntry.getKey());
                nextEntry = segments.higherEntry(segment.index());
            }
            descriptor.close();
        }
        Iterator iterator = segments.keySet().iterator();
        while (iterator.hasNext()) {
            JournalSegment previousSegment;
            Long segmentId = (Long)iterator.next();
            JournalSegment segment = (JournalSegment)segments.get(segmentId);
            Map.Entry previousEntry = segments.floorEntry(segmentId - 1L);
            if (previousEntry == null || (previousSegment = (JournalSegment)previousEntry.getValue()).lastIndex() == segment.index() - 1L) continue;
            this.log.warn("Found misaligned segment {}", (Object)segment);
            segments.remove(segmentId);
        }
        return segments.values();
    }

    void resetHead(long index) {
        for (SegmentedJournalReader<E> reader : this.readers) {
            if (reader.getNextIndex() >= index) continue;
            reader.reset(index);
        }
    }

    void resetTail(long index) {
        for (SegmentedJournalReader<E> reader : this.readers) {
            if (reader.getNextIndex() < index) continue;
            reader.reset(index);
        }
    }

    @Override
    public SegmentedJournalWriter<E> writer() {
        return this.writer;
    }

    @Override
    public SegmentedJournalReader<E> openReader(long index) {
        SegmentedJournalReader reader = new SegmentedJournalReader(this, index);
        this.readers.add(reader);
        return reader;
    }

    void closeReader(SegmentedJournalReader<E> reader) {
        this.readers.remove(reader);
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    public boolean isCompactable(long index) {
        Map.Entry<Long, JournalSegment<E>> segmentEntry = this.segments.floorEntry(index);
        return segmentEntry != null && this.segments.headMap(segmentEntry.getValue().index()).size() > 0;
    }

    public long getCompactableIndex(long index) {
        Map.Entry<Long, JournalSegment<E>> segmentEntry = this.segments.floorEntry(index);
        return segmentEntry != null ? segmentEntry.getValue().index() : 0L;
    }

    public void compact(long index) {
        SortedMap<Long, JournalSegment<E>> compactSegments;
        Map.Entry<Long, JournalSegment<E>> segmentEntry = this.segments.floorEntry(index);
        if (segmentEntry != null && !(compactSegments = this.segments.headMap(segmentEntry.getValue().index())).isEmpty()) {
            this.log.debug("{} - Compacting {} segment(s)", (Object)this.name, (Object)compactSegments.size());
            for (JournalSegment<E> segment : compactSegments.values()) {
                this.log.trace("Deleting segment: {}", segment);
                segment.close();
                segment.delete();
            }
            compactSegments.clear();
            this.resetHead(segmentEntry.getValue().index());
        }
    }

    @Override
    public void close() {
        this.segments.values().forEach(segment -> {
            this.log.debug("Closing segment: {}", segment);
            segment.close();
        });
        this.currentSegment = null;
        this.open = false;
    }

    public static class Builder<E>
    implements io.atomix.utils.Builder<SegmentedJournal<E>> {
        private static final String DEFAULT_NAME = "atomix";
        private static final String DEFAULT_DIRECTORY = System.getProperty("user.dir");
        private static final int DEFAULT_MAX_SEGMENT_SIZE = 0x2000000;
        private static final int DEFAULT_MAX_ENTRIES_PER_SEGMENT = 0x100000;
        private static final double DEFAULT_INDEX_DENSITY = 0.005;
        private static final int DEFAULT_CACHE_SIZE = 1024;
        protected String name = "atomix";
        protected StorageLevel storageLevel = StorageLevel.DISK;
        protected File directory = new File(DEFAULT_DIRECTORY);
        protected Serializer serializer;
        protected int maxSegmentSize = 0x2000000;
        protected int maxEntriesPerSegment = 0x100000;
        protected double indexDensity = 0.005;
        protected int cacheSize = 1024;

        protected Builder() {
        }

        public Builder<E> withName(String name) {
            this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name cannot be null");
            return this;
        }

        public Builder<E> withStorageLevel(StorageLevel storageLevel) {
            this.storageLevel = (StorageLevel)((Object)Preconditions.checkNotNull((Object)((Object)storageLevel), (Object)"storageLevel cannot be null"));
            return this;
        }

        public Builder<E> withDirectory(String directory) {
            return this.withDirectory(new File((String)Preconditions.checkNotNull((Object)directory, (Object)"directory cannot be null")));
        }

        public Builder<E> withDirectory(File directory) {
            this.directory = (File)Preconditions.checkNotNull((Object)directory, (Object)"directory cannot be null");
            return this;
        }

        public Builder<E> withSerializer(Serializer serializer) {
            this.serializer = (Serializer)Preconditions.checkNotNull((Object)serializer, (Object)"serializer cannot be null");
            return this;
        }

        public Builder<E> withMaxSegmentSize(int maxSegmentSize) {
            Preconditions.checkArgument((maxSegmentSize > 64 ? 1 : 0) != 0, (Object)"maxSegmentSize must be greater than 64");
            this.maxSegmentSize = maxSegmentSize;
            return this;
        }

        public Builder<E> withMaxEntriesPerSegment(int maxEntriesPerSegment) {
            Preconditions.checkArgument((maxEntriesPerSegment > 0 ? 1 : 0) != 0, (Object)"max entries per segment must be positive");
            Preconditions.checkArgument((maxEntriesPerSegment <= 0x100000 ? 1 : 0) != 0, (Object)"max entries per segment cannot be greater than 1048576");
            this.maxEntriesPerSegment = maxEntriesPerSegment;
            return this;
        }

        public Builder<E> withIndexDensity(double indexDensity) {
            Preconditions.checkArgument((indexDensity > 0.0 && indexDensity < 1.0 ? 1 : 0) != 0, (Object)"index density must be between 0 and 1");
            this.indexDensity = indexDensity;
            return this;
        }

        public Builder<E> withCacheSize(int cacheSize) {
            Preconditions.checkArgument((cacheSize >= 0 ? 1 : 0) != 0, (Object)"cacheSize must be positive");
            this.cacheSize = cacheSize;
            return this;
        }

        public SegmentedJournal<E> build() {
            return new SegmentedJournal(this.name, this.storageLevel, this.directory, this.serializer, this.maxSegmentSize, this.maxEntriesPerSegment, this.indexDensity, this.cacheSize);
        }
    }
}

