/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.protocols.raft.storage.compactor;

import io.atomix.protocols.raft.impl.RaftContext;
import io.atomix.protocols.raft.service.RaftServiceContext;
import io.atomix.storage.StorageLevel;
import io.atomix.utils.concurrent.ComposableFuture;
import io.atomix.utils.concurrent.OrderedFuture;
import io.atomix.utils.concurrent.ThreadContext;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RaftLogCompactor {
    private static final Logger LOGGER = LoggerFactory.getLogger(RaftLogCompactor.class);
    private static final Duration SNAPSHOT_INTERVAL = Duration.ofSeconds(10L);
    private static final Duration MIN_COMPACT_INTERVAL = Duration.ofSeconds(10L);
    private static final int SEGMENT_BUFFER_FACTOR = 5;
    private final RaftContext raft;
    private final ThreadContext threadContext;
    private final Random random = new Random();
    private volatile CompletableFuture<Void> compactFuture;
    private long lastCompacted;

    public RaftLogCompactor(RaftContext raft, ThreadContext threadContext) {
        this.raft = raft;
        this.threadContext = threadContext;
        this.scheduleSnapshots();
    }

    private boolean isRunningOutOfDiskSpace() {
        return this.raft.getStorage().statistics().getUsableSpace() < (long)(this.raft.getStorage().maxLogSegmentSize() * 5) || (double)this.raft.getStorage().statistics().getUsableSpace() / (double)this.raft.getStorage().statistics().getTotalSpace() < this.raft.getStorage().freeDiskBuffer();
    }

    private void scheduleSnapshots() {
        this.threadContext.schedule(SNAPSHOT_INTERVAL, () -> this.snapshotServices(true, false));
    }

    public CompletableFuture<Void> compact() {
        return this.snapshotServices(false, true);
    }

    private synchronized CompletableFuture<Void> snapshotServices(boolean rescheduleAfterCompletion, boolean force) {
        if (this.compactFuture != null) {
            if (rescheduleAfterCompletion) {
                this.compactFuture.whenComplete((r, e) -> this.scheduleSnapshots());
            }
            return this.compactFuture;
        }
        long lastApplied = this.raft.getLastApplied();
        if (this.raft.getLog().isCompactable(lastApplied) && this.raft.getLog().getCompactableIndex(lastApplied) > this.lastCompacted) {
            boolean runningOutOfDiskSpace = this.isRunningOutOfDiskSpace();
            if (!force && this.raft.getStorage().storageLevel() != StorageLevel.MEMORY && this.raft.getStorage().dynamicCompaction() && !runningOutOfDiskSpace && this.raft.getLoadMonitor().isUnderHighLoad()) {
                LOGGER.debug("Skipping compaction due to high load");
                if (rescheduleAfterCompletion) {
                    this.scheduleSnapshots();
                }
                return CompletableFuture.completedFuture(null);
            }
            LOGGER.debug("Snapshotting services");
            this.lastCompacted = lastApplied;
            ArrayList<RaftServiceContext> services = new ArrayList<RaftServiceContext>(this.raft.getServices().copyValues());
            this.compactFuture = new OrderedFuture();
            this.snapshotServices(services, lastApplied, force || runningOutOfDiskSpace).whenComplete((result, error) -> {
                if (force) {
                    this.compactLogs(lastApplied);
                } else {
                    this.scheduleCompaction(lastApplied);
                }
            });
            if (rescheduleAfterCompletion) {
                this.compactFuture.whenComplete((r, e) -> this.scheduleSnapshots());
            }
            return this.compactFuture;
        }
        if (rescheduleAfterCompletion) {
            this.scheduleSnapshots();
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> snapshotServices(List<RaftServiceContext> services, long index, boolean force) {
        return this.snapshotServices(services, index, force, 0, new ArrayList<CompletableFuture<Void>>());
    }

    private CompletableFuture<Void> snapshotServices(List<RaftServiceContext> services, long index, boolean force, int attempt, List<CompletableFuture<Void>> futures) {
        if (services.isEmpty()) {
            return CompletableFuture.allOf(futures.toArray(new CompletableFuture[futures.size()]));
        }
        RaftServiceContext nextService = this.selectService(services, force);
        if (nextService != null) {
            futures.add((CompletableFuture<Void>)nextService.takeSnapshot(index).thenCompose(snapshotIndex -> {
                if (force) {
                    return nextService.completeSnapshot((long)snapshotIndex);
                }
                return this.scheduleCompletion(nextService, (long)snapshotIndex);
            }));
            return this.snapshotServices(services, index, force, 0, futures);
        }
        return this.rescheduleSnapshots(services, index, force, attempt, futures);
    }

    private CompletableFuture<Void> rescheduleSnapshots(List<RaftServiceContext> services, long index, boolean force, int attempt, List<CompletableFuture<Void>> futures) {
        ComposableFuture future = new ComposableFuture();
        this.threadContext.schedule(Duration.ofSeconds(Math.min(2 ^ attempt, 10)), () -> this.snapshotServices(services, index, force || this.isRunningOutOfDiskSpace(), attempt + 1, futures).whenComplete((BiConsumer)future));
        return future;
    }

    private RaftServiceContext selectService(List<RaftServiceContext> services, boolean force) {
        Iterator<RaftServiceContext> iterator = services.iterator();
        while (iterator.hasNext()) {
            RaftServiceContext serviceContext = iterator.next();
            if (!force && this.raft.getStorage().dynamicCompaction() && serviceContext.isUnderHighLoad()) continue;
            iterator.remove();
            return serviceContext;
        }
        return null;
    }

    private CompletableFuture<Void> scheduleCompletion(RaftServiceContext serviceContext, long snapshotIndex) {
        ComposableFuture future = new ComposableFuture();
        Duration delay = SNAPSHOT_INTERVAL.plusMillis(this.random.nextInt((int)SNAPSHOT_INTERVAL.toMillis()));
        this.threadContext.schedule(delay, () -> serviceContext.completeSnapshot(snapshotIndex).whenComplete((BiConsumer)future));
        return future;
    }

    private void scheduleCompaction(long lastApplied) {
        Duration delay = MIN_COMPACT_INTERVAL.plusMillis(this.random.nextInt((int)MIN_COMPACT_INTERVAL.toMillis()));
        LOGGER.trace("Scheduling compaction in {}", (Object)delay);
        this.threadContext.schedule(delay, () -> this.compactLogs(lastApplied));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void compactLogs(long compactIndex) {
        LOGGER.debug("Compacting logs up to index {}", (Object)compactIndex);
        try {
            this.raft.getLog().compact(compactIndex);
        }
        catch (Exception e) {
            LOGGER.error("An exception occurred during log compaction: {}", (Throwable)e);
        }
        finally {
            this.compactFuture.complete(null);
            this.compactFuture = null;
            this.compact();
        }
    }
}

