/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.appium.java_client.android;

/**
 * Some common key codes for Android Key Events
 */
public interface AndroidKeyCode {

    int BACK = 4;
    int BACKSPACE = 67;
    int DEL = 67;
    int ENTER = 66;
    int HOME = 3;
    int MENU = 82;
    int SETTINGS = 176;
    int SPACE = 62;

    // getAction() value: the key has been pressed down.
    // (0x00000000)
    int ACTION_DOWN = 0;

    // getAction() value: multiple duplicate key events have occurred in a row, or a complex string is being delivered. If the key code is not {#link KEYCODE_UNKNOWN then the {#link getRepeatCount() method returns the number of times the given key code should be executed. Otherwise, if the key code is KEYCODE_UNKNOWN, then this is a sequence of characters as returned by getCharacters().
    // (0x00000002)
    int ACTION_MULTIPLE = 2;

    // getAction() value: the key has been released.
    // (0x00000001)
    int ACTION_UP = 1;

    // When associated with up key events, this indicates that the key press has been canceled. Typically this is used with virtual touch screen keys, where the user can slide from the virtual key area on to the display: in that case, the application will receive a canceled up event and should not perform the action normally associated with the key. Note that for this to work, the application can not perform an action for a key until it receives an up or the long press timeout has expired.
    // (0x00000020)
    int FLAG_CANCELED = 32;

    // Set when a key event has FLAG_CANCELED set because a long press action was executed while it was down.
    // (0x00000100)
    int FLAG_CANCELED_LONG_PRESS = 256;

    // This mask is used for compatibility, to identify enter keys that are coming from an IME whose enter key has been auto-labelled "next" or "done". This allows TextView to dispatch these as normal enter keys for old applications, but still do the appropriate action when receiving them.
    // (0x00000010)
    int FLAG_EDITOR_ACTION = 16;

    // Set when a key event has been synthesized to implement default behavior for an event that the application did not handle. Fallback key events are generated by unhandled trackball motions (to emulate a directional keypad) and by certain unhandled key presses that are declared in the key map (such as special function numeric keypad keys when numlock is off).
    // (0x00000400)
    int FLAG_FALLBACK = 1024;

    // This mask is set if an event was known to come from a trusted part of the system. That is, the event is known to come from the user, and could not have been spoofed by a third party component.
    // (0x00000008)
    int FLAG_FROM_SYSTEM = 8;

    // This mask is set if we don't want the key event to cause us to leave touch mode.
    // (0x00000004)
    int FLAG_KEEP_TOUCH_MODE = 4;

    // This flag is set for the first key repeat that occurs after the long press timeout.
    // (0x00000080)
    int FLAG_LONG_PRESS = 128;

    // This mask is set if the key event was generated by a software keyboard.
    // (0x00000002)
    int FLAG_SOFT_KEYBOARD = 2;

    // Set for ACTION_UP when this event's key code is still being tracked from its initial down. That is, somebody requested that tracking started on the key down and a long press has not caused the tracking to be canceled.
    // (0x00000200)
    int FLAG_TRACKING = 512;

    // This key event was generated by a virtual (on-screen) hard key area. Typically this is an area of the touchscreen, outside of the regular display, dedicated to "hardware" buttons.
    // (0x00000040)
    int FLAG_VIRTUAL_HARD_KEY = 64;

    // This constant was deprecated in API level 20.
    // This flag will never be set by the system since the system consumes all wake keys itself.
    // This mask is set if the device woke because of this key event.
    // (0x00000001)
    int FLAG_WOKE_HERE = 1;

    // Key code constant: '0' key.
    // (0x00000007)
    int KEYCODE_0 = 7;

    // Key code constant: '1' key.
    // (0x00000008)
    int KEYCODE_1 = 8;

    // Key code constant: '11' key.
    // (0x000000e3)
    int KEYCODE_11 = 227;

    // Key code constant: '12' key.
    // (0x000000e4)
    int KEYCODE_12 = 228;

    // Key code constant: '2' key.
    // (0x00000009)
    int KEYCODE_2 = 9;

    // Key code constant: '3' key.
    // (0x0000000a)
    int KEYCODE_3 = 10;

    // Key code constant: 3D Mode key. Toggles the display between 2D and 3D mode.
    // (0x000000ce)
    int KEYCODE_3D_MODE = 206;

    // Key code constant: '4' key.
    // (0x0000000b)
    int KEYCODE_4 = 11;

    // Key code constant: '5' key.
    // (0x0000000c)
    int KEYCODE_5 = 12;

    // Key code constant: '6' key.
    // (0x0000000d)
    int KEYCODE_6 = 13;

    // Key code constant: '7' key.
    // (0x0000000e)
    int KEYCODE_7 = 14;

    // Key code constant: '8' key.
    // (0x0000000f)
    int KEYCODE_8 = 15;

    // Key code constant: '9' key.
    // (0x00000010)
    int KEYCODE_9 = 16;

    // Key code constant: 'A' key.
    // (0x0000001d)
    int KEYCODE_A = 29;

    // Key code constant: Left Alt modifier key.
    // (0x00000039)
    int KEYCODE_ALT_LEFT = 57;

    // Key code constant: Right Alt modifier key.
    // (0x0000003a)
    int KEYCODE_ALT_RIGHT = 58;

    // Key code constant: ''' (apostrophe) key.
    // (0x0000004b)
    int KEYCODE_APOSTROPHE = 75;

    // Key code constant: App switch key. Should bring up the application switcher dialog.
    // (0x000000bb)
    int KEYCODE_APP_SWITCH = 187;

    // Key code constant: Assist key. Launches the global assist activity. Not delivered to applications.
    // (0x000000db)
    int KEYCODE_ASSIST = 219;

    // Key code constant: '@' key.
    // (0x0000004d)
    int KEYCODE_AT = 77;

    // Key code constant: A/V Receiver input key. On TV remotes, switches the input mode on an external A/V Receiver.
    // (0x000000b6)
    int KEYCODE_AVR_INPUT = 182;

    // Key code constant: A/V Receiver power key. On TV remotes, toggles the power on an external A/V Receiver.
    // (0x000000b5)
    int KEYCODE_AVR_POWER = 181;

    // Key code constant: 'B' key.
    // (0x0000001e)
    int KEYCODE_B = 30;

    // Key code constant: Back key.
    // (0x00000004)
    int KEYCODE_BACK = 4;

    // Key code constant: '\' key.
    // (0x00000049)
    int KEYCODE_BACKSLASH = 73;

    // Key code constant: Bookmark key. On some TV remotes, bookmarks content or web pages.
    // (0x000000ae)
    int KEYCODE_BOOKMARK = 174;

    // Key code constant: Break / Pause key.
    // (0x00000079)
    int KEYCODE_BREAK = 121;

    // Key code constant: Brightness Down key. Adjusts the screen brightness down.
    // (0x000000dc)
    int KEYCODE_BRIGHTNESS_DOWN = 220;

    // Key code constant: Brightness Up key. Adjusts the screen brightness up.
    // (0x000000dd)
    int KEYCODE_BRIGHTNESS_UP = 221;

    // Key code constant: Generic Game Pad Button #1.
    // (0x000000bc)
    int KEYCODE_BUTTON_1 = 188;

    // Key code constant: Generic Game Pad Button #10.
    // (0x000000c5)
    int KEYCODE_BUTTON_10 = 197;

    // Key code constant: Generic Game Pad Button #11.
    // (0x000000c6)
    int KEYCODE_BUTTON_11 = 198;

    // Key code constant: Generic Game Pad Button #12.
    // (0x000000c7)
    int KEYCODE_BUTTON_12 = 199;

    // Key code constant: Generic Game Pad Button #13.
    // (0x000000c8)
    int KEYCODE_BUTTON_13 = 200;

    // Key code constant: Generic Game Pad Button #14.
    // (0x000000c9)
    int KEYCODE_BUTTON_14 = 201;

    // Key code constant: Generic Game Pad Button #15.
    // (0x000000ca)
    int KEYCODE_BUTTON_15 = 202;

    // Key code constant: Generic Game Pad Button #16.
    // (0x000000cb)
    int KEYCODE_BUTTON_16 = 203;

    // Key code constant: Generic Game Pad Button #2.
    // (0x000000bd)
    int KEYCODE_BUTTON_2 = 189;

    // Key code constant: Generic Game Pad Button #3.
    // (0x000000be)
    int KEYCODE_BUTTON_3 = 190;

    // Key code constant: Generic Game Pad Button #4.
    // (0x000000bf)
    int KEYCODE_BUTTON_4 = 191;

    // Key code constant: Generic Game Pad Button #5.
    // (0x000000c0)
    int KEYCODE_BUTTON_5 = 192;

    // Key code constant: Generic Game Pad Button #6.
    // (0x000000c1)
    int KEYCODE_BUTTON_6 = 193;

    // Key code constant: Generic Game Pad Button #7.
    // (0x000000c2)
    int KEYCODE_BUTTON_7 = 194;

    // Key code constant: Generic Game Pad Button #8.
    // (0x000000c3)
    int KEYCODE_BUTTON_8 = 195;

    // Key code constant: Generic Game Pad Button #9.
    // (0x000000c4)
    int KEYCODE_BUTTON_9 = 196;

    // Key code constant: A Button key. On a game controller, the A button should be either the button labeled A or the first button on the bottom row of controller buttons.
    // (0x00000060)
    int KEYCODE_BUTTON_A = 96;

    // Key code constant: B Button key. On a game controller, the B button should be either the button labeled B or the second button on the bottom row of controller buttons.
    // (0x00000061)
    int KEYCODE_BUTTON_B = 97;

    // Key code constant: C Button key. On a game controller, the C button should be either the button labeled C or the third button on the bottom row of controller buttons.
    // (0x00000062)
    int KEYCODE_BUTTON_C = 98;

    // Key code constant: L1 Button key. On a game controller, the L1 button should be either the button labeled L1 (or L) or the top left trigger button.
    // (0x00000066)
    int KEYCODE_BUTTON_L1 = 102;

    // Key code constant: L2 Button key. On a game controller, the L2 button should be either the button labeled L2 or the bottom left trigger button.
    // (0x00000068)
    int KEYCODE_BUTTON_L2 = 104;

    // Key code constant: Mode Button key. On a game controller, the button labeled Mode.
    // (0x0000006e)
    int KEYCODE_BUTTON_MODE = 110;

    // Key code constant: R1 Button key. On a game controller, the R1 button should be either the button labeled R1 (or R) or the top right trigger button.
    // (0x00000067)
    int KEYCODE_BUTTON_R1 = 103;

    // Key code constant: R2 Button key. On a game controller, the R2 button should be either the button labeled R2 or the bottom right trigger button.
    // (0x00000069)
    int KEYCODE_BUTTON_R2 = 105;

    // Key code constant: Select Button key. On a game controller, the button labeled Select.
    // (0x0000006d)
    int KEYCODE_BUTTON_SELECT = 109;

    // Key code constant: Start Button key. On a game controller, the button labeled Start.
    // (0x0000006c)
    int KEYCODE_BUTTON_START = 108;

    // Key code constant: Left Thumb Button key. On a game controller, the left thumb button indicates that the left (or only) joystick is pressed.
    // (0x0000006a)
    int KEYCODE_BUTTON_THUMBL = 106;

    // Key code constant: Right Thumb Button key. On a game controller, the right thumb button indicates that the right joystick is pressed.
    // (0x0000006b)
    int KEYCODE_BUTTON_THUMBR = 107;

    // Key code constant: X Button key. On a game controller, the X button should be either the button labeled X or the first button on the upper row of controller buttons.
    // (0x00000063)
    int KEYCODE_BUTTON_X = 99;

    // Key code constant: Y Button key. On a game controller, the Y button should be either the button labeled Y or the second button on the upper row of controller buttons.
    // (0x00000064)
    int KEYCODE_BUTTON_Y = 100;

    // Key code constant: Z Button key. On a game controller, the Z button should be either the button labeled Z or the third button on the upper row of controller buttons.
    // (0x00000065)
    int KEYCODE_BUTTON_Z = 101;

    // Key code constant: 'C' key.
    // (0x0000001f)
    int KEYCODE_C = 31;

    // Key code constant: Calculator special function key. Used to launch a calculator application.
    // (0x000000d2)
    int KEYCODE_CALCULATOR = 210;

    // Key code constant: Calendar special function key. Used to launch a calendar application.
    // (0x000000d0)
    int KEYCODE_CALENDAR = 208;

    // Key code constant: Call key.
    // (0x00000005)
    int KEYCODE_CALL = 5;

    // Key code constant: Camera key. Used to launch a camera application or take pictures.
    // (0x0000001b)
    int KEYCODE_CAMERA = 27;

    // Key code constant: Caps Lock key.
    // (0x00000073)
    int KEYCODE_CAPS_LOCK = 115;

    // Key code constant: Toggle captions key. Switches the mode for closed-captioning text, for example during television shows.
    // (0x000000af)
    int KEYCODE_CAPTIONS = 175;

    // Key code constant: Channel down key. On TV remotes, decrements the television channel.
    // (0x000000a7)
    int KEYCODE_CHANNEL_DOWN = 167;

    // Key code constant: Channel up key. On TV remotes, increments the television channel.
    // (0x000000a6)
    int KEYCODE_CHANNEL_UP = 166;

    // Key code constant: Clear key.
    // (0x0000001c)
    int KEYCODE_CLEAR = 28;

    // Key code constant: ',' key.
    // (0x00000037)
    int KEYCODE_COMMA = 55;

    // Key code constant: Contacts special function key. Used to launch an address book application.
    // (0x000000cf)
    int KEYCODE_CONTACTS = 207;

    // Key code constant: Left Control modifier key.
    // (0x00000071)
    int KEYCODE_CTRL_LEFT = 113;

    // Key code constant: Right Control modifier key.
    // (0x00000072)
    int KEYCODE_CTRL_RIGHT = 114;

    // Key code constant: 'D' key.
    // (0x00000020)
    int KEYCODE_D = 32;

    // Key code constant: Backspace key. Deletes characters before the insertion point, unlike KEYCODE_FORWARD_DEL.
    // (0x00000043)
    int KEYCODE_DEL = 67;

    // Key code constant: Directional Pad Center key. May also be synthesized from trackball motions.
    // (0x00000017)
    int KEYCODE_DPAD_CENTER = 23;

    // Key code constant: Directional Pad Down key. May also be synthesized from trackball motions.
    // (0x00000014)
    int KEYCODE_DPAD_DOWN = 20;

    // Key code constant: Directional Pad Left key. May also be synthesized from trackball motions.
    // (0x00000015)
    int KEYCODE_DPAD_LEFT = 21;

    // Key code constant: Directional Pad Right key. May also be synthesized from trackball motions.
    // (0x00000016)
    int KEYCODE_DPAD_RIGHT = 22;

    // Key code constant: Directional Pad Up key. May also be synthesized from trackball motions.
    // (0x00000013)
    int KEYCODE_DPAD_UP = 19;

    // Key code constant: DVR key. On some TV remotes, switches to a DVR mode for recorded shows.
    // (0x000000ad)
    int KEYCODE_DVR = 173;

    // Key code constant: 'E' key.
    // (0x00000021)
    int KEYCODE_E = 33;

    // Key code constant: Japanese alphanumeric key.
    // (0x000000d4)
    int KEYCODE_EISU = 212;

    // Key code constant: End Call key.
    // (0x00000006)
    int KEYCODE_ENDCALL = 6;

    // Key code constant: Enter key.
    // (0x00000042)
    int KEYCODE_ENTER = 66;

    // Key code constant: Envelope special function key. Used to launch a mail application.
    // (0x00000041)
    int KEYCODE_ENVELOPE = 65;

    // Key code constant: '=' key.
    // (0x00000046)
    int KEYCODE_EQUALS = 70;

    // Key code constant: Escape key.
    // (0x0000006f)
    int KEYCODE_ESCAPE = 111;

    // Key code constant: Explorer special function key. Used to launch a browser application.
    // (0x00000040)
    int KEYCODE_EXPLORER = 64;

    // Key code constant: 'F' key.
    // (0x00000022)
    int KEYCODE_F = 34;

    // Key code constant: F1 key.
    // (0x00000083)
    int KEYCODE_F1 = 131;

    // Key code constant: F10 key.
    // (0x0000008c)
    int KEYCODE_F10 = 140;

    // Key code constant: F11 key.
    // (0x0000008d)
    int KEYCODE_F11 = 141;

    // Key code constant: F12 key.
    // (0x0000008e)
    int KEYCODE_F12 = 142;

    // Key code constant: F2 key.
    // (0x00000084)
    int KEYCODE_F2 = 132;

    // Key code constant: F3 key.
    // (0x00000085)
    int KEYCODE_F3 = 133;

    // Key code constant: F4 key.
    // (0x00000086)
    int KEYCODE_F4 = 134;

    // Key code constant: F5 key.
    // (0x00000087)
    int KEYCODE_F5 = 135;

    // Key code constant: F6 key.
    // (0x00000088)
    int KEYCODE_F6 = 136;

    // Key code constant: F7 key.
    // (0x00000089)
    int KEYCODE_F7 = 137;

    // Key code constant: F8 key.
    // (0x0000008a)
    int KEYCODE_F8 = 138;

    // Key code constant: F9 key.
    // (0x0000008b)
    int KEYCODE_F9 = 139;

    // Key code constant: Camera Focus key. Used to focus the camera.
    // (0x00000050)
    int KEYCODE_FOCUS = 80;

    // Key code constant: Forward key. Navigates forward in the history stack. Complement of KEYCODE_BACK.
    // (0x0000007d)
    int KEYCODE_FORWARD = 125;

    // Key code constant: Forward Delete key. Deletes characters ahead of the insertion point, unlike KEYCODE_DEL.
    // (0x00000070)
    int KEYCODE_FORWARD_DEL = 112;

    // Key code constant: Function modifier key.
    // (0x00000077)
    int KEYCODE_FUNCTION = 119;

    // Key code constant: 'G' key.
    // (0x00000023)
    int KEYCODE_G = 35;

    // Key code constant: '`' (backtick) key.
    // (0x00000044)
    int KEYCODE_GRAVE = 68;

    // Key code constant: Guide key. On TV remotes, shows a programming guide.
    // (0x000000ac)
    int KEYCODE_GUIDE = 172;

    // Key code constant: 'H' key.
    // (0x00000024)
    int KEYCODE_H = 36;

    // Key code constant: Headset Hook key. Used to hang up calls and stop media.
    // (0x0000004f)
    int KEYCODE_HEADSETHOOK = 79;

    // Key code constant: Help key.
    // (0x00000103)
    int KEYCODE_HELP = 259;

    // Key code constant: Japanese conversion key.
    // (0x000000d6)
    int KEYCODE_HENKAN = 214;

    // Key code constant: Home key. This key is handled by the framework and is never delivered to applications.
    // (0x00000003)
    int KEYCODE_HOME = 3;

    // Key code constant: 'I' key.
    // (0x00000025)
    int KEYCODE_I = 37;

    // Key code constant: Info key. Common on TV remotes to show additional information related to what is currently being viewed.
    // (0x000000a5)
    int KEYCODE_INFO = 165;

    // Key code constant: Insert key. Toggles insert / overwrite edit mode.
    // (0x0000007c)
    int KEYCODE_INSERT = 124;

    // Key code constant: 'J' key.
    // (0x00000026)
    int KEYCODE_J = 38;

    // Key code constant: 'K' key.
    // (0x00000027)
    int KEYCODE_K = 39;

    // Key code constant: Japanese kana key.
    // (0x000000da)
    int KEYCODE_KANA = 218;

    // Key code constant: Japanese katakana / hiragana key.
    // (0x000000d7)
    int KEYCODE_KATAKANA_HIRAGANA = 215;

    // Key code constant: 'L' key.
    // (0x00000028)
    int KEYCODE_L = 40;

    // Key code constant: Language Switch key. Toggles the current input language such as switching between English and Japanese on a QWERTY keyboard. On some devices, the same function may be performed by pressing Shift+Spacebar.
    // (0x000000cc)
    int KEYCODE_LANGUAGE_SWITCH = 204;

    // Key code constant: Last Channel key. Goes to the last viewed channel.
    // (0x000000e5)
    int KEYCODE_LAST_CHANNEL = 229;

    // Key code constant: '[' key.
    // (0x00000047)
    int KEYCODE_LEFT_BRACKET = 71;

    // Key code constant: 'M' key.
    // (0x00000029)
    int KEYCODE_M = 41;

    // Key code constant: Manner Mode key. Toggles silent or vibrate mode on and off to make the device behave more politely in certain settings such as on a crowded train. On some devices, the key may only operate when long-pressed.
    // (0x000000cd)
    int KEYCODE_MANNER_MODE = 205;

    // Key code constant: Audio Track key. Switches the audio tracks.
    // (0x000000de)
    int KEYCODE_MEDIA_AUDIO_TRACK = 222;

    // Key code constant: Close media key. May be used to close a CD tray, for example.
    // (0x00000080)
    int KEYCODE_MEDIA_CLOSE = 128;

    // Key code constant: Eject media key. May be used to eject a CD tray, for example.
    // (0x00000081)
    int KEYCODE_MEDIA_EJECT = 129;

    // Key code constant: Fast Forward media key.
    // (0x0000005a)
    int KEYCODE_MEDIA_FAST_FORWARD = 90;

    // Key code constant: Play Next media key.
    // (0x00000057)
    int KEYCODE_MEDIA_NEXT = 87;

    // Key code constant: Pause media key.
    // (0x0000007f)
    int KEYCODE_MEDIA_PAUSE = 127;

    // Key code constant: Play media key.
    // (0x0000007e)
    int KEYCODE_MEDIA_PLAY = 126;

    // Key code constant: Play/Pause media key.
    // (0x00000055)
    int KEYCODE_MEDIA_PLAY_PAUSE = 85;

    // Key code constant: Play Previous media key.
    // (0x00000058)
    int KEYCODE_MEDIA_PREVIOUS = 88;

    // Key code constant: Record media key.
    // (0x00000082)
    int KEYCODE_MEDIA_RECORD = 130;

    // Key code constant: Rewind media key.
    // (0x00000059)
    int KEYCODE_MEDIA_REWIND = 89;

    // Key code constant: Stop media key.
    // (0x00000056)
    int KEYCODE_MEDIA_STOP = 86;

    // Key code constant: Media Top Menu key. Goes to the top of media menu.
    // (0x000000e2)
    int KEYCODE_MEDIA_TOP_MENU = 226;

    // Key code constant: Menu key.
    // (0x00000052)
    int KEYCODE_MENU = 82;

    // Key code constant: Left Meta modifier key.
    // (0x00000075)
    int KEYCODE_META_LEFT = 117;

    // Key code constant: Right Meta modifier key.
    // (0x00000076)
    int KEYCODE_META_RIGHT = 118;

    // Key code constant: '-'.
    // (0x00000045)
    int KEYCODE_MINUS = 69;

    // Key code constant: End Movement key. Used for scrolling or moving the cursor around to the end of a line or to the bottom of a list.
    // (0x0000007b)
    int KEYCODE_MOVE_END = 123;

    // Key code constant: Home Movement key. Used for scrolling or moving the cursor around to the start of a line or to the top of a list.
    // (0x0000007a)
    int KEYCODE_MOVE_HOME = 122;

    // Key code constant: Japanese non-conversion key.
    // (0x000000d5)
    int KEYCODE_MUHENKAN = 213;

    // Key code constant: Music special function key. Used to launch a music player application.
    // (0x000000d1)
    int KEYCODE_MUSIC = 209;

    // Key code constant: Mute key. Mutes the microphone, unlike KEYCODE_VOLUME_MUTE.
    // (0x0000005b)
    int KEYCODE_MUTE = 91;

    // Key code constant: 'N' key.
    // (0x0000002a)
    int KEYCODE_N = 42;

    // Key code constant: Notification key.
    // (0x00000053)
    int KEYCODE_NOTIFICATION = 83;

    // Key code constant: Number modifier key. Used to enter numeric symbols. This key is not Num Lock; it is more like KEYCODE_ALT_LEFT and is interpreted as an ALT key by MetaKeyKeyListener.
    // (0x0000004e)
    int KEYCODE_NUM = 78;

    // Key code constant: Numeric keypad '0' key.
    // (0x00000090)
    int KEYCODE_NUMPAD_0 = 144;

    // Key code constant: Numeric keypad '1' key.
    // (0x00000091)
    int KEYCODE_NUMPAD_1 = 145;

    // Key code constant: Numeric keypad '2' key.
    // (0x00000092)
    int KEYCODE_NUMPAD_2 = 146;

    // Key code constant: Numeric keypad '3' key.
    // (0x00000093)
    int KEYCODE_NUMPAD_3 = 147;

    // Key code constant: Numeric keypad '4' key.
    // (0x00000094)
    int KEYCODE_NUMPAD_4 = 148;

    // Key code constant: Numeric keypad '5' key.
    // (0x00000095)
    int KEYCODE_NUMPAD_5 = 149;

    // Key code constant: Numeric keypad '6' key.
    // (0x00000096)
    int KEYCODE_NUMPAD_6 = 150;

    // Key code constant: Numeric keypad '7' key.
    // (0x00000097)
    int KEYCODE_NUMPAD_7 = 151;

    // Key code constant: Numeric keypad '8' key.
    // (0x00000098)
    int KEYCODE_NUMPAD_8 = 152;

    // Key code constant: Numeric keypad '9' key.
    // (0x00000099)
    int KEYCODE_NUMPAD_9 = 153;

    // Key code constant: Numeric keypad '+' key (for addition).
    // (0x0000009d)
    int KEYCODE_NUMPAD_ADD = 157;

    // Key code constant: Numeric keypad ',' key (for decimals or digit grouping).
    // (0x0000009f)
    int KEYCODE_NUMPAD_COMMA = 159;

    // Key code constant: Numeric keypad '/' key (for division).
    // (0x0000009a)
    int KEYCODE_NUMPAD_DIVIDE = 154;

    // Key code constant: Numeric keypad '.' key (for decimals or digit grouping).
    // (0x0000009e)
    int KEYCODE_NUMPAD_DOT = 158;

    // Key code constant: Numeric keypad Enter key.
    // (0x000000a0)
    int KEYCODE_NUMPAD_ENTER = 160;

    // Key code constant: Numeric keypad '=' key.
    // (0x000000a1)
    int KEYCODE_NUMPAD_EQUALS = 161;

    // Key code constant: Numeric keypad '(' key.
    // (0x000000a2)
    int KEYCODE_NUMPAD_LEFT_PAREN = 162;

    // Key code constant: Numeric keypad '*' key (for multiplication).
    // (0x0000009b)
    int KEYCODE_NUMPAD_MULTIPLY = 155;

    // Key code constant: Numeric keypad ')' key.
    // (0x000000a3)
    int KEYCODE_NUMPAD_RIGHT_PAREN = 163;

    // Key code constant: Numeric keypad '-' key (for subtraction).
    // (0x0000009c)
    int KEYCODE_NUMPAD_SUBTRACT = 156;

    // Key code constant: Num Lock key. This is the Num Lock key; it is different from KEYCODE_NUM. This key alters the behavior of other keys on the numeric keypad.
    // (0x0000008f)
    int KEYCODE_NUM_LOCK = 143;

    // Key code constant: 'O' key.
    // (0x0000002b)
    int KEYCODE_O = 43;

    // Key code constant: 'P' key.
    // (0x0000002c)
    int KEYCODE_P = 44;

    // Key code constant: Page Down key.
    // (0x0000005d)
    int KEYCODE_PAGE_DOWN = 93;

    // Key code constant: Page Up key.
    // (0x0000005c)
    int KEYCODE_PAGE_UP = 92;

    // Key code constant: Pairing key. Initiates peripheral pairing mode. Useful for pairing remote control devices or game controllers, especially if no other input mode is available.
    // (0x000000e1)
    int KEYCODE_PAIRING = 225;

    // Key code constant: '.' key.
    // (0x00000038)
    int KEYCODE_PERIOD = 56;

    // Key code constant: Picture Symbols modifier key. Used to switch symbol sets (Emoji, Kao-moji).
    // (0x0000005e)
    int KEYCODE_PICTSYMBOLS = 94;

    // Key code constant: '+' key.
    // (0x00000051)
    int KEYCODE_PLUS = 81;

    // Key code constant: '#' key.
    // (0x00000012)
    int KEYCODE_POUND = 18;

    // Key code constant: Power key.
    // (0x0000001a)
    int KEYCODE_POWER = 26;

    // Key code constant: Blue "programmable" key. On TV remotes, acts as a contextual/programmable key.
    // (0x000000ba)
    int KEYCODE_PROG_BLUE = 186;

    // Key code constant: Green "programmable" key. On TV remotes, actsas a contextual/programmable key.
    // (0x000000b8)
    int KEYCODE_PROG_GREEN = 184;

    // Key code constant: Red "programmable" key. On TV remotes, acts as a contextual/programmable key.
    // (0x000000b7)
    int KEYCODE_PROG_RED = 183;

    // Key code constant: Yellow "programmable" key. On TV remotes, acts as a contextual/programmable key.
    // (0x000000b9)
    int KEYCODE_PROG_YELLOW = 185;

    // Key code constant: 'Q' key.
    // (0x0000002d)
    int KEYCODE_Q = 45;

    // Key code constant: 'R' key.
    // (0x0000002e)
    int KEYCODE_R = 46;

    // Key code constant: ']' key.
    // (0x00000048)
    int KEYCODE_RIGHT_BRACKET = 72;

    // Key code constant: Japanese Ro key.
    // (0x000000d9)
    int KEYCODE_RO = 217;

    // Key code constant: 'S' key.
    // (0x0000002f)
    int KEYCODE_S = 47;

    // Key code constant: Scroll Lock key.
    // (0x00000074)
    int KEYCODE_SCROLL_LOCK = 116;

    // Key code constant: Search key.
    // (0x00000054)
    int KEYCODE_SEARCH = 84;

    // Key code constant: ';' key.
    // (0x0000004a)
    int KEYCODE_SEMICOLON = 74;

    // Key code constant: Settings key. Starts the system settings activity.
    // (0x000000b0)
    int KEYCODE_SETTINGS = 176;

    // Key code constant: Left Shift modifier key.
    // (0x0000003b)
    int KEYCODE_SHIFT_LEFT = 59;

    // Key code constant: Right Shift modifier key.
    // (0x0000003c)
    int KEYCODE_SHIFT_RIGHT = 60;

    // Key code constant: '/' key.
    // (0x0000004c)
    int KEYCODE_SLASH = 76;

    // Key code constant: Sleep key. Puts the device to sleep. Behaves somewhat like KEYCODE_POWER but it has no effect if the device is already asleep.
    // (0x000000df)
    int KEYCODE_SLEEP = 223;

    // Key code constant: Soft Left key. Usually situated below the display on phones and used as a multi-function feature key for selecting a software defined function shown on the bottom left of the display.
    // (0x00000001)
    int KEYCODE_SOFT_LEFT = 1;

    // Key code constant: Soft Right key. Usually situated below the display on phones and used as a multi-function feature key for selecting a software defined function shown on the bottom right of the display.
    // (0x00000002)
    int KEYCODE_SOFT_RIGHT = 2;

    // Key code constant: Space key.
    // (0x0000003e)
    int KEYCODE_SPACE = 62;

    // Key code constant: '*' key.
    // (0x00000011)
    int KEYCODE_STAR = 17;

    // Key code constant: Set-top-box input key. On TV remotes, switches the input mode on an external Set-top-box.
    // (0x000000b4)
    int KEYCODE_STB_INPUT = 180;

    // Key code constant: Set-top-box power key. On TV remotes, toggles the power on an external Set-top-box.
    // (0x000000b3)
    int KEYCODE_STB_POWER = 179;

    // Key code constant: Switch Charset modifier key. Used to switch character sets (Kanji, Katakana).
    // (0x0000005f)
    int KEYCODE_SWITCH_CHARSET = 95;

    // Key code constant: Symbol modifier key. Used to enter alternate symbols.
    // (0x0000003f)
    int KEYCODE_SYM = 63;

    // Key code constant: System Request / Print Screen key.
    // (0x00000078)
    int KEYCODE_SYSRQ = 120;

    // Key code constant: 'T' key.
    // (0x00000030)
    int KEYCODE_T = 48;

    // Key code constant: Tab key.
    // (0x0000003d)
    int KEYCODE_TAB = 61;

    // Key code constant: TV key. On TV remotes, switches to viewing live TV.
    // (0x000000aa)
    int KEYCODE_TV = 170;

    // Key code constant: Antenna/Cable key. Toggles broadcast input source between antenna and cable.
    // (0x000000f2)
    int KEYCODE_TV_ANTENNA_CABLE = 242;

    // Key code constant: Audio description key. Toggles audio description off / on.
    // (0x000000fc)
    int KEYCODE_TV_AUDIO_DESCRIPTION = 252;

    // Key code constant: Audio description mixing volume down key. Lessen audio description volume as compared with normal audio volume.
    // (0x000000fe)
    int KEYCODE_TV_AUDIO_DESCRIPTION_MIX_DOWN = 254;

    // Key code constant: Audio description mixing volume up key. Louden audio description volume as compared with normal audio volume.
    // (0x000000fd)
    int KEYCODE_TV_AUDIO_DESCRIPTION_MIX_UP = 253;

    // Key code constant: Contents menu key. Goes to the title list. Corresponds to Contents Menu (0x0B) of CEC User Control Code
    // (0x00000100)
    int KEYCODE_TV_CONTENTS_MENU = 256;

    // Key code constant: TV data service key. Displays data services like weather, sports.
    // (0x000000e6)
    int KEYCODE_TV_DATA_SERVICE = 230;

    // Key code constant: TV input key. On TV remotes, switches the input on a television screen.
    // (0x000000b2)
    int KEYCODE_TV_INPUT = 178;

    // Key code constant: Component #1 key. Switches to component video input #1.
    // (0x000000f9)
    int KEYCODE_TV_INPUT_COMPONENT_1 = 249;

    // Key code constant: Component #2 key. Switches to component video input #2.
    // (0x000000fa)
    int KEYCODE_TV_INPUT_COMPONENT_2 = 250;

    // Key code constant: Composite #1 key. Switches to composite video input #1.
    // (0x000000f7)
    int KEYCODE_TV_INPUT_COMPOSITE_1 = 247;

    // Key code constant: Composite #2 key. Switches to composite video input #2.
    // (0x000000f8)
    int KEYCODE_TV_INPUT_COMPOSITE_2 = 248;

    // Key code constant: HDMI #1 key. Switches to HDMI input #1.
    // (0x000000f3)
    int KEYCODE_TV_INPUT_HDMI_1 = 243;

    // Key code constant: HDMI #2 key. Switches to HDMI input #2.
    // (0x000000f4)
    int KEYCODE_TV_INPUT_HDMI_2 = 244;

    // Key code constant: HDMI #3 key. Switches to HDMI input #3.
    // (0x000000f5)
    int KEYCODE_TV_INPUT_HDMI_3 = 245;

    // Key code constant: HDMI #4 key. Switches to HDMI input #4.
    // (0x000000f6)
    int KEYCODE_TV_INPUT_HDMI_4 = 246;

    // Key code constant: VGA #1 key. Switches to VGA (analog RGB) input #1.
    // (0x000000fb)
    int KEYCODE_TV_INPUT_VGA_1 = 251;

    // Key code constant: Media context menu key. Goes to the context menu of media contents. Corresponds to Media Context-sensitive Menu (0x11) of CEC User Control Code.
    // (0x00000101)
    int KEYCODE_TV_MEDIA_CONTEXT_MENU = 257;

    // Key code constant: Toggle Network key. Toggles selecting broacast services.
    // (0x000000f1)
    int KEYCODE_TV_NETWORK = 241;

    // Key code constant: Number entry key. Initiates to enter multi-digit channel nubmber when each digit key is assigned for selecting separate channel. Corresponds to Number Entry Mode (0x1D) of CEC User Control Code.
    // (0x000000ea)
    int KEYCODE_TV_NUMBER_ENTRY = 234;

    // Key code constant: TV power key. On TV remotes, toggles the power on a television screen.
    // (0x000000b1)
    int KEYCODE_TV_POWER = 177;

    // Key code constant: Radio key. Toggles TV service / Radio service.
    // (0x000000e8)
    int KEYCODE_TV_RADIO_SERVICE = 232;

    // Key code constant: Satellite key. Switches to digital satellite broadcast service.
    // (0x000000ed)
    int KEYCODE_TV_SATELLITE = 237;

    // Key code constant: BS key. Switches to BS digital satellite broadcasting service available in Japan.
    // (0x000000ee)
    int KEYCODE_TV_SATELLITE_BS = 238;

    // Key code constant: CS key. Switches to CS digital satellite broadcasting service available in Japan.
    // (0x000000ef)
    int KEYCODE_TV_SATELLITE_CS = 239;

    // Key code constant: BS/CS key. Toggles between BS and CS digital satellite services.
    // (0x000000f0)
    int KEYCODE_TV_SATELLITE_SERVICE = 240;

    // Key code constant: Teletext key. Displays Teletext service.
    // (0x000000e9)
    int KEYCODE_TV_TELETEXT = 233;

    // Key code constant: Analog Terrestrial key. Switches to analog terrestrial broadcast service.
    // (0x000000eb)
    int KEYCODE_TV_TERRESTRIAL_ANALOG = 235;

    // Key code constant: Digital Terrestrial key. Switches to digital terrestrial broadcast service.
    // (0x000000ec)
    int KEYCODE_TV_TERRESTRIAL_DIGITAL = 236;

    // Key code constant: Timer programming key. Goes to the timer recording menu. Corresponds to Timer Programming (0x54) of CEC User Control Code.
    // (0x00000102)
    int KEYCODE_TV_TIMER_PROGRAMMING = 258;

    // Key code constant: Zoom mode key. Changes Zoom mode (Normal, Full, Zoom, Wide-zoom, etc.)
    // (0x000000ff)
    int KEYCODE_TV_ZOOM_MODE = 255;

    // Key code constant: 'U' key.
    // (0x00000031)
    int KEYCODE_U = 49;

    // Key code constant: Unknown key code.
    // (0x00000000)
    int KEYCODE_UNKNOWN = 0;

    // Key code constant: 'V' key.
    // (0x00000032)
    int KEYCODE_V = 50;

    // Key code constant: Voice Assist key. Launches the global voice assist activity. Not delivered to applications.
    // (0x000000e7)
    int KEYCODE_VOICE_ASSIST = 231;

    // Key code constant: Volume Down key. Adjusts the speaker volume down.
    // (0x00000019)
    int KEYCODE_VOLUME_DOWN = 25;

    // Key code constant: Volume Mute key. Mutes the speaker, unlike KEYCODE_MUTE. This key should normally be implemented as a toggle such that the first press mutes the speaker and the second press restores the original volume.
    // (0x000000a4)
    int KEYCODE_VOLUME_MUTE = 164;

    // Key code constant: Volume Up key. Adjusts the speaker volume up.
    // (0x00000018)
    int KEYCODE_VOLUME_UP = 24;

    // Key code constant: 'W' key.
    // (0x00000033)
    int KEYCODE_W = 51;

    // Key code constant: Wakeup key. Wakes up the device. Behaves somewhat like KEYCODE_POWER but it has no effect if the device is already awake.
    // (0x000000e0)
    int KEYCODE_WAKEUP = 224;

    // Key code constant: Window key. On TV remotes, toggles picture-in-picture mode or other windowing functions.
    // (0x000000ab)
    int KEYCODE_WINDOW = 171;

    // Key code constant: 'X' key.
    // (0x00000034)
    int KEYCODE_X = 52;

    // Key code constant: 'Y' key.
    // (0x00000035)
    int KEYCODE_Y = 53;

    // Key code constant: Japanese Yen key.
    // (0x000000d8)
    int KEYCODE_YEN = 216;

    // Key code constant: 'Z' key.
    // (0x00000036)
    int KEYCODE_Z = 54;

    // Key code constant: Japanese full-width / half-width key.
    // (0x000000d3)
    int KEYCODE_ZENKAKU_HANKAKU = 211;

    // Key code constant: Zoom in key.
    // (0x000000a8)
    int KEYCODE_ZOOM_IN = 168;

    // Key code constant: Zoom out key.
    // (0x000000a9)
    int KEYCODE_ZOOM_OUT = 169;

    // This constant was deprecated in API level 3.
    // There are now more than MAX_KEYCODE keycodes. Use getMaxKeyCode() instead.
    // (0x00000054)
    int MAX_KEYCODE = 84;

    // This mask is used to check whether the left ALT meta key is pressed.
    // See Also
    // isAltPressed()
    // getMetaState()
    // KEYCODE_ALT_LEFT
    // (0x00000010)
    int META_ALT_LEFT_ON = 16;

    // This mask is a combination of META_ALT_ON, META_ALT_LEFT_ON and META_ALT_RIGHT_ON.
    // (0x00000032)
    int META_ALT_MASK = 50;

    // This mask is used to check whether one of the ALT meta keys is pressed.
    // See Also
    // isAltPressed()
    // getMetaState()
    // KEYCODE_ALT_LEFT
    // KEYCODE_ALT_RIGHT
    // (0x00000002)
    int META_ALT_ON = 2;

    // This mask is used to check whether the right the ALT meta key is pressed.
    // See Also
    // isAltPressed()
    // getMetaState()
    // KEYCODE_ALT_RIGHT
    // (0x00000020)
    int META_ALT_RIGHT_ON = 32;

    // This mask is used to check whether the CAPS LOCK meta key is on.
    // See Also
    // isCapsLockOn()
    // getMetaState()
    // KEYCODE_CAPS_LOCK
    // (0x00100000)
    int META_CAPS_LOCK_ON = 1048576;

    // This mask is used to check whether the left CTRL meta key is pressed.
    // See Also
    // isCtrlPressed()
    // getMetaState()
    // KEYCODE_CTRL_LEFT
    // (0x00002000)
    int META_CTRL_LEFT_ON = 8192;

    // This mask is a combination of META_CTRL_ON, META_CTRL_LEFT_ON and META_CTRL_RIGHT_ON.
    // (0x00007000)
    int META_CTRL_MASK = 28672;

    // This mask is used to check whether one of the CTRL meta keys is pressed.
    // See Also
    // isCtrlPressed()
    // getMetaState()
    // KEYCODE_CTRL_LEFT
    // KEYCODE_CTRL_RIGHT
    // (0x00001000)
    int META_CTRL_ON = 4096;

    // This mask is used to check whether the right CTRL meta key is pressed.
    // See Also
    // isCtrlPressed()
    // getMetaState()
    // KEYCODE_CTRL_RIGHT
    // (0x00004000)
    int META_CTRL_RIGHT_ON = 16384;

    // This mask is used to check whether the FUNCTION meta key is pressed.
    // See Also
    // isFunctionPressed()
    // getMetaState()
    // (0x00000008)
    int META_FUNCTION_ON = 8;

    // This mask is used to check whether the left META meta key is pressed.
    // See Also
    // isMetaPressed()
    // getMetaState()
    // KEYCODE_META_LEFT
    // (0x00020000)
    int META_META_LEFT_ON = 131072;

    // This mask is a combination of META_META_ON, META_META_LEFT_ON and META_META_RIGHT_ON.
    // (0x00070000)
    int META_META_MASK = 458752;

    // This mask is used to check whether one of the META meta keys is pressed.
    // See Also
    // isMetaPressed()
    // getMetaState()
    // KEYCODE_META_LEFT
    // KEYCODE_META_RIGHT
    // (0x00010000)
    int META_META_ON = 65536;

    // This mask is used to check whether the right META meta key is pressed.
    // See Also
    // isMetaPressed()
    // getMetaState()
    // KEYCODE_META_RIGHT
    // (0x00040000)
    int META_META_RIGHT_ON = 262144;

    // This mask is used to check whether the NUM LOCK meta key is on.
    // See Also
    // isNumLockOn()
    // getMetaState()
    // KEYCODE_NUM_LOCK
    // (0x00200000)
    int META_NUM_LOCK_ON = 2097152;

    // This mask is used to check whether the SCROLL LOCK meta key is on.
    // See Also
    // isScrollLockOn()
    // getMetaState()
    // KEYCODE_SCROLL_LOCK
    // (0x00400000)
    int META_SCROLL_LOCK_ON = 4194304;

    // This mask is used to check whether the left SHIFT meta key is pressed.
    // See Also
    // isShiftPressed()
    // getMetaState()
    // KEYCODE_SHIFT_LEFT
    // (0x00000040)
    int META_SHIFT_LEFT_ON = 64;

    // This mask is a combination of META_SHIFT_ON, META_SHIFT_LEFT_ON and META_SHIFT_RIGHT_ON.
    // (0x000000c1)
    int META_SHIFT_MASK = 193;

    // This mask is used to check whether one of the SHIFT meta keys is pressed.
    // See Also
    // isShiftPressed()
    // getMetaState()
    // KEYCODE_SHIFT_LEFT
    // KEYCODE_SHIFT_RIGHT
    // (0x00000001)
    int META_SHIFT_ON = 1;

    // This mask is used to check whether the right SHIFT meta key is pressed.
    // See Also
    // isShiftPressed()
    // getMetaState()
    // KEYCODE_SHIFT_RIGHT
    // (0x00000080)
    int META_SHIFT_RIGHT_ON = 128;

    // This mask is used to check whether the SYM meta key is pressed.
    // See Also
    // isSymPressed()
    // getMetaState()
    // (0x00000004)
    int META_SYM_ON = 4;

}
