/*
 * Decompiled with CFR 0.152.
 */
package io.apiman.manager.api.notifications.email;

import com.google.common.collect.Maps;
import io.apiman.common.logging.ApimanLoggerFactory;
import io.apiman.common.logging.IApimanLogger;
import io.apiman.common.util.JsonUtil;
import io.apiman.manager.api.beans.notifications.EmailNotificationTemplate;
import io.apiman.manager.api.beans.notifications.NotificationCategory;
import io.apiman.manager.api.core.config.ApiManagerConfig;
import io.apiman.manager.api.notifications.email.EmailSender;
import io.apiman.manager.api.notifications.email.EmailTemplateFileEntry;
import io.apiman.manager.api.notifications.email.IEmailSender;
import io.apiman.manager.api.notifications.email.MockEmailSender;
import io.apiman.manager.api.notifications.email.QuteTemplateEngine;
import io.apiman.manager.api.notifications.email.SimpleEmail;
import io.apiman.manager.api.notifications.email.SmtpEmailConfiguration;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.context.Initialized;
import javax.enterprise.event.Observes;
import javax.inject.Inject;

@ApplicationScoped
@ParametersAreNonnullByDefault
public class SimpleMailNotificationService {
    private static final IApimanLogger LOGGER = ApimanLoggerFactory.getLogger(SimpleMailNotificationService.class);
    private static final Map<String, String> DEFAULT_HEADERS = Map.of("X-Notification-Producer", "Apiman");
    private IEmailSender emailSender = new NullEmailSender();
    private ApiManagerConfig config;
    private QuteTemplateEngine templateEngine;
    private final Map<String, Map<Locale, EmailNotificationTemplate>> reasonMap = new HashMap<String, Map<Locale, EmailNotificationTemplate>>();
    private final Map<NotificationCategory, Map<Locale, List<EmailNotificationTemplate>>> categoryToTemplateMap = new HashMap<NotificationCategory, Map<Locale, List<EmailNotificationTemplate>>>();
    private Set<Locale> supportedLocales;

    @Inject
    public SimpleMailNotificationService(ApiManagerConfig config, QuteTemplateEngine templateEngine) {
        this.templateEngine = templateEngine;
        this.config = config;
        SmtpEmailConfiguration smtpConfig = new SmtpEmailConfiguration(config.getEmailNotificationProperties());
        if (smtpConfig.isEnabled()) {
            this.emailSender = smtpConfig.isMock() ? new MockEmailSender() : new EmailSender(smtpConfig);
        }
        this.readEmailNotificationTemplatesFromFile();
    }

    public SimpleMailNotificationService() {
    }

    public void init(@Observes @Initialized(value=ApplicationScoped.class) Object init) {
    }

    public void send(SimpleEmail email) {
        LOGGER.debug("Sending email: {0}", new Object[]{email});
        EmailNotificationTemplate template = email.getTemplate();
        String html = this.templateEngine.applyTemplate(template.getHtmlBody(), email.getTemplateVariables());
        boolean htmlBlank = html.isBlank();
        String plain = this.templateEngine.applyTemplate(template.getPlainBody(), email.getTemplateVariables());
        boolean plainBlank = html.isBlank();
        String subject = this.templateEngine.applyTemplate(template.getSubject(), email.getTemplateVariables());
        if (subject.isBlank()) {
            throw new IllegalArgumentException("Non-blank subject is required for notification emails " + email);
        }
        if (htmlBlank && plainBlank) {
            throw new IllegalArgumentException("Both HTML and plain templates are blank in notification mail " + email);
        }
        if (htmlBlank) {
            this.sendPlaintext(email.getToEmail(), email.getToName(), subject, plain, email.getHeaders());
        } else if (plainBlank) {
            LOGGER.warn("Sending an HTML mail without a plaintext version is not recommended: {0}", new Object[]{email});
            this.sendHtml(email.getToEmail(), email.getToName(), subject, html, "", email.getHeaders());
        } else {
            this.sendHtml(email.getToEmail(), email.getToName(), subject, html, plain, email.getHeaders());
        }
    }

    public void sendPlaintext(String toEmail, String toName, String subject, String body) {
        this.emailSender.sendPlaintext(toEmail, toName, subject, body, DEFAULT_HEADERS);
    }

    public void sendPlaintext(String toEmail, String toName, String subject, String body, Map<String, String> headers) {
        HashMap copy = Maps.newHashMap(headers);
        copy.putAll(DEFAULT_HEADERS);
        this.emailSender.sendPlaintext(toEmail, toName, subject, body, copy);
    }

    public void sendHtml(String toEmail, String toName, String subject, String htmlBody, String plainBody) {
        this.emailSender.sendHtml(toEmail, toName, subject, htmlBody, plainBody, DEFAULT_HEADERS);
    }

    public void sendHtml(String toEmail, String toName, String subject, String htmlBody, String plainBody, Map<String, String> headers) {
        HashMap copy = Maps.newHashMap(headers);
        copy.putAll(DEFAULT_HEADERS);
        this.emailSender.sendHtml(toEmail, toName, subject, htmlBody, plainBody, headers);
    }

    public Optional<EmailNotificationTemplate> findTemplateFor(String reasonKey, String localeTag) {
        Locale bestMatchingLocale = Locale.lookup(Locale.LanguageRange.parse(localeTag), this.supportedLocales);
        return this.findTemplateFor(reasonKey, bestMatchingLocale);
    }

    public Optional<EmailNotificationTemplate> findTemplateFor(String reasonKey, Locale locale) {
        Objects.requireNonNull(reasonKey, "Notification reason must not be null");
        Objects.requireNonNull(locale, "Locale must not be null");
        Map<Locale, EmailNotificationTemplate> localeMap = this.reasonMap.get(reasonKey);
        if (localeMap == null || localeMap.isEmpty()) {
            LOGGER.debug("No email template found for reason {0}, including shorter paths", new Object[]{reasonKey});
            return Optional.empty();
        }
        List<Locale> bestMatchingLocale = Arrays.asList(Locale.lookup(Locale.LanguageRange.parse(locale.toLanguageTag()), this.supportedLocales), Locale.getDefault(), new Locale("en"));
        return bestMatchingLocale.stream().filter(Objects::nonNull).map(localeMap::get).filter(Objects::nonNull).findFirst();
    }

    private void readEmailNotificationTemplatesFromFile() {
        Path file = this.config.getConfigDirectory().resolve("notifications/email/notification-template-index.json");
        if (Files.notExists(file, new LinkOption[0])) {
            LOGGER.warn("No email notification templates found at {0}", new Object[]{file});
            return;
        }
        try {
            Map localeTplMap = JsonUtil.toPojo((String)Files.readString(file), String.class, ReasonMap.class, HashMap.class);
            LOGGER.debug("Email notification templates read from {0}", new Object[]{file.toAbsolutePath()});
            this.supportedLocales = localeTplMap.keySet().stream().map(Locale::forLanguageTag).collect(Collectors.toSet());
            localeTplMap.forEach((k, v) -> this.processReasonMap(file, (String)k, (ReasonMap)v));
        }
        catch (IOException ioe) {
            throw new UncheckedIOException(ioe);
        }
    }

    private void processReasonMap(Path root, String localeTag, ReasonMap reasonMapFromFile) {
        String reason = "";
        try {
            for (Map.Entry entry : reasonMapFromFile.entrySet()) {
                reason = (String)entry.getKey();
                EmailTemplateFileEntry fileEntry = (EmailTemplateFileEntry)entry.getValue();
                EmailNotificationTemplate tpl = new EmailNotificationTemplate();
                tpl.setNotificationReason(reason);
                tpl.setLocale(localeTag);
                tpl.setSubject(fileEntry.subject());
                tpl.setHtmlBody(Files.readString(root.getParent().resolve(fileEntry.html())));
                tpl.setPlainBody(Files.readString(root.getParent().resolve(fileEntry.plain())));
                tpl.setCategory(fileEntry.category());
                Locale locale = Locale.forLanguageTag(localeTag);
                this.reasonMap.computeIfAbsent(reason, val -> new HashMap(5)).put(locale, tpl);
                Map catLocales = this.categoryToTemplateMap.computeIfAbsent(tpl.getCategory(), val -> new HashMap());
                catLocales.computeIfAbsent(locale, val -> new ArrayList()).add(tpl);
                LOGGER.trace("Adding template ({0}): reason {1} -> {2}", new Object[]{locale, tpl.getNotificationReason(), tpl});
                LOGGER.trace("Adding template ({0}): category {1} -> {2}", new Object[]{locale, tpl.getCategory(), tpl});
            }
        }
        catch (IOException ioe) {
            LOGGER.error((Throwable)ioe, "An IO exception occurred attempting to process an email template. Reason code: {0}. Locale: {1}.", new Object[]{reason, localeTag});
            throw new UncheckedIOException(ioe);
        }
    }

    private static final class NullEmailSender
    implements IEmailSender {
        private NullEmailSender() {
        }

        @Override
        public void sendPlaintext(String toEmail, String toName, String subject, String body, Map<String, String> headers) throws IEmailSender.EmailException {
        }

        @Override
        public void sendHtml(String toEmail, String toName, String subject, String htmlBody, String plainBody, Map<String, String> headers) throws IEmailSender.EmailException {
        }
    }

    private static final class ReasonMap
    extends HashMap<String, EmailTemplateFileEntry> {
        private ReasonMap() {
        }
    }
}

