/*
 * Decompiled with CFR 0.152.
 */
package io.ably.lib.transport;

import io.ably.lib.http.HttpUtils;
import io.ably.lib.network.EngineType;
import io.ably.lib.network.NotConnectedException;
import io.ably.lib.network.WebSocketClient;
import io.ably.lib.network.WebSocketEngine;
import io.ably.lib.network.WebSocketEngineConfig;
import io.ably.lib.network.WebSocketEngineFactory;
import io.ably.lib.network.WebSocketListener;
import io.ably.lib.transport.ConnectionManager;
import io.ably.lib.transport.ITransport;
import io.ably.lib.transport.SafeSSLSocketFactory;
import io.ably.lib.types.AblyException;
import io.ably.lib.types.ErrorInfo;
import io.ably.lib.types.Param;
import io.ably.lib.types.ProtocolMessage;
import io.ably.lib.types.ProtocolSerializer;
import io.ably.lib.util.ClientOptionsUtils;
import io.ably.lib.util.Log;
import java.nio.ByteBuffer;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.Timer;
import java.util.TimerTask;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;

public class WebSocketTransport
implements ITransport {
    private static final String TAG = WebSocketTransport.class.getName();
    private static final int NEVER_CONNECTED = -1;
    private static final int BUGGYCLOSE = -2;
    private static final int CLOSE_NORMAL = 1000;
    private static final int GOING_AWAY = 1001;
    private static final int CLOSE_PROTOCOL_ERROR = 1002;
    private static final int REFUSE = 1003;
    private static final int ABNORMAL_CLOSE = 1006;
    private static final int NO_UTF8 = 1007;
    private static final int POLICY_VALIDATION = 1008;
    private static final int TOOBIG = 1009;
    private static final int EXTENSION = 1010;
    private static final int UNEXPECTED_CONDITION = 1011;
    private static final int TLS_ERROR = 1015;
    private final ITransport.TransportParams params;
    private final ConnectionManager connectionManager;
    private final boolean channelBinaryMode;
    private String wsUri;
    private ITransport.ConnectListener connectListener;
    private WebSocketClient webSocketClient;
    private final WebSocketEngine webSocketEngine;
    private boolean activityCheckTurnedOff = false;

    protected WebSocketTransport(ITransport.TransportParams params, ConnectionManager connectionManager) {
        this.params = params;
        this.connectionManager = connectionManager;
        this.channelBinaryMode = params.options.useBinaryProtocol;
        this.webSocketEngine = WebSocketTransport.createWebSocketEngine(params);
        params.heartbeats = !this.webSocketEngine.isPingListenerSupported();
    }

    private static WebSocketEngine createWebSocketEngine(ITransport.TransportParams params) {
        WebSocketEngineFactory engineFactory = WebSocketEngineFactory.getFirstAvailable();
        Log.v(TAG, String.format("Using %s WebSocket Engine", engineFactory.getEngineType().name()));
        WebSocketEngineConfig.WebSocketEngineConfigBuilder configBuilder = WebSocketEngineConfig.builder();
        configBuilder.tls(params.options.tls).host(params.host).proxy(ClientOptionsUtils.convertToProxyConfig(params.getClientOptions()));
        if (params.options.tls && engineFactory.getEngineType() != EngineType.OKHTTP) {
            try {
                SSLContext sslContext = SSLContext.getInstance("TLS");
                sslContext.init(null, null, null);
                SafeSSLSocketFactory factory = new SafeSSLSocketFactory(sslContext.getSocketFactory());
                configBuilder.sslSocketFactory((SSLSocketFactory)factory);
            }
            catch (KeyManagementException | NoSuchAlgorithmException e) {
                throw new IllegalStateException("Can't get safe tls algorithms", e);
            }
        }
        return engineFactory.create(configBuilder.build());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void connect(ITransport.ConnectListener connectListener) {
        this.connectListener = connectListener;
        try {
            boolean isTls = this.params.options.tls;
            String wsScheme = isTls ? "wss://" : "ws://";
            this.wsUri = wsScheme + this.params.host + ':' + this.params.port + "/";
            Param[] authParams = this.connectionManager.ably.auth.getAuthParams();
            Param[] connectParams = this.params.getConnectParams(authParams);
            if (connectParams.length > 0) {
                this.wsUri = HttpUtils.encodeParams(this.wsUri, connectParams);
            }
            Log.d(TAG, "connect(); wsUri = " + this.wsUri);
            WebSocketTransport webSocketTransport = this;
            synchronized (webSocketTransport) {
                this.webSocketClient = this.webSocketEngine.create(this.wsUri, (WebSocketListener)new WebSocketHandler(this::receive));
            }
            this.webSocketClient.connect();
        }
        catch (AblyException e) {
            Log.e(TAG, "Unexpected exception attempting connection; wsUri = " + this.wsUri, e);
            connectListener.onTransportUnavailable(this, e.errorInfo);
        }
        catch (Throwable t) {
            Log.e(TAG, "Unexpected exception attempting connection; wsUri = " + this.wsUri, t);
            connectListener.onTransportUnavailable(this, AblyException.fromThrowable((Throwable)t).errorInfo);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        Log.d(TAG, "close()");
        WebSocketTransport webSocketTransport = this;
        synchronized (webSocketTransport) {
            if (this.webSocketClient != null) {
                this.webSocketClient.close();
                this.webSocketClient = null;
            }
        }
    }

    @Override
    public void receive(ProtocolMessage msg) throws AblyException {
        this.connectionManager.onMessage(this, msg);
    }

    @Override
    public void send(ProtocolMessage msg) throws AblyException {
        Log.d(TAG, "send(); action = " + (Object)((Object)msg.action));
        try {
            if (this.channelBinaryMode) {
                byte[] encodedMsg = ProtocolSerializer.writeMsgpack(msg);
                if (Log.level <= 2) {
                    ProtocolMessage decodedMsg = ProtocolSerializer.readMsgpack(encodedMsg);
                    Log.v(TAG, "send(): " + (Object)((Object)decodedMsg.action) + ": " + new String(ProtocolSerializer.writeJSON(decodedMsg)));
                }
                this.webSocketClient.send(encodedMsg);
            } else {
                if (Log.level <= 2) {
                    Log.v(TAG, "send(): " + new String(ProtocolSerializer.writeJSON(msg)));
                }
                this.webSocketClient.send(ProtocolSerializer.writeJSON(msg));
            }
        }
        catch (NotConnectedException e) {
            if (this.connectListener != null) {
                this.connectListener.onTransportUnavailable(this, AblyException.fromThrowable((Throwable)e).errorInfo);
            }
            throw AblyException.fromThrowable(e);
        }
        catch (Exception e) {
            throw AblyException.fromThrowable(e);
        }
    }

    @Override
    public String getHost() {
        return this.params.host;
    }

    protected void preProcessReceivedMessage(ProtocolMessage message) {
    }

    protected void turnOffActivityCheckIfPingListenerIsNotSupported() {
        if (!this.webSocketEngine.isPingListenerSupported()) {
            this.activityCheckTurnedOff = true;
        }
    }

    public String toString() {
        return WebSocketTransport.class.getName() + " {" + this.getURL() + "}";
    }

    @Override
    public String getURL() {
        return this.wsUri;
    }

    class WebSocketHandler
    implements WebSocketListener {
        private final WebSocketReceiver receiver;
        private Timer timer = new Timer();
        private TimerTask activityTimerTask = null;
        private long lastActivityTime;

        WebSocketHandler(WebSocketReceiver receiver) {
            this.receiver = receiver;
        }

        public void onOpen() {
            Log.d(TAG, "onOpen()");
            WebSocketTransport.this.connectListener.onTransportAvailable(WebSocketTransport.this);
            this.flagActivity();
        }

        public void onMessage(ByteBuffer blob) {
            try {
                ProtocolMessage msg = ProtocolSerializer.readMsgpack(blob.array());
                Log.d(TAG, "onMessage(): msg (binary) = " + msg);
                WebSocketTransport.this.preProcessReceivedMessage(msg);
                this.receiver.onMessage(msg);
            }
            catch (AblyException e) {
                String msg = "Unexpected exception processing received binary message";
                Log.e(TAG, msg, e);
            }
            this.flagActivity();
        }

        public void onMessage(String string) {
            try {
                ProtocolMessage msg = ProtocolSerializer.fromJSON(string);
                Log.d(TAG, "onMessage(): msg (text) = " + msg);
                WebSocketTransport.this.preProcessReceivedMessage(msg);
                this.receiver.onMessage(msg);
            }
            catch (AblyException e) {
                String msg = "Unexpected exception processing received text message";
                Log.e(TAG, msg, e);
            }
            this.flagActivity();
        }

        public void onWebsocketPing() {
            Log.d(TAG, "onWebsocketPing()");
            this.flagActivity();
        }

        public void onClose(int wsCode, String wsReason) {
            ErrorInfo reason;
            Log.d(TAG, "onClose(): wsCode = " + wsCode + "; wsReason = " + wsReason + "; remote = " + false);
            switch (wsCode) {
                case -2: 
                case -1: 
                case 1000: 
                case 1001: 
                case 1006: {
                    reason = ConnectionManager.REASON_DISCONNECTED;
                    break;
                }
                case 1003: 
                case 1008: {
                    reason = ConnectionManager.REASON_REFUSED;
                    break;
                }
                case 1009: {
                    reason = ConnectionManager.REASON_TOO_BIG;
                    break;
                }
                default: {
                    reason = ConnectionManager.REASON_FAILED;
                }
            }
            WebSocketTransport.this.connectListener.onTransportUnavailable(WebSocketTransport.this, reason);
            this.dispose();
        }

        public void onError(Throwable throwable) {
            Log.e(TAG, "Connection error ", throwable);
            WebSocketTransport.this.connectListener.onTransportUnavailable(WebSocketTransport.this, new ErrorInfo(throwable.getMessage(), 503, 80000));
        }

        public void onOldJavaVersionDetected(Throwable throwable) {
            Log.w(TAG, "Error when trying to set SSL parameters, most likely due to an old Java API version", throwable);
        }

        private synchronized void dispose() {
            try {
                this.timer.cancel();
                this.timer = null;
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
        }

        private synchronized void flagActivity() {
            this.lastActivityTime = System.currentTimeMillis();
            WebSocketTransport.this.connectionManager.setLastActivity(this.lastActivityTime);
            if (this.activityTimerTask == null && ((WebSocketTransport)WebSocketTransport.this).connectionManager.maxIdleInterval != 0L && !WebSocketTransport.this.activityCheckTurnedOff) {
                this.checkActivity();
            }
        }

        private synchronized void checkActivity() {
            long timeout = this.getActivityTimeout();
            if (timeout == 0L) {
                Log.v(TAG, "checkActivity: infinite timeout");
                return;
            }
            if (this.activityTimerTask != null) {
                return;
            }
            this.startActivityTimer(timeout + 100L);
        }

        private synchronized void startActivityTimer(long timeout) {
            if (this.activityTimerTask == null) {
                this.activityTimerTask = new TimerTask(){

                    @Override
                    public void run() {
                        try {
                            WebSocketHandler.this.onActivityTimerExpiry();
                        }
                        catch (Throwable t) {
                            Log.e(TAG, "Unexpected exception in activity timer handler", t);
                        }
                    }
                };
                this.schedule(this.activityTimerTask, timeout);
            }
        }

        private synchronized void schedule(TimerTask task, long delay) {
            if (this.timer != null) {
                try {
                    this.timer.schedule(task, delay);
                }
                catch (IllegalStateException ise) {
                    Log.e(TAG, "Unexpected exception scheduling activity timer", ise);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void onActivityTimerExpiry() {
            long timeSinceLastActivity = System.currentTimeMillis() - this.lastActivityTime;
            long timeRemaining = this.getActivityTimeout() - timeSinceLastActivity;
            if (timeRemaining <= 0L) {
                Log.e(TAG, "No activity for " + this.getActivityTimeout() + "ms, closing connection");
                WebSocketTransport.this.webSocketClient.cancel(1006, "timed out");
                return;
            }
            WebSocketHandler webSocketHandler = this;
            synchronized (webSocketHandler) {
                this.activityTimerTask = null;
                Log.v(TAG, "onActivityTimerExpiry: ok");
                this.startActivityTimer(timeRemaining + 100L);
            }
        }

        private long getActivityTimeout() {
            return ((WebSocketTransport)WebSocketTransport.this).connectionManager.maxIdleInterval + ((WebSocketTransport)WebSocketTransport.this).connectionManager.ably.options.realtimeRequestTimeout;
        }
    }

    static interface WebSocketReceiver {
        public void onMessage(ProtocolMessage var1) throws AblyException;
    }

    public static class Factory
    implements ITransport.Factory {
        @Override
        public WebSocketTransport getTransport(ITransport.TransportParams params, ConnectionManager connectionManager) {
            return new WebSocketTransport(params, connectionManager);
        }
    }
}

