package im.zego.zegoexpress;

import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.entity.*;
import java.nio.*;
import java.util.*;
import org.json.*;
import im.zego.zegoexpress.*;
import im.zego.zegoexpress.internal.*;
import android.app.Application;
import android.graphics.*;


public abstract class ZegoExpressEngine {
    /**
     * Creates a singleton instance of ZegoExpressEngine.
     *
     * The engine needs to be created and initialized before calling other functions. The SDK only supports the creation of one instance of ZegoExpressEngine. Multiple calls to this function return the same object.
     *
     * @param appID Application ID issued by ZEGO for developers, please apply from the ZEGO Admin Console https://console-express.zego.im The value ranges from 0 to 4294967295.
     * @param appSign Application signature for each AppID, please apply from the ZEGO Admin Console. Application signature is a 64 character string. Each character has a range of '0' ~ '9', 'a' ~ 'z'.
     * @param isTestEnv Choose to use a test environment or a formal commercial environment, the formal environment needs to submit work order configuration in the ZEGO management console. The test environment is for test development, with a limit of 10 rooms and 50 users. Official environment App is officially launched. ZEGO will provide corresponding server resources according to the configuration records submitted by the developer in the management console. The test environment and the official environment are two sets of environments and cannot be interconnected.
     * @param scenario The application scenario. Developers can choose one of ZegoScenario based on the scenario of the app they are developing, and the engine will preset a more general setting for specific scenarios based on the set scenario. After setting specific scenarios, developers can still call specific functions to set specific parameters if they have customized parameter settings.
     * @param application Android Application Context.
     * @param eventHandler Event notification callback. [null] means not receiving any callback notifications.It can also be managed later via [setEventHandler]
     * @return Engine singleton instance.
     */
    public static ZegoExpressEngine createEngine(long appID, String appSign, boolean isTestEnv, ZegoScenario scenario, Application application, IZegoEventHandler eventHandler){
        return ZegoExpressEngineInternalImpl.createEngine(appID, appSign, isTestEnv, scenario, application, eventHandler);
    }

    /**
     * Destroys the singleton instance of ZegoExpressEngine asynchronously.
     *
     * Used to release resources used by ZegoExpressEngine.
     *
     * @param callback Notification callback for destroy engine completion. Developers can listen to this callback to ensure that device hardware resources are released. This callback is only used to notify the completion of the release of internal resources of the engine. Developers cannot release resources related to the engine within this callback. If the developer only uses SDK to implement audio and video functions, this parameter can be passed [null].
     */
    public static void destroyEngine(IZegoDestroyCompletionCallback callback){
        ZegoExpressEngineInternalImpl.destroyEngine(callback);
    }

    /**
     * Returns the singleton instance of ZegoExpressEngine.
     *
     * If the engine has not been created or has been destroyed, returns [null].
     *
     * @return Engine singleton instance
     */
    public static ZegoExpressEngine getEngine(){
        return ZegoExpressEngineInternalImpl.getEngine();
    }

    /**
     * Set advanced engine configuration.
     *
     * Developers need to call this function to set advanced function configuration when they need advanced functions of the engine.
     *
     * @param config Advanced engine configuration
     */
    public static void setEngineConfig(ZegoEngineConfig config){
        ZegoExpressEngineInternalImpl.setEngineConfig(config);
    }

    /**
     * Set log configuration.
     *
     * When developers need to customize the log file size and path, they need to call this function to complete the configuration.It must be set before calling [createEngine] to take effect. If it is set after [createEngine], it will take effect at the next [createEngine] after [destroyEngine].Once this interface is called, the old way of setting log size and path through [setEngineConfig] will be invalid before destroyEngine, that is, during the entire life cycle of the engine.It is recommended that once you use this method, you always only use this method to complete the requirements for setting the log path and size.
     *
     * @param config log configuration
     */
    public static void setLogConfig(ZegoLogConfig config){
        ZegoExpressEngineInternalImpl.setLogConfig(config);
    }

    /**
     * Gets the SDK's version number.
     *
     * When the SDK is running, the developer finds that it does not match the expected situation and submits the problem and related logs to the ZEGO technical staff for locating. The ZEGO technical staff may need the information of the engine version to assist in locating the problem.
     * Developers can also collect this information as the version information of the engine used by the app, so that the SDK corresponding to each version of the app on the line.
     *
     * @return SDK version
     */
    public static String getVersion(){
        return ZegoExpressEngineInternalImpl.getVersion();
    }

    /**
     * Set the setting of the execution result of the calling method. After setting, you can get the detailed information of the result of each execution of the ZEGO SDK method
     *
     * @param callback Method execution result callback
     */
    public static void setApiCalledCallback(IZegoApiCalledEventHandler callback){
        ZegoExpressEngineInternalImpl.setApiCalledCallback(callback);
    }

    /**
     * Sets up the event notification callbacks that need to be handled. If the eventHandler is set to [null], all the callbacks set previously will be cleared.
     *
     * Invoke this function will **OVERWRITE** the handler set in [createEngine] or the handler set by the last call to this method.
     *
     * @param eventHandler Event notification callback. Developers should override callback related methods to focus on specific notifications based on their own business scenarios. The main callback methods of SDK are in [IZegoEventHandler].
     */
    public abstract void setEventHandler(IZegoEventHandler eventHandler);

    /**
     * Uploads logs to the ZEGO server.
     *
     * By default, SDK creates and prints log files in the app's default directory. Each log file defaults to a maximum of 5MB. Three log files are written over and over in a circular fashion. When calling this function, SDK will auto package and upload the log files to the ZEGO server.
     * Developers can provide a business “feedback” channel in the app. When users feedback problems, they can call this function to upload the local log information of SDK to help locate user problems.
     * The function is valid for the entire life cycle of the SDK.
     */
    public abstract void uploadLog();

    /**
     * Uploads logs to the ZEGO server.
     *
     * By default, SDK creates and prints log files in the app's default directory. Each log file defaults to a maximum of 5MB. Three log files are written over and over in a circular fashion. When calling this function, SDK will auto package and upload the log files to the ZEGO server.
     * Developers can provide a business “feedback” channel in the app. When users feedback problems, they can call this function to upload the local log information of SDK to help locate user problems.
     * The function is valid for the entire life cycle of the SDK.
     *
     * @param callback Log upload result callback
     */
    public abstract void uploadLog(IZegoUploadLogResultCallback callback);

    /**
     * Turns on/off verbose debugging and sets up the log language.
     *
     * The debug switch is set to on and the language is English by default.
     *
     * @deprecated This method has been deprecated after version 2.3.0, please use the [setEngineConfig] function to set the advanced configuration property advancedConfig to achieve the original function.
     * @param enable Detailed debugging information switch
     * @param language Debugging information language
     */
    @Deprecated
    public abstract void setDebugVerbose(boolean enable, ZegoLanguage language);

    /**
     * Logs in to a room. You must log in to a room before publishing or playing streams.
     *
     * Different users who log in to the same room can get room related notifications in the same room (eg [onRoomUserUpdate], [onRoomStreamUpdate], etc.), and users in one room cannot receive room signaling notifications in another room.
     * Messages sent in one room (e.g. [setStreamExtraInfo], [sendBroadcastMessage], [sendBarrageMessage], [sendCustomCommand], etc.) cannot be received callback ((eg [onRoomStreamExtraInfoUpdate], [onIMRecvBroadcastMessage], [onIMRecvBarrageMessage], [onIMRecvCustomCommand], etc) in other rooms. Currently, SDK does not provide the ability to send messages across rooms. Developers can integrate the SDK of third-party IM to achieve.
     * SDK supports startPlayingStream audio and video streams from different rooms under the same appID, that is, startPlayingStream audio and video streams across rooms. Since ZegoExpressEngine's room related callback notifications are based on the same room, when developers want to startPlayingStream streams across rooms, developers need to maintain related messages and signaling notifications by themselves.
     * If the network is temporarily interrupted due to network quality reasons, the SDK will automatically reconnect internally. You can get the current connection status of the local room by listening to the [onRoomStateUpdate] callback method, and other users in the same room will receive [onRoomUserUpdate] callback notification.
     * It is strongly recommended that userID corresponds to the user ID of the business APP, that is, a userID and a real user are fixed and unique, and should not be passed to the SDK in a random userID. Because the unique and fixed userID allows ZEGO technicians to quickly locate online problems.
     * After the first login failure due to network reasons or the room is disconnected, the default time of SDK reconnection is 20min.
     * For restrictions on the use of this function, please refer to https://doc-en.zego.im/en/6419.html or contact ZEGO technical support.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param user User object instance, configure userID, userName. Note that the userID needs to be globally unique with the same appID, otherwise the user who logs in later will kick out the user who logged in first.
     */
    public abstract void loginRoom(String roomID, ZegoUser user);

    /**
     * Logs in to a room with advanced room configurations. You must log in to a room before publishing or playing streams.
     *
     * To prevent the app from being impersonated by a malicious user, you can add authentication before logging in to the room, that is, the [token] parameter in the ZegoRoomConfig object passed in by the [config] parameter.
     * Different users who log in to the same room can get room related notifications in the same room (eg [onRoomUserUpdate], [onRoomStreamUpdate], etc.), and users in one room cannot receive room signaling notifications in another room.
     * Messages sent in one room (e.g. [setStreamExtraInfo], [sendBroadcastMessage], [sendBarrageMessage], [sendCustomCommand], etc.) cannot be received callback ((eg [onRoomStreamExtraInfoUpdate], [onIMRecvBroadcastMessage], [onIMRecvBarrageMessage], [onIMRecvCustomCommand], etc) in other rooms. Currently, SDK does not provide the ability to send messages across rooms. Developers can integrate the SDK of third-party IM to achieve.
     * SDK supports startPlayingStream audio and video streams from different rooms under the same appID, that is, startPlayingStream audio and video streams across rooms. Since ZegoExpressEngine's room related callback notifications are based on the same room, when developers want to startPlayingStream streams across rooms, developers need to maintain related messages and signaling notifications by themselves.
     * If the network is temporarily interrupted due to network quality reasons, the SDK will automatically reconnect internally. You can get the current connection status of the local room by listening to the [onRoomStateUpdate] callback method, and other users in the same room will receive [onRoomUserUpdate] callback notification.
     * It is strongly recommended that userID corresponds to the user ID of the business APP, that is, a userID and a real user are fixed and unique, and should not be passed to the SDK in a random userID. Because the unique and fixed userID allows ZEGO technicians to quickly locate online problems.
     * After the first login failure due to network reasons or the room is disconnected, the default time of SDK reconnection is 20min.
     * For restrictions on the use of this function, please refer to https://doc-en.zego.im/en/6419.html or contact ZEGO technical support.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param user User object instance, configure userID, userName. Note that the userID needs to be globally unique with the same appID, otherwise the user who logs in later will kick out the user who logged in first.
     * @param config Advanced room configuration
     */
    public abstract void loginRoom(String roomID, ZegoUser user, ZegoRoomConfig config);

    /**
     * Logs in multi room.
     *
     * You must log in the main room with [loginRoom] before invoke this function to logging in to multi room.
     * Currently supports logging into 1 main room and 1 multi room at the same time.
     * When logging out, you must log out of the multi room before logging out of the main room.
     * User can only publish the stream in the main room, but can play the stream in the main room and multi room at the same time, and can receive the signaling and callback in each room.
     * The advantage of multi room is that you can login another room without leaving the current room, receive signaling and callback from another room, and play streams from another room.
     * To prevent the app from being impersonated by a malicious user, you can add authentication before logging in to the room, that is, the [token] parameter in the ZegoRoomConfig object passed in by the [config] parameter.
     * Different users who log in to the same room can get room related notifications in the same room (eg [onRoomUserUpdate], [onRoomStreamUpdate], etc.), and users in one room cannot receive room signaling notifications in another room.
     * Messages sent in one room (e.g. [setStreamExtraInfo], [sendBroadcastMessage], [sendBarrageMessage], [sendCustomCommand], etc.) cannot be received callback ((eg [onRoomStreamExtraInfoUpdate], [onIMRecvBroadcastMessage], [onIMRecvBarrageMessage], [onIMRecvCustomCommand], etc) in other rooms. Currently, SDK does not provide the ability to send messages across rooms. Developers can integrate the SDK of third-party IM to achieve.
     * SDK supports startPlayingStream audio and video streams from different rooms under the same appID, that is, startPlayingStream audio and video streams across rooms. Since ZegoExpressEngine's room related callback notifications are based on the same room, when developers want to startPlayingStream streams across rooms, developers need to maintain related messages and signaling notifications by themselves.
     * If the network is temporarily interrupted due to network quality reasons, the SDK will automatically reconnect internally. You can get the current connection status of the local room by listening to the [onRoomStateUpdate] callback method, and other users in the same room will receive [onRoomUserUpdate] callback notification.
     * It is strongly recommended that userID corresponds to the user ID of the business APP, that is, a userID and a real user are fixed and unique, and should not be passed to the SDK in a random userID. Because the unique and fixed userID allows ZEGO technicians to quickly locate online problems.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param config Advanced room configuration
     */
    public abstract void loginMultiRoom(String roomID, ZegoRoomConfig config);

    /**
     * Logs out of a room.
     *
     * Exiting the room will stop all publishing and playing streams for user, and inner audio and video engine will stop, and then SDK will auto stop local preview UI. If you want to keep the preview ability when switching rooms, please use the [switchRoom] method.
     * After calling this function, you will receive [onRoomStateUpdate] callback notification successfully exits the room, while other users in the same room will receive the [onRoomUserUpdate] callback notification(On the premise of enabling isUserStatusNotify configuration).'
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     */
    public abstract void logoutRoom(String roomID);

    /**
     * Switch the room.
     *
     * After successfully login room, if you need to quickly switch to the next room, you can call this function.
     * Calling this function is faster and easier to use than calling [logoutRoom] and then [loginRoom].
     * When this function is called, all streams currently publishing or playing will stop (but the local preview will not stop).
     *
     * @param fromRoomID Current roomID
     * @param toRoomID The next roomID
     */
    public abstract void switchRoom(String fromRoomID, String toRoomID);

    /**
     * Switch the room with advanced room configurations.
     *
     * After successfully login room, if you need to quickly switch to the next room, you can call this function.
     * Calling this function is faster and easier to use than calling [logoutRoom] and then [loginRoom].
     * When this function is called, all streams currently publishing or playing will stop (but the local preview will not stop).
     * To prevent the app from being impersonated by a malicious user, you can add authentication before logging in to the room, that is, the [token] parameter in the ZegoRoomConfig object passed in by the [config] parameter. This parameter configuration affects the room to be switched over.
     *
     * @param fromRoomID Current roomID
     * @param toRoomID The next roomID
     * @param config Advanced room configuration
     */
    public abstract void switchRoom(String fromRoomID, String toRoomID, ZegoRoomConfig config);

    /**
     * Set room extra information.
     *
     * User this function to set room extra info, the result will be notified via the IZegoRoomSetRoomExtraInfoCallback. For restrictions on the use of this function, please refer to https://doc-en.zego.im/en/6419.html or contact ZEGO technical support.
     *
     * @param roomID Room ID.
     * @param key key of the extra info.
     * @param value value if the extra info.
     * @param callback Callback for setting room extra information
     */
    public abstract void setRoomExtraInfo(String roomID, String key, String value, IZegoRoomSetRoomExtraInfoCallback callback);

    /**
     * Starts publishing a stream.
     *
     * This function allows users to publish their local audio and video streams to the ZEGO RTC server. Other users in the same room can use the streamID to play the audio and video streams for intercommunication.
     * Before you start to publish the stream, you need to join the room first by calling [loginRoom]. Other users in the same room can get the streamID by monitoring the [onRoomStreamUpdate] event callback after the local user publishing stream successfully.
     * In the case of poor network quality, user publish may be interrupted, and the SDK will attempt to reconnect. You can learn about the current state and error information of the stream published by monitoring the [onPublisherStateUpdate] event.
     * After the first publish stream failure due to network reasons or the publish stream is interrupted, the default time for SDK reconnection is 20min.
     *
     * @param streamID Stream ID, a string of up to 256 characters, needs to be globally unique within the entire AppID. If in the same AppID, different users publish each stream and the stream ID is the same, which will cause the user to publish the stream failure. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     */
    public abstract void startPublishingStream(String streamID);

    /**
     * Starts publishing a stream (for the specified channel). You can call this function to publish a second stream.
     *
     * This function allows users to publish their local audio and video streams to the ZEGO RTC server. Other users in the same room can use the streamID to play the audio and video streams for intercommunication.
     * Before you start to publish the stream, you need to join the room first by calling [loginRoom]. Other users in the same room can get the streamID by monitoring the [onRoomStreamUpdate] event callback after the local user publishing stream successfully.
     * In the case of poor network quality, user publish may be interrupted, and the SDK will attempt to reconnect. You can learn about the current state and error information of the stream published by monitoring the [onPublisherStateUpdate] event.
     * After the first publish stream failure due to network reasons or the publish stream is interrupted, the default time for SDK reconnection is 20min.
     *
     * @param streamID Stream ID, a string of up to 256 characters, needs to be globally unique within the entire AppID. If in the same AppID, different users publish each stream and the stream ID is the same, which will cause the user to publish the stream failure. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     * @param channel Publish stream channel
     */
    public abstract void startPublishingStream(String streamID, ZegoPublishChannel channel);

    /**
     * Stops publishing a stream.
     *
     * This function allows the user to stop sending local audio and video streams and end the call.
     * If the user has initiated publish flow, this function must be called to stop the publish of the current stream before publishing the new stream (new streamID), otherwise the new stream publish will return a failure.
     * After stopping streaming, the developer should stop the local preview based on whether the business situation requires it.
     */
    public abstract void stopPublishingStream();

    /**
     * Stops publishing a stream (for the specified channel).
     *
     * This function allows the user to stop sending local audio and video streams and end the call.
     * If the user has initiated publish flow, this function must be called to stop the publish of the current stream before publishing the new stream (new streamID), otherwise the new stream publish will return a failure.
     * After stopping streaming, the developer should stop the local preview based on whether the business situation requires it.
     * Use this function to stop publishing stream of aux channel.
     *
     * @param channel Publish stream channel
     */
    public abstract void stopPublishingStream(ZegoPublishChannel channel);

    /**
     * Sets the extra information of the stream being published.
     *
     * Use this function to set the extra info of the stream, the result will be notified via the [IZegoPublisherSetStreamExtraInfoCallback].
     * The stream extra information is an extra information identifier of the stream ID. Unlike the stream ID, which cannot be modified during the publishing process, the stream extra information can be modified midway through the stream corresponding to the stream ID.
     * Developers can synchronize variable content related to stream IDs based on stream additional information.
     *
     * @param extraInfo Stream extra information, a string of up to 1024 characters.
     * @param callback Set stream extra information execution result notification
     */
    public abstract void setStreamExtraInfo(String extraInfo, IZegoPublisherSetStreamExtraInfoCallback callback);

    /**
     * Sets the extra information of the stream being published (for the specified channel).
     *
     * Use this function to set the extra info of the stream, the result will be notified via the IZegoPublisherSetStreamExtraInfoCallback.
     *
     * @param extraInfo Stream extra information, a string of up to 1024 characters.
     * @param channel Publish stream channel
     * @param callback Set stream extra information execution result notification
     */
    public abstract void setStreamExtraInfo(String extraInfo, ZegoPublishChannel channel, IZegoPublisherSetStreamExtraInfoCallback callback);

    /**
     * Starts/Updates the local video preview.
     *
     * The user can see his own local image by calling this function. The preview function does not require you to log in to the room or publish the stream first. But after exiting the room, SDK internally actively stops previewing by default.
     * Local view and preview modes can be updated by calling this function again.
     * You can set the mirror mode of the preview by calling the [setVideoMirrorMode] function. The default preview setting is image mirrored.
     * When this function is called, the audio and video engine module inside SDK will start really, and it will start to try to collect audio and video. In addition to calling this function normally to preview the local screen, developers can also pass [null] to the canvas parameter, in conjunction with ZegoExpressEngine's sound wave function, in order to achieve the purpose of detecting whether the audio equipment is working properly before logging in to the room.
     *
     * @param canvas The view used to display the preview image. If the view is set to null, no preview will be made.
     */
    public abstract void startPreview(ZegoCanvas canvas);

    /**
     * Starts/Updates the local video preview (for the specified channel).
     *
     * The user can see his own local image by calling this function. The preview function does not require you to log in to the room or publish the stream first. But after exiting the room, SDK internally actively stops previewing by default.
     * Local view and preview modes can be updated by calling this function again.
     * You can set the mirror mode of the preview by calling the [setVideoMirrorMode] function. The default preview setting is image mirrored.
     * When this function is called, the audio and video engine module inside SDK will start really, and it will start to try to collect audio and video. In addition to calling this function normally to preview the local screen, developers can also pass [null] to the canvas parameter, in conjunction with ZegoExpressEngine's sound wave function, in order to achieve the purpose of detecting whether the audio equipment is working properly before logging in to the room.
     *
     * @param canvas The view used to display the preview image. If the view is set to null, no preview will be made.
     * @param channel Publish stream channel
     */
    public abstract void startPreview(ZegoCanvas canvas, ZegoPublishChannel channel);

    /**
     * Stops the local video preview.
     *
     * This function can be called to stop previewing when there is no need to see the preview locally.
     */
    public abstract void stopPreview();

    /**
     * Stops the local video preview (for the specified channel).
     *
     * This function can be called to stop previewing when there is no need to see the preview locally.
     *
     * @param channel Publish stream channel
     */
    public abstract void stopPreview(ZegoPublishChannel channel);

    /**
     * Sets up the video configurations.
     *
     * This function can be used to set the video frame rate, bit rate, video capture resolution, and video encoding output resolution. If you do not call this function, the default resolution is 360p, the bit rate is 600 kbps, and the frame rate is 15 fps.
     * It is necessary to set the relevant video configuration before publishing the stream or startPreview, and only support the modification of the encoding resolution and the bit rate after publishing the stream.
     * Developers should note that the wide and high resolution of the mobile end is opposite to the wide and high resolution of the PC. For example, in the case of 360p, the resolution of the mobile end is 360x640, and the resolution of the PC end is 640x360.
     *
     * @param config Video configuration, the SDK provides a common setting combination of resolution, frame rate and bit rate, they also can be customized.
     */
    public abstract void setVideoConfig(ZegoVideoConfig config);

    /**
     * Sets up the video configurations (for the specified channel).
     *
     * This function can be used to set the video frame rate, bit rate, video capture resolution, and video encoding output resolution. If you do not call this function, the default resolution is 360p, the bit rate is 600 kbps, and the frame rate is 15 fps.
     * It is necessary to set the relevant video configuration before publishing the stream, and only support the modification of the encoding resolution and the bit rate after publishing the stream.
     * Developers should note that the wide and high resolution of the mobile end is opposite to the wide and high resolution of the PC. For example, in the case of 360p, the resolution of the mobile end is 360x640, and the resolution of the PC end is 640x360.
     *
     * @param config Video configuration, the SDK provides a common setting combination of resolution, frame rate and bit rate, they also can be customized.
     * @param channel Publish stream channel
     */
    public abstract void setVideoConfig(ZegoVideoConfig config, ZegoPublishChannel channel);

    /**
     * Gets the current video configurations.
     *
     * This function can be used to get the main publish channel's current video frame rate, bit rate, video capture resolution, and video encoding output resolution.
     *
     * @return Video configuration object
     */
    public abstract ZegoVideoConfig getVideoConfig();

    /**
     * Gets the current video configurations (for the specified channel).
     *
     * This function can be used to get the specified publish channel's current video frame rate, bit rate, video capture resolution, and video encoding output resolution.
     *
     * @param channel Publish stream channel
     * @return Video configuration object
     */
    public abstract ZegoVideoConfig getVideoConfig(ZegoPublishChannel channel);

    /**
     * Sets the video mirroring mode.
     *
     * This function can be called to set whether the local preview video and the published video have mirror mode enabled.
     *
     * @param mirrorMode Mirror mode for previewing or publishing the stream
     */
    public abstract void setVideoMirrorMode(ZegoVideoMirrorMode mirrorMode);

    /**
     * Sets the video mirroring mode (for the specified channel).
     *
     * This function can be called to set whether the local preview video and the published video have mirror mode enabled.
     *
     * @param mirrorMode Mirror mode for previewing or publishing the stream
     * @param channel Publish stream channel
     */
    public abstract void setVideoMirrorMode(ZegoVideoMirrorMode mirrorMode, ZegoPublishChannel channel);

    /**
     * Sets the video orientation.
     *
     * This function sets the orientation of the video. The captured image is rotated 90, 180 or 270 degrees counterclockwise compared to the forward direction of the phone. After rotation, it is automatically adjusted to match the encoded image resolution.
     *
     * @param orientation Video orientation
     */
    public abstract void setAppOrientation(ZegoOrientation orientation);

    /**
     * Sets the video orientation (for the specified channel).
     *
     * This function sets the orientation of the video. The captured image is rotated 90, 180 or 270 degrees counterclockwise compared to the forward direction of the phone. After rotation, it is automatically adjusted to match the encoded image resolution.
     *
     * @param orientation Video orientation
     * @param channel Publish stream channel
     */
    public abstract void setAppOrientation(ZegoOrientation orientation, ZegoPublishChannel channel);

    /**
     * Sets up the audio configurations.
     *
     * You can set the combined value of the audio codec, bit rate, and audio channel through this function. If this function is not called, the default is standard quality mode. Should be used before publishing.
     * If the preset value cannot meet the developer's scenario, the developer can set the parameters according to the business requirements.
     *
     * @param config Audio config
     */
    public abstract void setAudioConfig(ZegoAudioConfig config);

    /**
     * Sets up the audio configurations.
     *
     * You can set the combined value of the audio codec, bit rate, and audio channel through this function. If this function is not called, the default is standard quality mode. Should be used before publishing.
     * If the preset value cannot meet the developer's scenario, the developer can set the parameters according to the business requirements.
     *
     * @param config Audio config
     * @param channel Publish stream channel
     */
    public abstract void setAudioConfig(ZegoAudioConfig config, ZegoPublishChannel channel);

    /**
     * Gets the current audio configurations.
     *
     * You can get the current audio codec, bit rate, and audio channel through this function.
     *
     * @return Audio config
     */
    public abstract ZegoAudioConfig getAudioConfig();

    /**
     * Gets the current audio configurations.
     *
     * You can get the current audio codec, bit rate, and audio channel through this function.
     *
     * @param channel Publish stream channel
     * @return Audio config
     */
    public abstract ZegoAudioConfig getAudioConfig(ZegoPublishChannel channel);

    /**
     * Set encryption key for the publishing stream.
     *
     * Called before and after [startPublishingStream] can both take effect.
     * Calling [stopPublishingStream] or [logoutRoom] will clear the encryption key.
     * Support calling this function to update the encryption key while publishing stream. Note that developers need to update the player's decryption key before updating the publisher's encryption key.
     * This function is only valid when publishing stream to the Zego RTC server.
     *
     * @param key The encryption key, note that the key length only supports 16/24/32 bytes.
     */
    public abstract void setPublishStreamEncryptionKey(String key);

    /**
     * Set encryption key for the publishing stream.
     *
     * Called before and after [startPublishingStream] can both take effect.
     * Calling [stopPublishingStream] or [logoutRoom] will clear the encryption key.
     * Support calling this function to update the encryption key while publishing stream. Note that developers need to update the player's decryption key before updating the publisher's encryption key.
     * This function is only valid when publishing stream to the Zego RTC server.
     *
     * @param key The encryption key, note that the key length only supports 16/24/32 bytes.
     * @param channel Publish stream channel
     */
    public abstract void setPublishStreamEncryptionKey(String key, ZegoPublishChannel channel);

    /**
     * Take a snapshot of the publishing stream.
     *
     * Please call this function after calling [startPublishingStream] or [startPreview]
     * The resolution of the snapshot is the encoding resolution set in [setVideoConfig]. If you need to change it to capture resolution, please call [setCapturePipelineScaleMode] to change the capture pipeline scale mode to [Post]
     *
     * @param callback Results of take publish stream snapshot
     */
    public abstract void takePublishStreamSnapshot(IZegoPublisherTakeSnapshotCallback callback);

    /**
     * Take a snapshot of the publishing stream (for the specified channel).
     *
     * Please call this function after calling [startPublishingStream] or [startPreview]
     * The resolution of the snapshot is the encoding resolution set in [setVideoConfig]. If you need to change it to capture resolution, please call [setCapturePipelineScaleMode] to change the capture pipeline scale mode to [Post]
     *
     * @param callback Results of take publish stream snapshot
     * @param channel Publish stream channel
     */
    public abstract void takePublishStreamSnapshot(IZegoPublisherTakeSnapshotCallback callback, ZegoPublishChannel channel);

    /**
     * Stops or resumes sending the audio part of a stream.
     *
     * This function can be called when publishing the stream to realize not publishing the audio data stream. The SDK still collects and processes the audio, but does not send the audio data to the network. It can be set before and after publishing.
     * If you stop sending audio streams, the remote user that play stream of local user publishing stream can receive `Mute` status change notification by monitoring [onRemoteMicStateUpdate] callbacks,
     *
     * @param mute Whether to stop sending audio streams, true means not to send audio stream, and false means sending audio stream. The default is false.
     */
    public abstract void mutePublishStreamAudio(boolean mute);

    /**
     * Stops or resumes sending the audio part of a stream (for the specified channel).
     *
     * This function can be called when publishing the stream to realize not publishing the audio data stream. The SDK still collects and processes the audio, but does not send the audio data to the network. It can be set before and after publishing.
     * If you stop sending audio streams, the remote user that play stream of local user publishing stream can receive `Mute` status change notification by monitoring [onRemoteMicStateUpdate] callbacks,
     *
     * @param mute Whether to stop sending audio streams, true means not to send audio stream, and false means sending audio stream. The default is false.
     * @param channel Publish stream channel
     */
    public abstract void mutePublishStreamAudio(boolean mute, ZegoPublishChannel channel);

    /**
     * Stops or resumes sending the video part of a stream.
     *
     * This function can be called when publishing the stream to realize not publishing the video stream. The local camera can still work normally, can capture, preview and process video images normally, but does not send the video data to the network. It can be set before and after publishing.
     * If you stop sending video streams locally, the remote user that play stream of local user publishing stream can receive `Mute` status change notification by monitoring [onRemoteCameraStateUpdate] callbacks,
     *
     * @param mute Whether to stop sending video streams, true means not to send video stream, and false means sending video stream. The default is false.
     */
    public abstract void mutePublishStreamVideo(boolean mute);

    /**
     * Stops or resumes sending the video part of a stream (for the specified channel).
     *
     * This function can be called when publishing the stream to realize not publishing the video stream. The local camera can still work normally, can capture, preview and process video images normally, but does not send the video data to the network. It can be set before and after publishing.
     * If you stop sending video streams locally, the remote user that play stream of local user publishing stream can receive `Mute` status change notification by monitoring [onRemoteCameraStateUpdate] callbacks,
     *
     * @param mute Whether to stop sending video streams, true means not to send video stream, and false means sending video stream. The default is false.
     * @param channel Publish stream channel
     */
    public abstract void mutePublishStreamVideo(boolean mute, ZegoPublishChannel channel);

    /**
     * Enables or disables traffic control.
     *
     * Traffic control enables SDK to dynamically adjust the bitrate of audio and video streaming according to its own and peer current network environment status.
     * Automatically adapt to the current network environment and fluctuations, so as to ensure the smooth publishing of stream.
     *
     * @param enable Whether to enable traffic control. The default is ture.
     * @param property Adjustable property of traffic control, bitmask format. Should be one or the combinations of [ZegoTrafficControlProperty] enumeration. [AdaptiveFPS] as default.
     */
    public abstract void enableTrafficControl(boolean enable, int property);

    /**
     * Enables or disables traffic control.
     *
     * Traffic control enables SDK to dynamically adjust the bitrate of audio and video streaming according to its own and peer current network environment status.
     * Automatically adapt to the current network environment and fluctuations, so as to ensure the smooth publishing of stream.
     *
     * @param enable Whether to enable traffic control. The default is ture.
     * @param property Adjustable property of traffic control, bitmask format. Should be one or the combinations of [ZegoTrafficControlProperty] enumeration. [AdaptiveFPS] as default.
     * @param channel Publish stream channel
     */
    public abstract void enableTrafficControl(boolean enable, int property, ZegoPublishChannel channel);

    /**
     * Sets the minimum video bitrate for traffic control.
     *
     * Set how should SDK send video data when the network conditions are poor and the minimum video bitrate cannot be met.
     * When this function is not called, the SDK will automatically adjust the sent video data frames according to the current network uplink conditions by default.
     *
     * @param bitrate Minimum video bitrate (kbps)
     * @param mode Video sending mode below the minimum bitrate.
     */
    public abstract void setMinVideoBitrateForTrafficControl(int bitrate, ZegoTrafficControlMinVideoBitrateMode mode);

    /**
     * Sets the minimum video bitrate for traffic control.
     *
     * Set how should SDK send video data when the network conditions are poor and the minimum video bitrate cannot be met.
     * When this function is not called, the SDK will automatically adjust the sent video data frames according to the current network uplink conditions by default.
     *
     * @param bitrate Minimum video bitrate (kbps)
     * @param mode Video sending mode below the minimum bitrate.
     * @param channel Publish stream channel
     */
    public abstract void setMinVideoBitrateForTrafficControl(int bitrate, ZegoTrafficControlMinVideoBitrateMode mode, ZegoPublishChannel channel);

    /**
     * Set the factors of concern that trigger traffic control
     *
     * When the traffic control of the specified push channel is enabled through the enableTrafficControl interface, the interface can be used to control whether the traffic control is started due to poor remote network conditions
     * The function is valid before the stream is pushed. If you do not reset the settings before each push, you will continue to use the last configuration.
     *
     * @param mode When LOCAL_ONLY is selected, only the local network status is concerned. When choosing REMOTE, also take into account the remote network.
     */
    public abstract void setTrafficControlFocusOn(ZegoTrafficControlFocusOnMode mode);

    /**
     * Set the factors of concern that trigger traffic control
     *
     * When the traffic control of the specified push channel is enabled through the enableTrafficControl interface, the interface can be used to control whether the traffic control is started due to poor remote network conditions
     * The function is valid before the stream is pushed. If you do not reset the settings before each push, you will continue to use the last configuration.
     *
     * @param mode When LOCAL_ONLY is selected, only the local network status is concerned. When choosing REMOTE, also take into account the remote network.
     * @param channel Publish stream channel
     */
    public abstract void setTrafficControlFocusOn(ZegoTrafficControlFocusOnMode mode, ZegoPublishChannel channel);

    /**
     * Sets the audio recording volume for stream publishing.
     *
     * This function is used to set the audio collection volume. The local user can control the volume of the audio stream sent to the far end. It can be set before publishing.
     *
     * @param volume Volume percentage. The range is 0 to 200. Default value is 100.
     */
    public abstract void setCaptureVolume(int volume);

    /**
     * Set audio capture stereo mode.
     *
     * This function is used to set the audio stereo capture mode. The default is mono, that is, dual channel collection is not enabled.
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param mode Audio stereo capture mode
     */
    public abstract void setAudioCaptureStereoMode(ZegoAudioCaptureStereoMode mode);

    /**
     * Adds a target CDN URL to which the stream will be relayed from ZEGO RTC server.
     *
     * Developers can call this function to publish the audio and video streams that have been published to the ZEGO RTC server to a custom CDN content distribution network that has high latency but supports high concurrent playing stream.
     * Because this called function is essentially a dynamic relay of the audio and video streams published to the ZEGO RTC server to different CDNs, this function needs to be called after the audio and video stream is published to ZEGO RTC server successfully.
     * Since ZEGO RTC server itself can be configured to support CDN(content distribution networks), this function is mainly used by developers who have CDN content distribution services themselves.
     * You can use ZEGO's CDN audio and video streaming content distribution service at the same time by calling this function and then use the developer who owns the CDN content distribution service.
     * This function supports dynamic relay to the CDN content distribution network, so developers can use this function as a disaster recovery solution for CDN content distribution services.
     * When the [enablePublishDirectToCDN] function is set to true to publish the stream straight to the CDN, then calling this function will have no effect.
     *
     * @param streamID Stream ID
     * @param targetURL CDN relay address, supported address format is rtmp.
     * @param callback The execution result of update the relay CDN operation
     */
    public abstract void addPublishCdnUrl(String streamID, String targetURL, IZegoPublisherUpdateCdnUrlCallback callback);

    /**
     * Deletes the specified CDN URL, which is used for relaying streams from ZEGO RTC server to CDN.
     *
     * This function is called when a CDN relayed address has been added and needs to stop propagating the stream to the CDN.
     * This function does not stop publishing audio and video stream to the ZEGO ZEGO RTC server.
     *
     * @param streamID Stream ID
     * @param targetURL CDN relay address, supported address format rtmp.
     * @param callback The execution result of update the relay CDN operation
     */
    public abstract void removePublishCdnUrl(String streamID, String targetURL, IZegoPublisherUpdateCdnUrlCallback callback);

    /**
     * Whether to publish streams directly from the client to CDN without passing through Zego RTC server.
     *
     * This function needs to be set before [startPublishingStream].
     * After calling this function to publish the audio and video stream directly to the CDN, calling [addPublishCdnUrl] and [removePublishCdnUrl] to dynamically relay to the CDN no longer takes effect,
     * because these two functions are to relay or stop relaying the audio and video stream from ZEGO RTC server to CDN,
     * if you enable the direct publish of audio and video streams to CDN, you will not be able to dynamically relay the audio and video streams to the CDN through the ZEGO RTC server.
     *
     * @param enable Whether to enable direct publish CDN, true: enable direct publish CDN, false: disable direct publish CDN
     * @param config CDN configuration, if null, use Zego's background default configuration
     */
    public abstract void enablePublishDirectToCDN(boolean enable, ZegoCDNConfig config);

    /**
     * Whether to publish streams directly from the client to CDN without passing through Zego RTC server (for the specified channel).
     *
     * This function needs to be set before [startPublishingStream].
     * After calling this function to publish the audio and video stream directly to the CDN, calling [addPublishCdnUrl] and [removePublishCdnUrl] to dynamically relay to the CDN no longer takes effect,
     * because these two functions are to relay or stop relaying the audio and video stream from ZEGO RTC server to CDN,
     * if you enable the direct publish of audio and video streams to CDN, you will not be able to dynamically relay the audio and video streams to the CDN through the ZEGO RTC server.
     *
     * @param enable Whether to enable direct publish CDN, true: enable direct publish CDN, false: disable direct publish CDN
     * @param config CDN configuration, if null, use Zego's background default configuration
     * @param channel Publish stream channel
     */
    public abstract void enablePublishDirectToCDN(boolean enable, ZegoCDNConfig config, ZegoPublishChannel channel);

    /**
     * Sets up the stream watermark before stream publishing.
     *
     * The layout of the watermark cannot exceed the video encoding resolution of the stream. It can be set at any time before or during the publishing stream
     *
     * @param watermark The upper left corner of the watermark layout is the origin of the coordinate system, and the area cannot exceed the size set by the encoding resolution. If it is null, the watermark is cancelled.
     * @param isPreviewVisible Whether the watermark can be seen in the local preview.
     */
    public abstract void setPublishWatermark(ZegoWatermark watermark, boolean isPreviewVisible);

    /**
     * Sets up the stream watermark before stream publishing (for the specified channel).
     *
     * The layout of the watermark cannot exceed the video encoding resolution of the stream. It can be set at any time before or during the publishing stream.
     *
     * @param watermark The upper left corner of the watermark layout is the origin of the coordinate system, and the area cannot exceed the size set by the encoding resolution. If it is null, the watermark is cancelled.
     * @param isPreviewVisible the watermark is visible on local preview
     * @param channel Publish stream channel
     */
    public abstract void setPublishWatermark(ZegoWatermark watermark, boolean isPreviewVisible, ZegoPublishChannel channel);

    /**
     * Set the Supplemental Enhancement Information type
     *
     * It must be set before [startPublishingStream].
     *
     * @param config SEI configuration. The SEI defined by ZEGO is used by default.
     */
    public abstract void setSEIConfig(ZegoSEIConfig config);

    /**
     * Sends Supplemental Enhancement Information.
     *
     * This function can synchronize some other additional information while the developer publishes streaming audio and video streaming data while sending streaming media enhancement supplementary information.
     * Generally, for scenarios such as synchronizing music lyrics or precise layout of video canvas, you can choose to use this function.
     * After the anchor sends the SEI, the audience can obtain the SEI content by monitoring the callback of [onPlayerRecvSEI].
     * Since SEI information follows video frames, and because of network problems, frames may be dropped, so SEI information may also be dropped. To solve this situation, it should be sent several times within the limited frequency.
     * After calling [startPublishingStream] to publish the stream successfully, you can call this function.
     * Limit frequency: Do not exceed 30 times per second.
     * The SEI data length is limited to 4096 bytes.
     *
     * @param data SEI data
     */
    public abstract void sendSEI(byte[] data);

    /**
     * Sends Supplemental Enhancement Information.
     *
     * This function can synchronize some other additional information while the developer publishes streaming audio and video streaming data while sending streaming media enhancement supplementary information.
     * Generally, for scenarios such as synchronizing music lyrics or precise layout of video canvas, you can choose to use this function.
     * After the anchor sends the SEI, the audience can obtain the SEI content by monitoring the callback of [onPlayerRecvSEI].
     * Since SEI information follows video frames, and because of network problems, frames may be dropped, so SEI information may also be dropped. To solve this situation, it should be sent several times within the limited frequency.
     * After calling [startPublishingStream] to publish the stream successfully, you can call this function.
     * Limit frequency: Do not exceed 30 times per second.
     * The SEI data length is limited to 4096 bytes.
     *
     * @param data SEI data
     * @param channel Publish stream channel
     */
    public abstract void sendSEI(byte[] data, ZegoPublishChannel channel);

    /**
     * Enables or disables hardware encoding.
     *
     * Whether to use the hardware encoding function when publishing the stream, the GPU is used to encode the stream and to reduce the CPU usage. The setting can take effect before the stream published. If it is set after the stream published, the stream should be stopped first before it takes effect.
     * Because hard-coded support is not particularly good for a few models, SDK uses software encoding by default. If the developer finds that the device is hot when publishing a high-resolution audio and video stream during testing of some models, you can consider calling this function to enable hard coding.
     *
     * @param enable Whether to enable hardware encoding, true: enable hardware encoding, false: disable hardware encoding
     */
    public abstract void enableHardwareEncoder(boolean enable);

    /**
     * Sets the timing of video scaling in the video capture workflow. You can choose to do video scaling right after video capture (the default value) or before encoding.
     *
     * This function needs to be set before previewing or streaming.
     * The main effect is whether the local preview is affected when the acquisition resolution is different from the encoding resolution.
     *
     * @param mode The capture scale timing mode
     */
    public abstract void setCapturePipelineScaleMode(ZegoCapturePipelineScaleMode mode);

    /**
     * Starts playing a stream from ZEGO RTC server.
     *
     * This function allows users to play audio and video streams from the ZEGO RTC server.
     * Before starting to play the stream, you need to join the room first, you can get the new streamID in the room by listening to the [onRoomStreamUpdate] event callback.
     * In the case of poor network quality, user play may be interrupted, the SDK will try to reconnect, and the current play status and error information can be obtained by listening to the [onPlayerStateUpdate] event.
     * Playing the stream ID that does not exist, the SDK continues to try to play after calling this function. After the stream ID is successfully published, the audio and video stream can be actually played.
     * The developer can update the player canvas by calling this function again (the streamID must be the same).
     * After the first play stream failure due to network reasons or the play stream is interrupted, the default time for SDK reconnection is 20min.
     *
     * @param streamID Stream ID, a string of up to 256 characters. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     * @param canvas The view used to display the play audio and video stream's image. When the view is set to [null], no video is displayed, only audio is played.
     */
    public abstract void startPlayingStream(String streamID, ZegoCanvas canvas);

    /**
     * Starts playing a stream from ZEGO RTC server or from third-party CDN.
     *
     * This function allows users to play audio and video streams both from the ZEGO RTC server or from third-party cdn.
     * Before starting to play the stream, you need to join the room first, you can get the new streamID in the room by listening to the [onRoomStreamUpdate] event callback.
     * In the case of poor network quality, user play may be interrupted, the SDK will try to reconnect, and the current play status and error information can be obtained by listening to the [onPlayerStateUpdate] event.
     * Playing the stream ID that does not exist, the SDK continues to try to play after calling this function. After the stream ID is successfully published, the audio and video stream can be actually played.
     * The developer can update the player canvas by calling this function again (the streamID must be the same).
     * After the first play stream failure due to network reasons or the play stream is interrupted, the default time for SDK reconnection is 20min.
     *
     * @param streamID Stream ID, a string of up to 256 characters. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     * @param canvas The view used to display the play audio and video stream's image. When the view is set to [null], no video is displayed, only audio is played.
     * @param config Advanced player configuration
     */
    public abstract void startPlayingStream(String streamID, ZegoCanvas canvas, ZegoPlayerConfig config);

    /**
     * Stops playing a stream.
     *
     * This function allows the user to stop playing the stream. When stopped, the attributes set for this stream previously, such as [setPlayVolume], [mutePlayStreamAudio], [mutePlayStreamVideo], etc., will be invalid and need to be reset when playing the the stream next time.
     *
     * @param streamID Stream ID
     */
    public abstract void stopPlayingStream(String streamID);

    /**
     * Set decryption key for the playing stream.
     *
     * Called before and after [startPlayingStream] can both take effect.
     * Calling [stopPlayingStream] or [logoutRoom] will clear the decryption key.
     * Support calling this function to update the decryption key while playing stream. Note that developers need to update the player's decryption key before updating the publisher's encryption key.
     * This function is only valid when playing stream from Zego RTC or L3 server.
     *
     * @param streamID Stream ID
     * @param key The decryption key, note that the key length only supports 16/24/32 bytes.
     */
    public abstract void setPlayStreamDecryptionKey(String streamID, String key);

    /**
     * Take a snapshot of the playing stream.
     *
     * Please call this function after calling [startPlayingStream]
     *
     * @param streamID Stream ID to be snapshot
     * @param callback Results of take play stream snapshot
     */
    public abstract void takePlayStreamSnapshot(String streamID, IZegoPlayerTakeSnapshotCallback callback);

    /**
     * Sets the stream playback volume.
     *
     * This function is used to set the playback volume of the stream. Need to be called after calling startPlayingStream.
     * You need to reset after [stopPlayingStream] and [startPlayingStream].
     *
     * @param streamID Stream ID. Set volume for all streams playing by set streamID as null or empty.
     * @param volume Volume percentage. The value ranges from 0 to 200, and the default value is 100.
     */
    public abstract void setPlayVolume(String streamID, int volume);

    /**
     * Sets the all stream playback volume.
     *
     * This function is used to set the sound size of all streaming streams, and the local user can control the playback volume of all audio streams.
     *
     * @param volume Volume percentage. The value ranges from 0 to 200, and the default value is 100.
     */
    public abstract void setAllPlayStreamVolume(int volume);

    /**
     * Set the selected video layer of playing stream.
     *
     * When the publisher has set the codecID to SVC through [setVideoConfig], the player can dynamically set whether to use the standard layer or the base layer (the resolution of the base layer is one-half of the standard layer)
     * Under normal circumstances, when the network is weak or the rendered UI form is small, you can choose to use the video that plays the base layer to save bandwidth.
     * It can be set before and after playing stream.
     *
     * @deprecated This function has been deprecated since version 2.3.0. Please use [setPlayStreamVideoType] instead.
     * @param streamID Stream ID.
     * @param videoLayer Video layer of playing stream. AUTO by default.
     */
    @Deprecated
    public abstract void setPlayStreamVideoLayer(String streamID, ZegoPlayerVideoLayer videoLayer);

    /**
     * Set play video stream type
     *
     * When the publish stream sets the codecID to SVC through [setVideoConfig], the puller can dynamically set and select different stream types (small resolution is one-half of the standard layer).
     * In general, when the network is weak or the rendered UI window is small, you can choose to pull videos with small resolutions to save bandwidth.
     * It can be set before and after pulling the stream.
     *
     * @param streamID Stream ID.
     * @param streamType Video stream type
     */
    public abstract void setPlayStreamVideoType(String streamID, ZegoVideoStreamType streamType);

    /**
     * Set the adaptive adjustment interval range of the buffer for playing stream.
     *
     * When the upper limit of the cache interval set by the developer exceeds 4000ms, the value will be 4000ms.
     * When the upper limit of the cache interval set by the developer is less than the lower limit of the cache interval, the upper limit will be automatically set as the lower limit.
     * It can be set before and after playing stream.
     *
     * @param streamID Stream ID.
     * @param minBufferInterval The lower limit of the buffer adaptation interval, in milliseconds. The default value is 0ms
     * @param maxBufferInterval The upper limit of the buffer adaptation interval, in milliseconds. The default value is 4000ms
     */
    public abstract void setPlayStreamBufferIntervalRange(String streamID, int minBufferInterval, int maxBufferInterval);

    /**
     * Set the weight of the pull stream priority.
     *
     * The stream that is set to focus will give priority to ensuring its quality. By default, all streams have the same weight.
     * When the local network is not good, while ensuring the focus flow, it may cause more jams.
     *
     * @param streamID Stream ID.
     */
    public abstract void setPlayStreamFocusOn(String streamID);

    /**
     * Stops or resumes playing the audio part of a stream.
     *
     * This function can be used to stop playing/retrieving the audio data of the stream. It can be called before and after playing the stream.
     *
     * @param streamID Stream ID
     * @param mute Mute flag, true: mute play stream audio, false: resume play stream audio
     */
    public abstract void mutePlayStreamAudio(String streamID, boolean mute);

    /**
     * Stops or resumes playing the video part of a stream.
     *
     * This function can be used to stop playing/retrieving the video data of the stream. It can be called before and after playing the stream.
     *
     * @param streamID Stream ID
     * @param mute mute flag, true: mute play stream video, false: resume play stream video
     */
    public abstract void mutePlayStreamVideo(String streamID, boolean mute);

    /**
     * Stop or resume all pulled audio streams.
     *
     * This function can be called when the stream is pulled, so that the audio data of all remote users is not pulled, and it can be called before and after the stream is pulled.
     * This function does not affect the life cycle of the `mutePlayStreamAudio` interface. This means that neither this function nor the `mutePlayStreamAudio` function must prohibit audio data before audio data can be received.
     *
     * @param mute Mute flag, true: mute play stream audio, false: resume play stream audio
     */
    public abstract void muteAllPlayStreamAudio(boolean mute);

    /**
     * Stop or resume pulling all video streams.
     *
     * This function can be called when the stream is pulled, so that the video data of all remote users is not pulled, and it can be called before and after the stream is pulled.
     * This function does not affect the life cycle of the `mutePlayStreamVideo` interface. This means that neither this function nor the `mutePlayStreamVideo` function prohibits video data before receiving video data.
     *
     * @param mute mute flag, true: mute play stream video, false: resume play stream video
     */
    public abstract void muteAllPlayStreamVideo(boolean mute);

    /**
     * Enables or disables hardware decoding.
     *
     * Turn on hardware decoding and use hardware to improve decoding efficiency. Need to be called before calling startPlayingStream.
     * Because hard-decoded support is not particularly good for a few models, SDK uses software decoding by default. If the developer finds that the device is hot when playing a high-resolution audio and video stream during testing of some models, you can consider calling this function to enable hard decoding.
     *
     * @param enable Whether to turn on hardware decoding switch, true: enable hardware decoding, false: disable hardware decoding. The default is false
     */
    public abstract void enableHardwareDecoder(boolean enable);

    /**
     * Enables or disables frame order detection.
     *
     * @param enable Whether to turn on frame order detection, true: enable check poc,not support B frames, false: disable check poc, support B frames but the screen may temporary splash. The default is true
     */
    public abstract void enableCheckPoc(boolean enable);

    /**
     * Starts a stream mixing task.
     *
     * Due to the performance considerations of the client device, ZegoExpressEngine's mix stream is to start the mixing stream task on the server side of the ZEGO RTC server for mixing stream.
     * After calling this function, SDK initiates a mixing stream request to the ZEGO RTC server. The server will find the current publishing stream and perform video layer blending according to the parameters of the mixing stream task requested by SDK.
     * When you need to update the mixing stream task, that is, the input stream list needs to be updated when the input stream increases or decreases, you can update the field of the [ZegoMixerTask] object inputList and call this function again to pass the same [ZegoMixerTask] object to update the mixing stream task.
     * If an exception occurs when requesting to start the mixing stream task, for example, the most common mix input stream does not exist, it will be given from the callback error code. For specific error codes, please refer to the error codes document https://doc-en.zego.im/en/5548.html for details.
     * If an input stream does not exist in the middle, the mixing stream task will automatically retry playing the input stream for 90 seconds, and will not retry after 90 seconds.
     *
     * @param task Stream mixing task object
     * @param callback Start stream mixing task result callback notification
     */
    public abstract void startMixerTask(ZegoMixerTask task, IZegoMixerStartCallback callback);

    /**
     * Stops a stream mixing task.
     *
     * Similar to [startMixerTask], after calling this function, SDK initiates a request to end the mixing stream task to the ZEGO RTC server.
     * If you starts the next mixing stream task without stopping the previous mixing stream task, the previous mixing stream task will not stop automatically. The previous mixing stream task will not be stopped automatically until 90 seconds after the input stream of the previous mixing stream task does not exist.
     * Developers should pay attention when using the stream mixing function that, before starting the next mixer task, they should stop the previous mixer task, so as avoid that when an anchor has start the next mixer task to mix stream with other anchors, and the audience is still playing the previous mixer task's output stream.
     *
     * @param task Stream mixing task object
     * @param callback Stop stream mixing task result callback notification
     */
    public abstract void stopMixerTask(ZegoMixerTask task, IZegoMixerStopCallback callback);

    /**
     * Mutes or unmutes the microphone.
     *
     * This function is used to control whether the collected audio data is used. When the microphone is muted (disabled), the data is collected and discarded, and the microphone is still occupied.
     * The microphone is still occupied because closing or opening the microphone on the hardware is a relatively heavy operation, and real users may have frequent operations. For trade-off reasons, this function simply discards the collected data.
     * If you really want SDK to give up occupy the microphone, you can call the [enableAudioCaptureDevice] function.
     * Developers who want to control whether to use microphone on the UI should use this function to avoid unnecessary performance overhead by using the [enableAudioCaptureDevice].
     *
     * @param mute Whether to mute (disable) the microphone, true: mute (disable) microphone, false: enable microphone. The default is false.
     */
    public abstract void muteMicrophone(boolean mute);

    /**
     * Checks whether the microphone is muted.
     *
     * Can be used with [muteMicrophone], determine whether the microphone is muted.
     *
     * @return Whether the microphone is muted; true: the microphone is muted; false: the microphone is enable (not muted)
     */
    public abstract boolean isMicrophoneMuted();

    /**
     * Mutes or unmutes the audio output speaker.
     *
     * After mute speaker, all the SDK sounds will not play, including playing stream, mediaplayer, etc. But the SDK will still occupy the output device.
     *
     * @param mute Whether to mute (disable) speaker audio output, true: mute (disable) speaker audio output, false: enable speaker audio output. The default value is false
     */
    public abstract void muteSpeaker(boolean mute);

    /**
     * Checks whether the audio output speaker is muted.
     *
     * Can be used with [muteSpeaker], determine whether the speaker audio output is muted.
     *
     * @return Whether the speaker is muted; true: the speaker is muted; false: the speaker is enable (not muted)
     */
    public abstract boolean isSpeakerMuted();

    /**
     * Enables or disables the audio capture device.
     *
     * This function is used to control whether to release the audio collection device. When the audio collection device is turned off, the SDK will no longer occupy the audio device. Of course, if the stream is being published at this time, there is no audio data.
     * Occupying the audio capture device and giving up Occupying the audio device is a relatively heavy operation, and the [muteMicrophone] function is generally recommended.
     *
     * @param enable Whether to enable the audio capture device, true: disable audio capture device, false: enable audio capture device
     */
    public abstract void enableAudioCaptureDevice(boolean enable);

    /**
     * get current audio route type
     */
    public abstract ZegoAudioRoute getAudioRouteType();

    /**
     * Whether to use the built-in speaker to play audio.
     *
     * When you choose not to use the built-in speaker to play sound, that is, set to [false], the SDK will select the currently highest priority audio output device to play the sound according to the system schedule
     *
     * @param defaultToSpeaker Whether to use the built-in speaker to play sound, true: use the built-in speaker to play sound, false: use the highest priority audio output device scheduled by the current system to play sound
     */
    public abstract void setAudioRouteToSpeaker(boolean defaultToSpeaker);

    /**
     * Turns on/off the camera.
     *
     * This function is used to control whether to start the camera acquisition. After the camera is turned off, video capture will not be performed. At this time, the publish stream will also have no video data.
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param enable Whether to turn on the camera, true: turn on camera, false: turn off camera
     */
    public abstract void enableCamera(boolean enable);

    /**
     * Turns on/off the camera (for the specified channel).
     *
     * This function is used to control whether to start the camera acquisition. After the camera is turned off, video capture will not be performed. At this time, the publish stream will also have no video data.
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param enable Whether to turn on the camera, true: turn on camera, false: turn off camera
     * @param channel Publishing stream channel
     */
    public abstract void enableCamera(boolean enable, ZegoPublishChannel channel);

    /**
     * Switches to the front or the rear camera.
     *
     * This function is used to control the front or rear camera
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param enable Whether to use the front camera, true: use the front camera, false: use the the rear camera. The default value is true
     */
    public abstract void useFrontCamera(boolean enable);

    /**
     * Switches to the front or the rear camera (for the specified channel).
     *
     * This function is used to control the front or rear camera
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param enable Whether to use the front camera, true: use the front camera, false: use the the rear camera. The default value is true
     * @param channel Publishing stream channel
     */
    public abstract void useFrontCamera(boolean enable, ZegoPublishChannel channel);

    /**
     * Set the camera zoom factor.
     *
     * Every time the camera is restarted, the camera zoom factor will be restored to its initial value.
     *
     * @param factor The zoom factor of the camera, the minimum value is 1.0, and the maximum value is the return value of [getCameraMaxZoomFactor].
     */
    public abstract void setCameraZoomFactor(float factor);

    /**
     * Set the camera zoom factor.
     *
     * Every time the camera is restarted, the camera zoom factor will be restored to its initial value.
     *
     * @param factor The zoom factor of the camera, the minimum value is 1.0, and the maximum value is the return value of [getCameraMaxZoomFactor].
     * @param channel Publishing stream channel
     */
    public abstract void setCameraZoomFactor(float factor, ZegoPublishChannel channel);

    /**
     * Get the maximum zoom factor of the camera.
     *
     * This is only available after the camera has been successfully started, and can generally be called when the captured first frame is received, aka [onPublisherCapturedVideoFirstFrame] callback.
     *
     * @return The maximum zoom factor of the camera.
     */
    public abstract float getCameraMaxZoomFactor();

    /**
     * Get the maximum zoom factor of the camera.
     *
     * This is only available after the camera has been successfully started, and can generally be called when the captured first frame is received, aka [onPublisherCapturedVideoFirstFrame] callback.
     *
     * @param channel Publishing stream channel
     * @return The maximum zoom factor of the camera.
     */
    public abstract float getCameraMaxZoomFactor(ZegoPublishChannel channel);

    /**
     * Starts sound level monitoring.
     *
     * After starting monitoring, you can receive local audio sound level via [onCapturedSoundLevelUpdate] callback, and receive remote audio sound level via [onRemoteSoundLevelUpdate] callback.
     * Before entering the room, you can call [startPreview] with this function and combine it with [onCapturedSoundLevelUpdate] callback to determine whether the audio device is working properly.
     * [onCapturedSoundLevelUpdate] and [onRemoteSoundLevelUpdate] callback notification period is 100 ms.
     */
    public abstract void startSoundLevelMonitor();

    /**
     * Starts sound level monitoring. Support setting the listening interval.
     *
     * After starting monitoring, you can receive local audio sound level via [onCapturedSoundLevelUpdate] callback, and receive remote audio sound level via [onRemoteSoundLevelUpdate] callback.
     * Before entering the room, you can call [startPreview] with this function and combine it with [onCapturedSoundLevelUpdate] callback to determine whether the audio device is working properly.
     * [onCapturedSoundLevelUpdate] and [onRemoteSoundLevelUpdate] callback notification period is the value set by the parameter.
     *
     * @param millisecond Monitoring time period of the sound level, in milliseconds, has a value range of [100, 3000]. Default is 100 ms.
     */
    public abstract void startSoundLevelMonitor(int millisecond);

    /**
     * Stops sound level monitoring.
     *
     * After the monitoring is stopped, the callback of the local/remote audio sound level will be stopped.
     */
    public abstract void stopSoundLevelMonitor();

    /**
     * Starts audio spectrum monitoring.
     *
     * After starting monitoring, you can receive local audio spectrum via [onCapturedAudioSpectrumUpdate] callback, and receive remote audio spectrum via [onRemoteAudioSpectrumUpdate] callback.
     * [onCapturedAudioSpectrumUpdate] and [onRemoteAudioSpectrumUpdate] callback notification period is 100 ms.
     */
    public abstract void startAudioSpectrumMonitor();

    /**
     * Starts audio spectrum monitoring. Support setting the listening interval.
     *
     * After starting monitoring, you can receive local audio spectrum via [onCapturedAudioSpectrumUpdate] callback, and receive remote audio spectrum via [onRemoteAudioSpectrumUpdate] callback.
     * [onCapturedAudioSpectrumUpdate] and [onRemoteAudioSpectrumUpdate] callback notification period is the value set by the parameter.
     *
     * @param millisecond Monitoring time period of the audio spectrum, in milliseconds, has a value range of [100, 3000]. Default is 100 ms.
     */
    public abstract void startAudioSpectrumMonitor(int millisecond);

    /**
     * Stops audio spectrum monitoring.
     *
     * After the monitoring is stopped, the callback of the local/remote audio spectrum will be stopped.
     */
    public abstract void stopAudioSpectrumMonitor();

    /**
     * Enables or disables headphone monitoring.
     *
     * enable/disable headphone monitor, this setting takes effect when the headset is connected.
     *
     * @param enable Whether to use headphone monitor, true: enable, false: disable
     */
    public abstract void enableHeadphoneMonitor(boolean enable);

    /**
     * Sets the headphone monitor volume.
     *
     * set headphone monitor volume, this setting takes effect when the headset is connected.
     *
     * @param volume headphone monitor volume, range from 0 to 200, 100 as default
     */
    public abstract void setHeadphoneMonitorVolume(int volume);

    /**
     * Enables or disables acoustic echo cancellation (AEC).
     *
     * Turning on echo cancellation, the SDK filters the collected audio data to reduce the echo component in the audio.
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable echo cancellation, true: enable, false: disable
     */
    public abstract void enableAEC(boolean enable);

    /**
     * Whether to turn on acoustic echo cancellation (AEC) when using the headphone.
     *
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable, true: enable, false: disable
     */
    public abstract void enableHeadphoneAEC(boolean enable);

    /**
     * Sets the acoustic echo cancellation (AEC) mode.
     *
     * Switch different echo cancellation modes to control the extent to which echo data is eliminated.
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param mode Echo cancellation mode
     */
    public abstract void setAECMode(ZegoAECMode mode);

    /**
     * Enables or disables automatic gain control (AGC).
     *
     * When the auto gain is turned on, the sound will be amplified, but it will affect the sound quality to some extent.
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable automatic gain control, true: enable, false: disable
     */
    public abstract void enableAGC(boolean enable);

    /**
     * Enables or disables active noise suppression (ANS, aka ANC).
     *
     * Turning on the noise suppression switch can reduce the noise in the audio data and make the human voice clearer.
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable noise suppression, true: enable, false: disable
     */
    public abstract void enableANS(boolean enable);

    /**
     * Enables or disables transient noise suppression.
     *
     * Suppress transient noises such as keyboard and desk knocks
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable transient noise suppression, true: enable, false: disable
     */
    public abstract void enableTransientANS(boolean enable);

    /**
     * Sets the automatic noise suppression (ANS) mode.
     *
     * Default is medium mode
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param mode Audio Noise Suppression mode
     */
    public abstract void setANSMode(ZegoANSMode mode);

    /**
     * Enables or disables audio mixing.
     *
     * Enable audio mixing, work with setAudioMixingHandler providing the audio data for mixing
     *
     * @param enable Whether to enable audio mixting, true: enable, false: disable
     */
    public abstract void enableAudioMixing(boolean enable);

    /**
     * Sets up the audio mixing event handler.
     *
     * @param handler Audio mixing callback handler
     */
    public abstract void setAudioMixingHandler(IZegoAudioMixingHandler handler);

    /**
     * Stops or resumes playing the mixed audio locally.
     *
     * When stop play audio mixing locally, the audio will not be heard on the mix side (publisher side) while still be heard on the remote side (player side).
     *
     * @param mute Whether to mute local audio mixting, true: mute, false: unmute
     */
    public abstract void muteLocalAudioMixing(boolean mute);

    /**
     * Sets the audio mixing output volume for both local playback and the stream to be published.
     *
     * This function will modify the volume of the local playback and the mixed data that is mixed into the publishing stream at the same time.
     *
     * @param volume The audio mixing volume, range from 0 to 200, 100 as default
     */
    public abstract void setAudioMixingVolume(int volume);

    /**
     * Sets the audio mixing output volume for either local playback or the stream to published.
     *
     * This function can individually set the mixing volume of local playback or the mixing volume of the publishing stream.
     *
     * @param volume The audio mixing volume, range from 0 to 200, 100 as default
     * @param type volume local playback / volume in stream published
     */
    public abstract void setAudioMixingVolume(int volume, ZegoVolumeType type);

    /**
     * Enables or disables the beauty features.
     *
     * The current beauty function is simple and may not meet the developer's expectations, it is recommended to use [enableCustomVideoCapture] function to connect to a third party professional beauty SDK to get the best results.
     * The [setBeautifyOption] function can be called to adjust the beauty parameters after the beauty function is enabled.
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param featureBitmask Beauty features, bitmask format, you can choose to enable several features in [ZegoBeautifyFeature] at the same time
     */
    public abstract void enableBeautify(int featureBitmask);

    /**
     * Enables or disables the beauty features (for the specified channel).
     *
     * The current beauty function is simple and may not meet the developer's expectations, it is recommended to use [enableCustomVideoCapture] function to connect to a third party professional beauty SDK to get the best results.
     * The [setBeautifyOption] function can be called to adjust the beauty parameters after the beauty function is enabled.
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param featureBitmask Beauty features, bitmask format, you can choose to enable several features in [ZegoBeautifyFeature] at the same time
     * @param channel Publishing stream channel
     */
    public abstract void enableBeautify(int featureBitmask, ZegoPublishChannel channel);

    /**
     * Sets up the beauty parameters.
     *
     * Developer need to call [enableBeautify] function first to enable the beautify function before calling this function
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param option Beauty configuration options
     */
    public abstract void setBeautifyOption(ZegoBeautifyOption option);

    /**
     * Sets up the beauty parameters (for the specified channel).
     *
     * Developer need to call [enableBeautify] function first to enable the beautify function before calling this function
     * In the case of using a custom video capture function, because the developer has taken over the video data capturing, the SDK is no longer responsible for the video data capturing, this function is no longer valid.
     *
     * @param option Beauty configuration options
     * @param channel Publishing stream channel
     */
    public abstract void setBeautifyOption(ZegoBeautifyOption option, ZegoPublishChannel channel);

    /**
     * Set the sound equalizer (EQ).
     *
     * @param bandIndex Band frequency index, the value range is [0, 9], corresponding to 10 frequency bands, and the center frequencies are [31, 62, 125, 250, 500, 1K, 2K, 4K, 8K, 16K] Hz.
     * @param bandGain Band gain for the index, the value range is [-15, 15]. Default value is 0, if all gain values in all frequency bands are 0, EQ function will be disabled.
     */
    public abstract void setAudioEqualizerGain(int bandIndex, float bandGain);

    /**
     * Setting up the voice changer via preset enumeration.
     *
     * Voice changer effect is only effective for the captured sound.
     * This function is an encapsulated version of [setVoiceChangerParam], which provides some preset values. If you need to configure the voice changer effects, please use [setVoiceChangerParam]
     * This function is mutually exclusive with [setReverbPreset]. If used at the same time, it will produce undefined effects.
     * Some enumerated preset will modify the parameters of reverberation or reverberation echo, so after calling this function, calling [setVoiceChangerParam], [setReverbAdvancedParam], [setReverbEchoParam] may affect the voice changer effect.
     * If you need to configure the reverb/echo/voice changer effect, please use [setReverbAdvancedParam], [setReverbEchoParam], [setVoiceChangerParam] together.
     *
     * @param preset The voice changer preset enumeration
     */
    public abstract void setVoiceChangerPreset(ZegoVoiceChangerPreset preset);

    /**
     * Setting up the specific voice changer parameters.
     *
     * Voice changer effect is only effective for the captured sound.
     * This function is an advanced version of [setVoiceChangerPreset], you can configure the voice changer effect by yourself.
     * If you need to configure the reverb/echo/voice changer effect, please use [setReverbAdvancedParam], [setReverbEchoParam], [setVoiceChangerParam] together.
     *
     * @param param Voice changer parameters
     */
    public abstract void setVoiceChangerParam(ZegoVoiceChangerParam param);

    /**
     * Setting up the reverberation via preset enumeration.
     *
     * Support dynamic settings when publishing stream.
     * This function is a encapsulated version of [setReverbAdvancedParam], which provides some preset values. If you need to configure the reverb, please use [setReverbAdvancedParam]
     * This function is mutually exclusive with [setVoiceChangerPreset]. If used at the same time, it will produce undefined effects.
     * If you need to configure the reverb/echo/voice changer effect, please use [setReverbAdvancedParam], [setReverbEchoParam], [setVoiceChangerParam] together.
     *
     * @param preset The reverberation preset enumeration
     */
    public abstract void setReverbPreset(ZegoReverbPreset preset);

    /**
     * Setting up the specific reverberation parameters.
     *
     * Different values dynamically set during publishing stream will take effect. When all parameters are set to 0, the reverberation is turned off.
     * This function is an advanced version of [setReverbPreset], you can configure the reverb effect by yourself.
     * If you need to configure the reverb/echo/voice changer effect, please use [setReverbAdvancedParam], [setReverbEchoParam], [setVoiceChangerParam] together.
     *
     * @param param Reverb advanced parameter
     */
    public abstract void setReverbAdvancedParam(ZegoReverbAdvancedParam param);

    /**
     * Setting up the specific reverberation echo parameters.
     *
     * This function can be used with voice changer and reverb to achieve a variety of custom sound effects
     * If you need to configure the reverb/echo/voice changer effect, please use [setReverbAdvancedParam], [setReverbEchoParam], [setVoiceChangerParam] together.
     *
     * @param param The reverberation echo parameter
     */
    public abstract void setReverbEchoParam(ZegoReverbEchoParam param);

    /**
     * Enables the virtual stereo feature.
     *
     * Note: You need to set up a dual channel setAudioConfig for the virtual stereo to take effect!
     *
     * @param enable true to turn on the virtual stereo, false to turn off the virtual stereo
     * @param angle angle of the sound source in the virtual stereo, ranging from 0 to 180, with 90 being the front, and 0 and 180 being respectively Corresponds to rightmost and leftmost, usually use 90.
     */
    public abstract void enableVirtualStereo(boolean enable, int angle);

    /**
     * Sends a Broadcast Message.
     *
     * The sending frequency of broadcast messages in the same room cannot be higher than 10 messages/s.
     * A certain number of users in the same room who entered the room earlier can receive this callback. The message is reliable. It is generally used when the number of people in the live room is less than a certain number. The specific number is determined by the configuration of the ZEGO server.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param message Message content, no longer than 1024 bytes
     * @param callback Send broadcast message result callback
     */
    public abstract void sendBroadcastMessage(String roomID, String message, IZegoIMSendBroadcastMessageCallback callback);

    /**
     * Sends a Barrage Message (bullet screen) to all users in the same room, without guaranteeing the delivery.
     *
     * The frequency of sending barrage messages in the same room cannot be higher than 20 messages/s.
     * The message is unreliable. When the frequency of sending barrage messages in the entire room is greater than 20 messages/s, the recipient may not receive the message. It is generally used in scenarios where there is a large number of messages sent and received in the room and the reliability of the messages is not required, such as live broadcast barrage.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param message Message content, no longer than 1024 bytes
     * @param callback Send barrage message result callback
     */
    public abstract void sendBarrageMessage(String roomID, String message, IZegoIMSendBarrageMessageCallback callback);

    /**
     * Sends a Custom Command to the specified users in the same room.
     *
     * The frequency of custom messages sent to a single user in the same room cannot be higher than 200 messages/s, and the frequency of custom messages sent to multiple users cannot be higher than 10 messages/s.
     * The point-to-point signaling type in the same room is generally used for remote control signaling or for sending messages between users. The messages are reliable.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param command Custom command content, no longer than 1024 bytes
     * @param toUserList The users who will receive the command
     * @param callback Send command result callback
     */
    public abstract void sendCustomCommand(String roomID, String command, ArrayList<ZegoUser> toUserList, IZegoIMSendCustomCommandCallback callback);

    /**
     * Creates a media player instance.
     *
     * Currently, a maximum of 4 instances can be created, after which it will return null. The more instances of a media player, the greater the performance overhead on the device.
     *
     * @return Media player instance, null will be returned when the maximum number is exceeded.
     */
    public abstract ZegoMediaPlayer createMediaPlayer();

    /**
     * Destroys a media player instance.
     *
     * @param mediaPlayer The media player instance object to be destroyed
     */
    public abstract void destroyMediaPlayer(ZegoMediaPlayer mediaPlayer);

    /**
     * Creates a audio effect player instance.
     *
     * Currently, a maximum of 1 instances can be created, after which it will return null.
     *
     * @return audio effect player instance, null will be returned when the maximum number is exceeded.
     */
    public abstract ZegoAudioEffectPlayer createAudioEffectPlayer();

    /**
     * Destroys a audio effect player instance.
     *
     * @param audioEffectPlayer The audio effect player instance object to be destroyed
     */
    public abstract void destroyAudioEffectPlayer(ZegoAudioEffectPlayer audioEffectPlayer);

    /**
     * Starts to record locally captured audio or video and directly save the data to a file.
     *
     * Currently only one task can be recorded simultaneously.
     * This function needs to be called after the success of [startPreview] or [startPublishingStream] to be effective.
     * Developers should not [stopPreview] or [stopPublishingStream] during recording, otherwise the SDK will end the current recording task.
     * Developers will receive the [onCapturedDataRecordStateUpdate] and the [onCapturedDataRecordProgressUpdate] callback after start recording.
     *
     * @param config Record config
     * @param channel Publishing stream channel
     */
    public abstract void startRecordingCapturedData(ZegoDataRecordConfig config, ZegoPublishChannel channel);

    /**
     * Stops recording locally captured audio or video.
     *
     * @param channel Publishing stream channel
     */
    public abstract void stopRecordingCapturedData(ZegoPublishChannel channel);

    /**
     * Sets up the event callback handler for data recording.
     *
     * Implemente the functions of ZegoDataRecordEventHandler to get notified when recode state and process changed
     *
     * @param eventHandler Data record event handler
     */
    public abstract void setDataRecordEventHandler(IZegoDataRecordEventHandler eventHandler);

    /**
     * Starts system performance status monitoring. Support setting the listening interval.
     *
     * After starting monitoring, you can receive system performance status via [onPerformanceStatusUpdate] callback.
     * [onPerformanceStatusUpdate] callback notification period is the value set by the parameter.
     *
     * @param millisecond Monitoring time period of the audio spectrum, in milliseconds, has a value range of [1000, 10000]. Default is 2000 ms.
     */
    public abstract void startPerformanceMonitor(int millisecond);

    /**
     * Stops system performance status monitoring.
     *
     * After the monitoring is stopped, the callback of the system performance status will be stopped.
     */
    public abstract void stopPerformanceMonitor();

    /**
     * start network probe.
     *
     * Some local network problems may cause audio and video calls to fail. Using this function to probe the network protocols, assist in locating and solving related network problems.
     * The SDK internally detects http, tcp, and udp in sequence. If the probe fails in the middle, the subsequent detection will not continue. Therefore, when reading the values ​​in the probe result, please pay attention to check if the value is null.
     * The SDK will not perform multiple network probe at the same time, that is, if the network probe is in progress, the SDK will not work if you call this method repeatedly.
     * Network probe may take a long time. Developers can call [stopNetworkProbe] to stop network probe as needed.
     *
     * @param config network probe config
     * @param callback Network probe result callback.
     */
    public abstract void startNetworkProbe(ZegoNetworkProbeConfig config, IZegoNetworkProbeResultCallback callback);

    /**
     * stop network probe.
     */
    public abstract void stopNetworkProbe();

    /**
     * Start network speed test.
     *
     * Developers can listen to the [onNetworkSpeedTestQualityUpdate] callback to get the speed test result, which will be called back every 3 seconds.
     * If an error occurs during the speed measurement process, [onNetworkSpeedTestError] callback will be triggered.
     * If this function is repeatedly called multiple times, the last invoke's configuration will be used.
     *
     * @param config Network speed test configuration.
     */
    public abstract void startNetworkSpeedTest(ZegoNetworkSpeedTestConfig config);

    /**
     * Stop network speed test.
     *
     * After stopping the speed test, [onNetworkSpeedTestQualityUpdate] will no longer call back.
     */
    public abstract void stopNetworkSpeedTest();

    /**
     * Enables or disables custom video rendering.
     *
     * It must be set before the engine starts, that is, before calling [startPreview], [startPublishingStream], [startPlayingStream]; and the configuration can be modified after the engine is stopped， that is after calling [logoutRoom].
     * When the developer starts custom rendering, by calling [setCustomVideoRenderHandler], you can set to receive local and remote video frame data for custom rendering
     *
     * @param enable enable or disable
     * @param config custom video render config
     */
    public abstract void enableCustomVideoRender(boolean enable, ZegoCustomVideoRenderConfig config);

    /**
     * Sets up the event callback handler for custom video rendering.
     *
     * Custom video render, developers are responsible for rendering video data to UI components. This feature is generally used by developers who use third-party beauty features or use third-party rendering frameworks.
     * When you use the advanced features of ZegoExpressEngine's custom video render, you need to call this function to set a callback object for developers to transfer video data.
     * When you call the start preview [startPreview], start publishing stream [startPublishingStream], or start playing stream [startPlayingStream], the callback function that transfer video data to you will be triggered.
     * You can render video images according to the callback of SDK transfering video data.
     * Custom video rendering can be used with custom video capture.
     *
     * @param handler Custom video render handler
     */
    public abstract void setCustomVideoRenderHandler(IZegoCustomVideoRenderHandler handler);

    /**
     * Enables or disables custom video capture.
     *
     * It must be set before the engine starts, that is, before calling [startPreview], [startPublishingStream]; and the configuration can be modified after the engine is stopped, that is, after calling [logoutRoom].
     * When the developer starts the custom capture, it can be set to receive notification of the start and stop of the custom capture by calling [setCustomVideoCaptureHandler].
     *
     * @param enable enable or disable
     * @param config custom video capture config
     */
    public abstract void enableCustomVideoCapture(boolean enable, ZegoCustomVideoCaptureConfig config);

    /**
     * Enables or disables custom video capture (for the specified channel).
     *
     * It must be set before the engine starts, that is, before calling [startPreview], [startPublishingStream]; and the configuration can be modified after the engine is stopped, that is, after calling [logoutRoom].
     * When the developer starts the custom capture, it can be set to receive notification of the start and stop of the custom capture by calling [setCustomVideoCaptureHandler].
     *
     * @param enable enable or disable
     * @param config custom video capture config
     * @param channel publish channel
     */
    public abstract void enableCustomVideoCapture(boolean enable, ZegoCustomVideoCaptureConfig config, ZegoPublishChannel channel);

    /**
     * Sets the event callback handler for custom video capture.
     *
     * Custom video capture, that is, the developer is responsible for collecting video data and sending the collected video data to SDK for video data encoding and publishing to the ZEGO RTC server. This feature is generally used by developers who use third-party beauty features or record game screen living.
     * When you use the advanced features of ZegoExpressEngine's custom video capture, you need to call this function to set SDK to notify that you can start sending video data to ZegoExpressEngine.
     * When you calls startPreview and startPublishingStream, a callback function that notifies you to start sending video data will be triggered. When you stop preview [stopPreview] and stop publishing stream [stopPublishingStream], it will trigger a callback function that notify that you can stop sending video data.
     * Because when using custom video capture, SDK will no longer start the camera to capture video data. You need to collect video data from video sources by yourself.
     * Custom video capture can be used with custom video rendering.
     *
     * @param handler Custom video capture handler
     */
    public abstract void setCustomVideoCaptureHandler(IZegoCustomVideoCaptureHandler handler);

    /**
     * Sends the video frames (Raw Data) produced by custom video capture to the SDK.
     *
     * This function need to be start called after the [onStart] callback notification and to be stop called call after the [onStop] callback notification.
     *
     * @param data video frame data
     * @param dataLength video frame data length
     * @param params video frame param
     * @param referenceTimeMillisecond video frame reference time, UNIX timestamp, in milliseconds.
     */
    public abstract void sendCustomVideoCaptureRawData(ByteBuffer data, int dataLength, ZegoVideoFrameParam params, long referenceTimeMillisecond);

    /**
     * Sends the video frames (Raw Data) produced by custom video capture to the SDK (for the specified channel).
     *
     * This function need to be start called after the [onStart] callback notification and to be stop called call after the [onStop] callback notification.
     *
     * @param data video frame data
     * @param dataLength video frame data length
     * @param params video frame param
     * @param referenceTimeMillisecond video frame reference time, UNIX timestamp, in milliseconds.
     * @param channel Publishing stream channel
     */
    public abstract void sendCustomVideoCaptureRawData(ByteBuffer data, int dataLength, ZegoVideoFrameParam params, long referenceTimeMillisecond, ZegoPublishChannel channel);

    /**
     * Sends the video frames (Texture Data) produced by custom video capture to the SDK.
     *
     * This function need to be start called after the [onStart] callback notification and to be stop called call after the [onStop] callback notification.
     *
     * @param textureID texture ID
     * @param width Video frame width
     * @param height Video frame height
     * @param referenceTimeMillisecond Timestamp of this video frame
     */
    public abstract void sendCustomVideoCaptureTextureData(int textureID, int width, int height, double referenceTimeMillisecond);

    /**
     * Sends the video frames (Texture Data) produced by custom video capture to the SDK (for the specified channel).
     *
     * This function need to be start called after the [onStart] callback notification and to be stop called call after the [onStop] callback notification.
     *
     * @param textureID texture ID
     * @param width Video frame width
     * @param height Video frame height
     * @param referenceTimeMillisecond Timestamp of this video frame
     * @param channel Publishing stream channel
     */
    public abstract void sendCustomVideoCaptureTextureData(int textureID, int width, int height, double referenceTimeMillisecond, ZegoPublishChannel channel);

    /**
     * Gets the SurfaceTexture instance.
     *
     * @return SurfaceTexture instance
     */
    public abstract SurfaceTexture getCustomVideoCaptureSurfaceTexture();

    /**
     * Gets the SurfaceTexture instance (for the specified channel).
     *
     * @param channel Publishing stream channel
     * @return SurfaceTexture instance
     */
    public abstract SurfaceTexture getCustomVideoCaptureSurfaceTexture(ZegoPublishChannel channel);

    /**
     * Sends the video frames (Encoded Data) produced by custom video capture to the SDK.
     *
     * This function need to be start called after the [onStart] callback notification and to be stop called call after the [onStop] callback notification.
     *
     * @param data video encoded frame data
     * @param dataLength video encoded frame data length
     * @param params video encoded frame param
     * @param referenceTimeMillisecond video frame reference time, UNIX timestamp, in milliseconds.
     */
    public abstract void sendCustomVideoCaptureEncodedData(ByteBuffer data, int dataLength, ZegoVideoEncodedFrameParam params, long referenceTimeMillisecond);

    /**
     * Sends the video frames (Encoded Data) produced by custom video capture to the SDK (for the specified channel).
     *
     * This function need to be start called after the [onStart] callback notification and to be stop called call after the [onStop] callback notification.
     *
     * @param data video frame encoded data
     * @param dataLength video frame data length
     * @param params video frame param
     * @param referenceTimeMillisecond video frame reference time, UNIX timestamp, in milliseconds.
     * @param channel Publishing stream channel
     */
    public abstract void sendCustomVideoCaptureEncodedData(ByteBuffer data, int dataLength, ZegoVideoEncodedFrameParam params, long referenceTimeMillisecond, ZegoPublishChannel channel);

    /**
     * Sets the video fill mode of custom video capture.
     *
     * @param mode View mode
     */
    public abstract void setCustomVideoCaptureFillMode(ZegoViewMode mode);

    /**
     * Sets the video fill mode of custom video capture (for the specified channel).
     *
     * @param mode View mode
     * @param channel Publishing stream channel
     */
    public abstract void setCustomVideoCaptureFillMode(ZegoViewMode mode, ZegoPublishChannel channel);

    /**
     * Sets the video flip mode of custom video capture (for the specified channel). This function takes effect only if the custom video buffer type is Texture2D.
     *
     * @param mode Flip mode
     */
    public abstract void setCustomVideoCaptureFlipMode(ZegoVideoFlipMode mode);

    /**
     * Sets the video flip mode of custom video capture (for the specified channel). This function takes effect only if the custom video buffer type is Texture2D.
     *
     * @param mode Flip mode
     * @param channel Publishing stream channel
     */
    public abstract void setCustomVideoCaptureFlipMode(ZegoVideoFlipMode mode, ZegoPublishChannel channel);

    /**
     * Enables or disables custom video processing.
     *
     * When developers to open before the custom processing, by calling [setCustomVideoCaptureHandler] can be set up to receive a custom video processing of the video data
     * Precondition： Call [CreateEngine] to initialize the Zego SDK
     * Call timing： must be set before calling [startPreview], [startPublishingStream]; The configuration cannot be changed again until the [logoutRoom] is called, otherwise the call will not take effect
     * Supported version： 2.2.0
     *
     * @param enable enable or disable. disable by default
     * @param config custom video processing configuration. If NULL is passed, the platform default value is used.
     */
    public abstract void enableCustomVideoProcessing(boolean enable, ZegoCustomVideoProcessConfig config);

    /**
     * Enables or disables custom video processing.
     *
     * When developers to open before the custom processing, by calling [setCustomVideoCaptureHandler] can be set up to receive a custom video processing of the original video data
     * Precondition： Call [CreateEngine] to initialize the Zego SDK
     * Call timing： must be set before calling [startPreview], [startPublishingStream]; The configuration cannot be changed again until the [logoutRoom] is called, otherwise the call will not take effect
     * Supported version： 2.2.0
     *
     * @param enable enable or disable. disable by default
     * @param config custom video processing configuration. If NULL is passed, the platform default value is used.
     * @param channel Publishing stream channel
     */
    public abstract void enableCustomVideoProcessing(boolean enable, ZegoCustomVideoProcessConfig config, ZegoPublishChannel channel);

    /**
     * Sets up the event callback handler for custom video processing.
     *
     * Set a custom video processing callback to obtain the original video data
     * Precondition：Call [createEngine] to initialize Zego SDK
     * Call timing：Call before [startPreview] and [startPublishingStream], otherwise it may cause too slow time to get video data
     * Supported version：2.2.0
     *
     * @param handler Custom video process handler
     */
    public abstract void setCustomVideoProcessHandler(IZegoCustomVideoProcessHandler handler);

    /**
     * Send the [Texture] type video data after the pre-processing of the custom video to the SDK
     *
     * This interface takes effect when [enableCustomVideoProcessing] is called to enable custom video processing and the bufferType of config is passed in [ZEGO_VIDEO_BUFFER_TYPE_GL_TEXTURE_2D]
     * Developers are requested to ensure that they have sufficient openGL programming experience before using Texture type video processing, otherwise it is easy to cause unforeseen problems
     * Precondition：Call [createEngine] to initialize Zego SDK
     * Call timing：It must be called in the [onCapturedUnprocessedTextureData] callback
     * Supported version：2.2.0
     *
     * @param textureID texture ID
     * @param width Texture width
     * @param height Texture height
     * @param referenceTimeMillisecond video frame reference time, UNIX timestamp, in milliseconds.
     */
    public abstract void sendCustomVideoProcessedTextureData(int textureID, int width, int height, long referenceTimeMillisecond);

    /**
     * Send the [Texture] type video data after the pre-processing of the custom video to the SDK (for the specified channel).
     *
     * This interface takes effect when [enableCustomVideoProcessing] is called to enable custom video processing and the bufferType of config is passed in [ZEGO_VIDEO_BUFFER_TYPE_GL_TEXTURE_2D]
     * Developers are requested to ensure that they have sufficient openGL programming experience before using Texture type video processing, otherwise it is easy to cause unforeseen problems
     * Precondition：Call [createEngine] to initialize Zego SDK
     * Call timing：It must be called in the [onCapturedUnprocessedTextureData] callback
     * Supported version：2.2.0
     *
     * @param textureID texture ID
     * @param width Texture width
     * @param height Texture height
     * @param referenceTimeMillisecond video frame reference time, UNIX timestamp, in milliseconds.
     * @param channel Publishing stream channel
     */
    public abstract void sendCustomVideoProcessedTextureData(int textureID, int width, int height, long referenceTimeMillisecond, ZegoPublishChannel channel);

    /**
     * Get SurfaceTexture for output data
     *
     * This interface takes effect when [enableCustomVideoProcessing] is called to enable custom video pre-processing and the bufferType of config is passed in [ZEGO_VIDEO_BUFFER_TYPE_SURFACETEXTURE]
     * After the developer receives the original video data, they can call this interface to get the SurfaceTexture used to output the video data
     * After the developer obtains the Output SurfaceTexture, it needs to be converted to the Surface type as the producer of the output data. After the developer completes the video processing, send the processed video data to the SDK through this Surface
     * Precondition：Call [createEngine] to initialize Zego SDK
     * Call timing： It is recommended to call in the callback [onGetSurfaceTexture]
     * Supported version： 2.2.0
     *
     * @param width Specify the width of the output video data
     * @param height Specify the height of the output video data
     * @return SurfaceTexture instance
     */
    public abstract SurfaceTexture getCustomVideoProcessOutputSurfaceTexture(int width, int height);

    /**
     * Get SurfaceTexture for output data (for the specified channel).
     *
     * This interface takes effect when [enableCustomVideoProcessing] is called to enable custom video pre-processing and the bufferType of config is passed in [ZEGO_VIDEO_BUFFER_TYPE_SURFACETEXTURE]
     * After the developer receives the original video data, they can call this interface to get the SurfaceTexture used to output the video data
     * After the developer obtains the Output SurfaceTexture, it needs to be converted to the Surface type as the producer of the output data. After the developer completes the video processing, send the processed video data to the SDK through this Surface
     * Precondition：Call [createEngine] to initialize Zego SDK
     * Call timing：It is recommended to call in the callback [onGetSurfaceTexture]
     * Supported version：2.2.0
     *
     * @param width Specify the width of the output video data
     * @param height Specify the height of the output video data
     * @param channel Publishing stream channel
     * @return SurfaceTexture instance
     */
    public abstract SurfaceTexture getCustomVideoProcessOutputSurfaceTexture(int width, int height, ZegoPublishChannel channel);

    /**
     * Enable local capture custom audio processing.
     *
     * When enabled, developers can receive local captured audio frame through [onProcessCapturedAudioData], and can modify the audio data.
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable local capture custom audio processing.
     * @param config Custom audio processing configuration.
     */
    public abstract void enableCustomAudioCaptureProcessing(boolean enable, ZegoCustomAudioProcessConfig config);

    /**
     * Enable custom audio processing for remote playing stream.
     *
     * When enabled, developers can receive audio frame from remote playing stream through [onProcessRemoteAudioData], and can modify the audio data.
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable custom audio processing for remote playing stream.
     * @param config Custom audio processing configuration.
     */
    public abstract void enableCustomAudioRemoteProcessing(boolean enable, ZegoCustomAudioProcessConfig config);

    /**
     * Set up callback handler for custom audio processing.
     *
     * Developers can modify the processing of audio frame data in the callback.
     *
     * @param handler Callback handler for custom audio processing.
     */
    public abstract void setCustomAudioProcessHandler(IZegoCustomAudioProcessHandler handler);

    /**
     * Enables the callback for receiving audio data.
     *
     * The callback to the corresponding setting of [setAudioDataHandler] is triggered when this function is called and at publishing stream state or playing stream state. If you want to enable the [onPlayerAudioData] callback, the sample rate passed in by calling the [enableAudioDataCallback] function does not support 8000Hz, 22050Hz and 24000Hz.
     *
     * @param enable Whether to enable audio data callback
     * @param callbackBitMask The callback function bitmask marker for receive audio data, refer to enum [ZegoAudioDataCallbackBitMask], when this param converted to binary, 0b01 that means 1 << 0 for triggering [onCapturedAudioData], 0x10 that means 1 << 1 for triggering [onPlaybackAudioData], 0x100 that means 1 << 2 for triggering [onMixedAudioData], 0x1000 that means 1 << 3 for triggering [onPlayerAudioData]. The masks can be combined to allow different callbacks to be triggered simultaneously.
     * @param param param of audio frame
     */
    public abstract void enableAudioDataCallback(boolean enable, int callbackBitMask, ZegoAudioFrameParam param);

    /**
     * Sets up the event callback handler for receiving audio data.
     *
     * You can call this function to receive audio data from the SDK when you need to get the audio data of the remote user or get the data collected by the local microphone for other purposes (such as audio-only recording, audio-only third-party monitoring, and audio-only real-time analysis).
     * The set callback needs to be after the call to [enableAudioDataCallback] and in a publishing or playing stream state for the callback to work.
     * The format of the audio data callback by SDK is pcm. The SDK still controls the collection and playback of the sound device. This function is to copy a piece of data collected or played inside the SDK to the developer for use.
     *
     * @param handler Audio data handler for receive audio data
     */
    public abstract void setAudioDataHandler(IZegoAudioDataHandler handler);

    /**
     * Enables the custom audio I/O function.
     *
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable custom audio IO, default is false
     * @param config Custom audio IO config
     */
    public abstract void enableCustomAudioIO(boolean enable, ZegoCustomAudioConfig config);

    /**
     * Enables the custom audio I/O function (for the specified channel).
     *
     * It needs to be invoked before [startPublishingStream], [startPlayingStream], [startPreview], [createMediaPlayer] and [createAudioEffectPlayer] to take effect.
     *
     * @param enable Whether to enable custom audio IO, default is false
     * @param config Custom audio IO config
     * @param channel Specify the publish channel to enable custom audio IO
     */
    public abstract void enableCustomAudioIO(boolean enable, ZegoCustomAudioConfig config, ZegoPublishChannel channel);

    /**
     * Sends AAC audio data produced by custom audio capture to the SDK.
     *
     * @param data AAC buffer data
     * @param dataLength The total length of the buffer data
     * @param configLength The length of AAC specific config (Note: The AAC encoded data length is 'encodedLength = dataLength - configLength')
     * @param referenceTimeMillisecond The UNIX timestamp of this AAC audio frame in millisecond.
     * @param param The param of this AAC audio frame
     */
    public abstract void sendCustomAudioCaptureAACData(ByteBuffer data, int dataLength, int configLength, long referenceTimeMillisecond, ZegoAudioFrameParam param);

    /**
     * Sends AAC audio data produced by custom audio capture to the SDK (for the specified channel).
     *
     * @param data AAC buffer data
     * @param dataLength The total length of the buffer data
     * @param configLength The length of AAC specific config (Note: The AAC encoded data length is 'encodedLength = dataLength - configLength')
     * @param referenceTimeMillisecond The UNIX timestamp of this AAC audio frame in millisecond.
     * @param param The param of this AAC audio frame
     * @param channel Publish channel for capturing audio frames
     */
    public abstract void sendCustomAudioCaptureAACData(ByteBuffer data, int dataLength, int configLength, long referenceTimeMillisecond, ZegoAudioFrameParam param, ZegoPublishChannel channel);

    /**
     * Sends PCM audio data produced by custom audio capture to the SDK.
     *
     * @param data PCM buffer data
     * @param dataLength The total length of the buffer data
     * @param param The param of this PCM audio frame
     */
    public abstract void sendCustomAudioCapturePCMData(ByteBuffer data, int dataLength, ZegoAudioFrameParam param);

    /**
     * Sends PCM audio data produced by custom audio capture to the SDK (for the specified channel).
     *
     * @param data PCM buffer data
     * @param dataLength The total length of the buffer data
     * @param param The param of this PCM audio frame
     * @param channel Publish channel for capturing audio frames
     */
    public abstract void sendCustomAudioCapturePCMData(ByteBuffer data, int dataLength, ZegoAudioFrameParam param, ZegoPublishChannel channel);

    /**
     * Fetches PCM audio data of the remote stream for custom audio rendering.
     *
     * It is recommended to use the system framework to periodically invoke this function to drive audio data rendering
     *
     * @param data A block of memory for storing audio PCM data that requires user to manage the memory block's lifecycle, the SDK will copy the audio frame rendering data to this memory block
     * @param dataLength The length of the audio data to be fetch this time (dataLength = duration * sampleRate * channels * 2(16 bit depth i.e. 2 Btye))
     * @param param Specify the parameters of the fetched audio frame
     */
    public abstract void fetchCustomAudioRenderPCMData(ByteBuffer data, int dataLength, ZegoAudioFrameParam param);

    /**
     * Setting up the specific reverberation parameters.
     *
     * Different values dynamically set during publishing stream will take effect. When all parameters are set to 0, the reverberation is turned off.
     * This function is an advanced version of [setReverbPreset], you can configure the reverb effect by yourself.
     * If you need to configure the reverb/echo/voice changer effect, please use [setReverbAdvancedParam], [setReverbEchoParam], [setVoiceChangerParam] together.
     *
     * @deprecated This function has been deprecated since version 1.18.0, please use [setReverbAdvancedParam] instead.
     * @param param Reverb parameter
     */
    @Deprecated
    public abstract void setReverbParam(ZegoReverbParam param);

    /**
     * Whether to use the built-in speaker to play audio.This function has been deprecated since version 2.3.0. Please use [setAudioRouteToSpeaker] instead.
     *
     * When you choose not to use the built-in speaker to play sound, that is, set to false, the SDK will select the currently highest priority audio output device to play the sound according to the system schedule
     *
     * @deprecated This function has been deprecated since version 2.3.0. Please use [setAudioRouteToSpeaker] instead.
     * @param enable Whether to use the built-in speaker to play sound, true: use the built-in speaker to play sound, false: use the highest priority audio output device scheduled by the current system to play sound
     */
    @Deprecated
    public abstract void setBuiltInSpeakerOn(boolean enable);

}

