package im.zego.zegoexpress.internal;

import android.os.Handler;
import android.os.Looper;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import im.zego.zegoexpress.ZegoAudioEffectPlayer;
import im.zego.zegoexpress.callback.IZegoAudioEffectPlayerEventHandler;
import im.zego.zegoexpress.callback.IZegoAudioEffectPlayerLoadResourceCallback;
import im.zego.zegoexpress.callback.IZegoAudioEffectPlayerSeekToCallback;
import im.zego.zegoexpress.entity.ZegoAudioEffectPlayConfig;
import im.zego.zegoexpress.utils.ZegoDebugLevel;

public class ZegoAudioEffectPlayerInternalImpl extends ZegoAudioEffectPlayer {
    public static Handler mUIHandler = new Handler(Looper.getMainLooper());

    static class IdxAndHandler {
        int kAudioEffectPlayerIdx;
        ConcurrentHashMap<Integer, IZegoAudioEffectPlayerSeekToCallback> seekToTimeCallbackHashMap;
        ConcurrentHashMap<Integer, IZegoAudioEffectPlayerLoadResourceCallback> loadResourceCallbackHashMap;
        IZegoAudioEffectPlayerEventHandler eventHandler;

        IdxAndHandler(int idx) {
            this.kAudioEffectPlayerIdx = idx;
        }
    }

    public static ConcurrentHashMap<ZegoAudioEffectPlayer, IdxAndHandler> audioEffectPlayerToIdxAndEventhandler = new ConcurrentHashMap<>();

    public static ZegoAudioEffectPlayer createAudioEffectPlayer() {
        int index = ZegoAudioEffectPlayerJniAPI.createZegoAudioEffectPlayerJni();
        if (index >= 0) {
            ZegoAudioEffectPlayer zegoAudioEffectPlayer = new ZegoAudioEffectPlayerInternalImpl();
            IdxAndHandler idxAndHandler = new IdxAndHandler(index);
            idxAndHandler.seekToTimeCallbackHashMap = new ConcurrentHashMap<>();
            idxAndHandler.loadResourceCallbackHashMap = new ConcurrentHashMap<>();
            audioEffectPlayerToIdxAndEventhandler.put(zegoAudioEffectPlayer, idxAndHandler);
            return zegoAudioEffectPlayer;
        } else {
            ZegoExpressEngineInternalImpl.printDebugInfo(ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "createAudioEffectPlayer", 1014004);

            return null;
        }
    }

    public static void destroyAudioEffectPlayer(ZegoAudioEffectPlayer audioEffectPlayer) {
        for (Iterator<Map.Entry<ZegoAudioEffectPlayer, IdxAndHandler>> it = audioEffectPlayerToIdxAndEventhandler.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<ZegoAudioEffectPlayer, IdxAndHandler> item = it.next();
            if (item.getKey() == audioEffectPlayer) {
                item.getValue().eventHandler = null;
                item.getValue().loadResourceCallbackHashMap = null;
                Iterator<Map.Entry<Integer, IZegoAudioEffectPlayerSeekToCallback>> itSeekToTimeCallbackHashMap = item.getValue().seekToTimeCallbackHashMap.entrySet().iterator();
                for (; itSeekToTimeCallbackHashMap.hasNext(); ) {
                    itSeekToTimeCallbackHashMap.remove();
                }
                item.getValue().seekToTimeCallbackHashMap = null;
                int idx = item.getValue().kAudioEffectPlayerIdx;
                int errorCode = ZegoAudioEffectPlayerJniAPI.destroyAudioEffectPlayerJni(idx);
                it.remove();
                ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                        ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "destroyAudioEffectPlayer", errorCode);

            }
        }
    }

    @Override
    public void setEventHandler(IZegoAudioEffectPlayerEventHandler handler) {
        IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            idxAndHandler.eventHandler = handler;
        }
    }

    @Override
    public void loadResource(int audioEffectID, String path, IZegoAudioEffectPlayerLoadResourceCallback callback) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int seq = ZegoAudioEffectPlayerJniAPI.loadResourceJni(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx, path);
            synchronized (ZegoAudioEffectPlayerInternalImpl.class) {
                idxAndHandler.loadResourceCallbackHashMap.put(seq, callback);
            }
        }
    }

    @Override
    public void unloadResource(int audioEffectID) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.unloadResource(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "unloadResource", errorCode);

        }
    }

    @Override
    public void start(int audioEffectID, String path, ZegoAudioEffectPlayConfig config) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.start(audioEffectID, path, config, idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "start", errorCode);

        }
    }

    @Override
    public void stop(int audioEffectID) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.stop(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "stop", errorCode);

        }
    }

    @Override
    public void pause(int audioEffectID) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.pause(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "pause", errorCode);

        }
    }

    @Override
    public void resume(int audioEffectID) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.resume(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "resume", errorCode);

        }
    }

    @Override
    public void stopAll() {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
           int errorCode = ZegoAudioEffectPlayerJniAPI.stopAll(idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "stopAll", errorCode);

        }
    }

    @Override
    public void pauseAll() {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.pauseAll(idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "pauseAll", errorCode);

        }
    }

    @Override
    public void resumeAll() {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.resumeAll(idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "resumeAll", errorCode);

        }
    }

    @Override
    public void seekTo(int audioEffectID, long millisecond, IZegoAudioEffectPlayerSeekToCallback callback) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int seq = ZegoAudioEffectPlayerJniAPI.seekToJni(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx, millisecond);
            synchronized (ZegoAudioEffectPlayerInternalImpl.class) {
                idxAndHandler.seekToTimeCallbackHashMap.put(seq, callback);
            }
        }
    }

    @Override
    public void setVolume(int audioEffectID, int volume) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.setVolume(audioEffectID, volume, idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "setVolume", errorCode);

        }
    }

    @Override
    public void setVolumeAll(int volume) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            int errorCode = ZegoAudioEffectPlayerJniAPI.setVolumeAll(volume, idxAndHandler.kAudioEffectPlayerIdx);
            ZegoExpressEngineInternalImpl.printDebugInfo((errorCode == 0) ? ZegoDebugLevel.INFO : ZegoDebugLevel.ERROR,
                    ZegoExpressEngineInternalImpl.ZegoInnerModule.AUDIO_EFFECT_PLAYER, "setVolumeAll", errorCode);

        }
    }

    @Override
    public long getTotalDuration(int audioEffectID) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            return ZegoAudioEffectPlayerJniAPI.getTotalDuration(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx);
        }
        return -1;
    }

    @Override
    public long getCurrentProgress(int audioEffectID) {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            return ZegoAudioEffectPlayerJniAPI.getCurrentProgress(audioEffectID, idxAndHandler.kAudioEffectPlayerIdx);
        }
        return -1;
    }

    @Override
    public int getIndex() {
        final IdxAndHandler idxAndHandler = audioEffectPlayerToIdxAndEventhandler.get(this);
        if (idxAndHandler != null) {
            return idxAndHandler.kAudioEffectPlayerIdx;
        }
        return -1;
    }
}
