package im.zego.zegoexpress;

import android.os.Handler;
import android.os.Looper;

import im.zego.zegoexpress.constants.ZegoAudioChannel;
import im.zego.zegoexpress.constants.ZegoAudioSampleRate;
import im.zego.zegoexpress.constants.ZegoMediaPlayerNetworkEvent;
import im.zego.zegoexpress.constants.ZegoMediaPlayerState;
import im.zego.zegoexpress.constants.ZegoVideoFrameFormat;
import im.zego.zegoexpress.entity.ZegoAudioFrameParam;
import im.zego.zegoexpress.entity.ZegoCanvas;
import im.zego.zegoexpress.entity.ZegoVideoFrameParam;
import im.zego.zegoexpress.callback.IZegoMediaPlayerAudioHandler;
import im.zego.zegoexpress.callback.IZegoMediaPlayerEventHandler;
import im.zego.zegoexpress.callback.IZegoMediaPlayerLoadResourceCallback;
import im.zego.zegoexpress.callback.IZegoMediaPlayerSeekToCallback;
import im.zego.zegoexpress.utils.ZegoCallbackHelpers;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public final class ZegoMediaPlayerJni {

    private static Handler mUIHandler = new Handler(Looper.getMainLooper());




    /* 在 ZegoMediaplayerJni 里同时管理对于JNI的调用回调与 Mediaplayer 多实例的管理*/

    static class IdxAndHandler {
        int kMediaPlayerIdx;
        HashMap<Integer, IZegoMediaPlayerSeekToCallback> seekToTimeCallbackHashMap;
        IZegoMediaPlayerLoadResourceCallback loadResourceCallback;
        IZegoMediaPlayerEventHandler eventHandler;
        IZegoMediaPlayerAudioHandler audioHandler;
        Object videoHandler;

        IdxAndHandler(int idx) {
            this.kMediaPlayerIdx = idx;
        }
    }

    private static HashMap<ZegoMediaPlayer, IdxAndHandler> mediaplayerToIdxAndEventhandler = new HashMap<>();

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void setEventHandler(ZegoMediaPlayer mediaplayer, IZegoMediaPlayerEventHandler handler) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            idxAndHandler.eventHandler = handler;
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void onMediaPlayerStateUpdateCallback(int state, final int errorcode, int idx) {

        final ZegoMediaPlayerState zegoMediaPlayerState = ZegoMediaPlayerState.values()[state];
        synchronized (ZegoMediaPlayerJni.class) {
            for (final Map.Entry<ZegoMediaPlayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()) {
                if (item.getValue().kMediaPlayerIdx == idx) {
                    final IZegoMediaPlayerEventHandler iZegoMediaPlayerEventHandler = item.getValue().eventHandler;
                    if (iZegoMediaPlayerEventHandler == null) {
                        //todo: 这里需要增加日志打印
                        return;
                    } else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                iZegoMediaPlayerEventHandler.onMediaPlayerStateUpdate(item.getKey(), zegoMediaPlayerState, errorcode);
                            }
                        });
                    }
                }
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void onMediaPlayerNetWorkEventCallback(int event, int idx) {

        final ZegoMediaPlayerNetworkEvent zegoMediaPlayerNetworkEvent = ZegoMediaPlayerNetworkEvent.values()[event];
        synchronized (ZegoMediaPlayerJni.class) {
            for (final Map.Entry<ZegoMediaPlayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()) {
                if (item.getValue().kMediaPlayerIdx == idx) {
                    final IZegoMediaPlayerEventHandler iZegoMediaPlayerEventHandler = item.getValue().eventHandler;
                    if (iZegoMediaPlayerEventHandler == null) {
                        //todo: 这里需要增加日志打印
                        return;
                    } else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                iZegoMediaPlayerEventHandler.onMediaPlayerNetworkEvent(item.getKey(), zegoMediaPlayerNetworkEvent);
                            }
                        });
                    }
                }
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void onMediaPlayerPlayingProgressCallback(int idx, final long millisecond) {
        synchronized (ZegoMediaPlayerJni.class) {
            for (final Map.Entry<ZegoMediaPlayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()) {
                if (item.getValue().kMediaPlayerIdx == idx) {
                    final IZegoMediaPlayerEventHandler iZegoMediaPlayerEventHandler = item.getValue().eventHandler;
                    if (iZegoMediaPlayerEventHandler == null) {
                        //todo: 这里需要增加日志打印
                        return;
                    } else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                synchronized (ZegoMediaPlayerJni.class) {
                                    iZegoMediaPlayerEventHandler.onMediaPlayerPlayingProgress(item.getKey(), millisecond);
                                }
                            }
                        });
                    }
                }
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void setVideoHandler(ZegoMediaPlayer mediaplayer, ZegoVideoFrameFormat frameFormat, Object handler) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            idxAndHandler.videoHandler = handler;
            if (handler == null) {
                ZegoMediaPlayerJni.enableVideoDataJni(frameFormat.value(), false, idxAndHandler.kMediaPlayerIdx);
            } else {
                ZegoMediaPlayerJni.enableVideoDataJni(frameFormat.value(), true, idxAndHandler.kMediaPlayerIdx);
            }
        }
    }
    // opt_content_end: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void onMediaPlayerVideoHandlerCallback(int idx, final ByteBuffer[] buffer, final int[] datalength, int[] strides, int format, int width, int height) {
        final ZegoVideoFrameParam videoFrameParam = new ZegoVideoFrameParam();
        videoFrameParam.height = height;
        for (int i = 0; i < strides.length; i++) {
            videoFrameParam.strides[i] = strides[i];
        }
        videoFrameParam.width = width;
        videoFrameParam.format = ZegoVideoFrameFormat.values()[format];

        synchronized (ZegoMediaPlayerJni.class) {
            for (final Map.Entry<ZegoMediaPlayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()) {
                if (item.getValue().kMediaPlayerIdx == idx) {
                    final Object mediaPlayerVideoHandler = item.getValue().videoHandler;
                    if (mediaPlayerVideoHandler == null) {
                        //todo: 这里需要增加日志打印
                        return;
                    } else {
                        ZegoCallbackHelpers.callMediaVideoFrameMethod(mediaPlayerVideoHandler, item.getKey(), buffer, datalength, videoFrameParam);
                    }
                }
            }
        }
    }
    // opt_content_end: audio_or_video = video; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: audio_or_video = audio; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void setAudioHandler(ZegoMediaPlayer mediaplayer, IZegoMediaPlayerAudioHandler handler) {
        IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            idxAndHandler.audioHandler = handler;
            if (handler == null) {
                ZegoMediaPlayerJni.enableAudioDataJni(false, idxAndHandler.kMediaPlayerIdx);
            } else {
                ZegoMediaPlayerJni.enableAudioDataJni(true, idxAndHandler.kMediaPlayerIdx);
            }
        }
    }
    // opt_content_end: audio_or_video = audio; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: audio_or_video = audio; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void onMediaPlayerAudioHandlerCallback(int idx, final ByteBuffer buffer, final int bufferLength, int channel, int samplesRate) {

        final ZegoAudioFrameParam audioFrame = new ZegoAudioFrameParam();
        audioFrame.channel = ZegoAudioChannel.values()[channel];
        audioFrame.sampleRate = ZegoAudioSampleRate.getZegoAudioSampleRate(samplesRate);

        synchronized (ZegoMediaPlayerJni.class) {
            for (final Map.Entry<ZegoMediaPlayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()) {
                if (item.getValue().kMediaPlayerIdx == idx) {
                    final IZegoMediaPlayerAudioHandler mediaPlayerAudioHandler = item.getValue().audioHandler;
                    if (mediaPlayerAudioHandler == null) {
                        //todo: 这里需要增加日志打印
                        return;
                    } else {
                        mediaPlayerAudioHandler.onAudioFrame(item.getKey(), buffer, bufferLength, audioFrame);
                    }
                }
            }
        }
    }
    // opt_content_end: audio_or_video = audio; sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void loadResource(ZegoMediaPlayer mediaplayer, String path, IZegoMediaPlayerLoadResourceCallback callback) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.loadResourceJni(idxAndHandler.kMediaPlayerIdx, path);
            idxAndHandler.loadResourceCallback = callback;
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    public static int getIndex(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            return idxAndHandler.kMediaPlayerIdx;
        }
        return 0;
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void onLoadResourceCallback(int idx, final int errorcode) {
        synchronized (ZegoMediaPlayerJni.class) {
            for (final Map.Entry<ZegoMediaPlayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()) {
                if (item.getValue().kMediaPlayerIdx == idx) {
                    final IZegoMediaPlayerLoadResourceCallback iZegoMediaPlayerLoadResourceCallback = item.getValue().loadResourceCallback;
                    if (iZegoMediaPlayerLoadResourceCallback == null) {
                        //todo: 这里需要增加日志打印
                        return;
                    } else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                iZegoMediaPlayerLoadResourceCallback.onLoadResourceCallback(errorcode);
                            }
                        });
                    }
                }
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void start(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.startJni(idxAndHandler.kMediaPlayerIdx);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void stop(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.stopJni(idxAndHandler.kMediaPlayerIdx);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void pause(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.pauseJni(idxAndHandler.kMediaPlayerIdx);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void resume(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.resumeJni(idxAndHandler.kMediaPlayerIdx);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void seekTo(ZegoMediaPlayer mediaplayer, long millisecond, IZegoMediaPlayerSeekToCallback callback) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            int seq = ZegoMediaPlayerJni.seekToJni(idxAndHandler.kMediaPlayerIdx, millisecond);
            synchronized (ZegoMediaPlayerJni.class) {
                idxAndHandler.seekToTimeCallbackHashMap.put(seq, callback);
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void onSeekToTimeCallback(final int seq, int idx, final int errorcode) {
        synchronized (ZegoMediaPlayerJni.class) {
            for (final Map.Entry<ZegoMediaPlayer, IdxAndHandler> item : mediaplayerToIdxAndEventhandler.entrySet()) {
                if (item.getValue().kMediaPlayerIdx == idx) {
                    final IZegoMediaPlayerSeekToCallback seekToTimeCallback = item.getValue().seekToTimeCallbackHashMap.get(seq);
                    if (seekToTimeCallback == null) {
                        //todo: 这里需要增加日志打印
                        return;
                    } else {
                        mUIHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                seekToTimeCallback.onSeekToTimeCallback(errorcode);
                                item.getValue().seekToTimeCallbackHashMap.remove(seq);
                            }
                        });
                    }
                }
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void enableRepeat(ZegoMediaPlayer mediaplayer, boolean enable) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.enableRepeatJni(idxAndHandler.kMediaPlayerIdx, enable);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    public static void enableAux(ZegoMediaPlayer mediaplayer, boolean enable) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.enableAuxJni(idxAndHandler.kMediaPlayerIdx, enable);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    public static void muteLocal(ZegoMediaPlayer mediaplayer, boolean mute) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.muteLocalJni(idxAndHandler.kMediaPlayerIdx, mute);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void setPlayerCanvas(ZegoMediaPlayer mediaplayer, ZegoCanvas canvas) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            if (canvas != null) {
                ZegoMediaPlayerJni.setPlayerCanvasJni(idxAndHandler.kMediaPlayerIdx, canvas.view, canvas.viewMode.value(), canvas.backgroundColor);
            } else {
                ZegoMediaPlayerJni.setPlayerCanvasJni(idxAndHandler.kMediaPlayerIdx, null, 0, 0);

            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void setVolume(ZegoMediaPlayer mediaplayer, int volume) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.setVolumeJni(idxAndHandler.kMediaPlayerIdx, volume);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static int getVolume(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            return ZegoMediaPlayerJni.getVolumeJni(idxAndHandler.kMediaPlayerIdx);
        }
        return -1;
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void setProgressInterval(ZegoMediaPlayer mediaplayer, long millisecond) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            ZegoMediaPlayerJni.setProgressIntervalJni(idxAndHandler.kMediaPlayerIdx, millisecond);
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static long getTotalDuration(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            return ZegoMediaPlayerJni.getTotalDurationJni(idxAndHandler.kMediaPlayerIdx);
        }
        return -1;
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static long getCurrentProgress(ZegoMediaPlayer mediaplayer) {
        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            return ZegoMediaPlayerJni.getCurrentProgressJni(idxAndHandler.kMediaPlayerIdx);
        }
        return -1;
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER


    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    public static ZegoMediaPlayer createMediaPlayer() {
        int index = ZegoMediaPlayerJni.createMediaPlayerJni();
        if (index >= 0) {
            ZegoMediaPlayer zegoMediaPlayer = new ZegoMediaPlayer();
            IdxAndHandler idxAndHandler = new IdxAndHandler(index);
            idxAndHandler.seekToTimeCallbackHashMap = new HashMap<>();
            mediaplayerToIdxAndEventhandler.put(zegoMediaPlayer, idxAndHandler);
            return zegoMediaPlayer;
        } else {
            return null;
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static void destroyMediaPlayer(ZegoMediaPlayer mediaplayer) {
        synchronized (ZegoMediaPlayerJni.class) {
            for (Iterator<Map.Entry<ZegoMediaPlayer, IdxAndHandler>> it = mediaplayerToIdxAndEventhandler.entrySet().iterator(); it.hasNext(); ) {
                Map.Entry<ZegoMediaPlayer, IdxAndHandler> item = it.next();
                if (item.getKey() == mediaplayer) {
                    item.getValue().videoHandler = null;
                    item.getValue().audioHandler = null;
                    item.getValue().eventHandler = null;
                    item.getValue().loadResourceCallback = null;
                    Iterator<Map.Entry<Integer, IZegoMediaPlayerSeekToCallback>> itSeekToTimeCallbackHashMap = item.getValue().seekToTimeCallbackHashMap.entrySet().iterator();
                    for (; itSeekToTimeCallbackHashMap.hasNext(); ) {
                        itSeekToTimeCallbackHashMap.remove();
                    }
                    item.getValue().seekToTimeCallbackHashMap = null;
                    int idx = item.getValue().kMediaPlayerIdx;
                    ZegoMediaPlayerJni.destroyMediaPlayerJni(idx);
                    it.remove();
                }
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    public static void destroyAllMediaPlayer() {
        synchronized (ZegoMediaPlayerJni.class) {
            for (Iterator<Map.Entry<ZegoMediaPlayer, IdxAndHandler>> it = mediaplayerToIdxAndEventhandler.entrySet().iterator(); it.hasNext(); ) {
                Map.Entry<ZegoMediaPlayer, IdxAndHandler> item = it.next();
                item.getValue().videoHandler = null;
                item.getValue().audioHandler = null;
                item.getValue().eventHandler = null;
                item.getValue().loadResourceCallback = null;
                Iterator<Map.Entry<Integer, IZegoMediaPlayerSeekToCallback>> itSeekToTimeCallbackHashMap = item.getValue().seekToTimeCallbackHashMap.entrySet().iterator();
                for (; itSeekToTimeCallbackHashMap.hasNext(); ) {
                    itSeekToTimeCallbackHashMap.remove();
                }
                item.getValue().seekToTimeCallbackHashMap = null;
                int idx = item.getValue().kMediaPlayerIdx;
                ZegoMediaPlayerJni.destroyMediaPlayerJni(idx);
                it.remove();
            }
        }
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    // opt_content_start: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER
    static ZegoMediaPlayerState getCurrentState(ZegoMediaPlayer mediaplayer) {

        final IdxAndHandler idxAndHandler = mediaplayerToIdxAndEventhandler.get(mediaplayer);
        if (idxAndHandler != null) {
            int state = ZegoMediaPlayerJni.getCurrentStateJni(idxAndHandler.kMediaPlayerIdx);
            final ZegoMediaPlayerState zegoMediaPlayerState;
            if (state == ZegoMediaPlayerState.NO_PLAY.value()) {
                zegoMediaPlayerState = ZegoMediaPlayerState.NO_PLAY;
            } else if (state == ZegoMediaPlayerState.PLAYING.value()) {
                zegoMediaPlayerState = ZegoMediaPlayerState.PLAYING;
            } else if (state == ZegoMediaPlayerState.PAUSING.value()) {
                zegoMediaPlayerState = ZegoMediaPlayerState.PAUSING;
            } else if (state == ZegoMediaPlayerState.PLAY_ENDED.value()) {
                zegoMediaPlayerState = ZegoMediaPlayerState.PLAY_ENDED;
            } else {
                zegoMediaPlayerState = ZegoMediaPlayerState.NO_PLAY;
            }
            return zegoMediaPlayerState;
        }
        return ZegoMediaPlayerState.NO_PLAY;
    }
    // opt_content_end: sdk_exclusive_strategy = ZEGO_ENABLE_FEATURE_MEDIA_PLAYER

    native static int getCurrentStateJni(int idx);

    native static int destroyMediaPlayerJni(int idx);

    native static int createMediaPlayerJni();

    native static int startJni(int idx);

    native static int stopJni(int idx);

    native static int pauseJni(int idx);

    native static int resumeJni(int idx);

    native static int enableRepeatJni(int idx, boolean enable);

    native static int enableAuxJni(int idx, boolean enable);

    native static int muteLocalJni(int idx, boolean mute);

    native static int setPlayerCanvasJni(int idx, Object view, int viewMode, int backgroundColor);

    native static int setVolumeJni(int idx, int volume);

    native static int setProgressIntervalJni(int idx, long millisecond);

    native static long getTotalDurationJni(int idx);

    native static long getCurrentProgressJni(int idx);

    native static int seekToJni(int idx, long millisecond);

    native static int loadResourceJni(int idx, String path);

    native static int enableAudioDataJni(boolean enable, int idx);

    native static int enableVideoDataJni(int value, boolean enable, int idx);

    native static int getVolumeJni(int idx);
}

