package im.zego.zegoexpress.entity;

import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.entity.*;
import java.nio.*;
import java.util.*;
import org.json.*;
import im.zego.zegoexpress.*;
import im.zego.zegoexpress.internal.*;
import android.app.Application;
import android.graphics.*;

/**
 * Video config
 *
 * Configure parameters used for publishing stream, such as bitrate, frame rate, and resolution.
 * Developers should note that the width and height resolution of the mobile and desktop are opposite. For example, 360p, the resolution of the mobile is 360x640, and the desktop is 640x360.
 */
public class ZegoVideoConfig {

    /** Capture resolution width, control the width of camera image acquisition. Only the camera is not started and the custom video capture is not used, the setting is effective */
    public int captureWidth;

    /** Capture resolution height, control the height of camera image acquisition. Only the camera is not started and the custom video capture is not used, the setting is effective */
    public int captureHeight;

    /** Encode resolution width, control the image width of the encoder when publishing stream. The settings before and after publishing stream can be effective */
    public int encodeWidth;

    /** Encode resolution height, control the image height of the encoder when publishing stream. The settings before and after publishing stream can be effective */
    public int encodeHeight;

    /** frame rate, control the frame rate of the camera and the frame rate of the encoder. Only the camera is not started, the setting is effective */
    public int fps;

    /** Bit rate in kbps. The settings before and after publishing stream can be effective */
    public int bitrate;

    /** The codec id to be used, the default value is [default]. Settings only take effect before publishing stream */
    public ZegoVideoCodecID codecID;

    /**
     * Create video configuration with preset enumeration values
     */
    public ZegoVideoConfig(ZegoVideoConfigPreset preset){
        this.codecID = ZegoVideoCodecID.DEFAULT;
        switch (preset) {
            case PRESET_180P:
                this.captureWidth = 180;
                this.captureHeight = 320;
                this.encodeWidth = 180;
                this.encodeHeight = 320;
                this.bitrate = 300;
                this.fps = 15;
                break;
            case PRESET_270P:
                this.captureWidth = 270;
                this.captureHeight = 480;
                this.encodeWidth = 270;
                this.encodeHeight = 480;
                this.bitrate = 400;
                this.fps = 15;
                break;
            case PRESET_360P:
                this.captureWidth = 360;
                this.captureHeight = 640;
                this.encodeWidth = 360;
                this.encodeHeight = 640;
                this.bitrate = 600;
                this.fps = 15;
                break;
            case PRESET_540P:
                this.captureWidth = 540;
                this.captureHeight = 960;
                this.encodeWidth = 540;
                this.encodeHeight = 960;
                this.bitrate = 1200;
                this.fps = 15;
                break;
            case PRESET_720P:
                this.captureWidth = 720;
                this.captureHeight = 1280;
                this.encodeWidth = 720;
                this.encodeHeight = 1280;
                this.bitrate = 1500;
                this.fps = 15;
                break;
            case PRESET_1080P:
                this.captureWidth = 1080;
                this.captureHeight = 1920;
                this.encodeWidth = 1080;
                this.encodeHeight = 1920;
                this.bitrate = 3000;
                this.fps = 15;
                break;
        }
    }

    /**
     * Create default video configuration(360p, 15fps, 600000bps)
     *
     * 360p, 15fps, 600kbps
     */
    public ZegoVideoConfig(){
        this(ZegoVideoConfigPreset.PRESET_360P);
    }

    /**
     * Set the width and height of the capture resolution to control the resolution of the camera image acquisition. Only the camera is not started and the custom video capture is not used, the setting is effective
     */
    public void setCaptureResolution(int width, int height){
        this.captureWidth = width;
        this.captureHeight = height;
    }

    /**
     * Set the width and height of the encoding resolution to control the image resolution of the encoder to publish stream. The settings before and after publishing stream can be effective
     */
    public void setEncodeResolution(int width, int height){
        this.encodeWidth = width;
        this.encodeHeight = height;
    }

    /**
     * Set the video frame rate, used to control the frame rate of the camera acquisition and the size of the encoder frame rate. Only the camera is not started, the setting is effective
     */
    public void setVideoFPS(int fps){
        this.fps = fps;
    }

    /**
     * Set bit rate of the encode resolution, unit is kbps. The settings before and after publishing stream can be effective
     */
    public void setVideoBitrate(int bitrate){
        this.bitrate = bitrate;
    }

    /**
     * The codec id to be used, the default value is [default]. Settings only take effect before publishing stream
     */
    public void setCodecID(ZegoVideoCodecID codecID){
        this.codecID = codecID;
    }

    /**
     * @deprecated This interface is deprecated, please use the overloaded interface with the same name
     */
    public ZegoVideoConfig(ZegoResolution resolution){
        this.codecID = ZegoVideoCodecID.DEFAULT;
        switch (resolution) {
            case RESOLUTION_180x320:
                this.captureWidth = 180;
                this.captureHeight = 320;
                this.encodeWidth = 180;
                this.encodeHeight = 320;
                this.bitrate = 300;
                this.fps = 15;
                break;
            case RESOLUTION_270x480:
                this.captureWidth = 270;
                this.captureHeight = 480;
                this.encodeWidth = 270;
                this.encodeHeight = 480;
                this.bitrate = 400;
                this.fps = 15;
                break;
            case RESOLUTION_360x640:
                this.captureWidth = 360;
                this.captureHeight = 640;
                this.encodeWidth = 360;
                this.encodeHeight = 640;
                this.bitrate = 600;
                this.fps = 15;
                break;
            case RESOLUTION_540x960:
                this.captureWidth = 540;
                this.captureHeight = 960;
                this.encodeWidth = 540;
                this.encodeHeight = 960;
                this.bitrate = 1200;
                this.fps = 15;
                break;
            case RESOLUTION_720x1280:
                this.captureWidth = 720;
                this.captureHeight = 1280;
                this.encodeWidth = 720;
                this.encodeHeight = 1280;
                this.bitrate = 1500;
                this.fps = 15;
                break;
            case RESOLUTION_1080x1920:
                this.captureWidth = 1080;
                this.captureHeight = 1920;
                this.encodeWidth = 1080;
                this.encodeHeight = 1920;
                this.bitrate = 3000;
                this.fps = 15;
                break;
        }
    }

}
