
package de.flapdoodle.embed.process.config;

import de.flapdoodle.net.ProxyFactory;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Immutable implementation of {@link DownloadConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableDownloadConfig.builder()}.
 */
@SuppressWarnings({"all", "deprecation", "removal"})
public final class ImmutableDownloadConfig implements DownloadConfig {
  private final ProxyFactory proxyFactory;
  private final String userAgent;
  private final String authorization;
  private final TimeoutConfig timeoutConfig;

  private ImmutableDownloadConfig(ImmutableDownloadConfig.Builder builder) {
    this.proxyFactory = builder.proxyFactory;
    this.authorization = builder.authorization;
    if (builder.userAgent != null) {
      initShim.userAgent(builder.userAgent);
    }
    if (builder.timeoutConfig != null) {
      initShim.timeoutConfig(builder.timeoutConfig);
    }
    this.userAgent = initShim.getUserAgent();
    this.timeoutConfig = initShim.getTimeoutConfig();
    this.initShim = null;
  }

  private ImmutableDownloadConfig(
      ProxyFactory proxyFactory,
      String userAgent,
      String authorization,
      TimeoutConfig timeoutConfig) {
    this.proxyFactory = proxyFactory;
    this.userAgent = userAgent;
    this.authorization = authorization;
    this.timeoutConfig = timeoutConfig;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte userAgentBuildStage = STAGE_UNINITIALIZED;
    private String userAgent;

    String getUserAgent() {
      if (userAgentBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (userAgentBuildStage == STAGE_UNINITIALIZED) {
        userAgentBuildStage = STAGE_INITIALIZING;
        this.userAgent = Objects.requireNonNull(getUserAgentInitialize(), "userAgent");
        userAgentBuildStage = STAGE_INITIALIZED;
      }
      return this.userAgent;
    }

    void userAgent(String userAgent) {
      this.userAgent = userAgent;
      userAgentBuildStage = STAGE_INITIALIZED;
    }

    private byte timeoutConfigBuildStage = STAGE_UNINITIALIZED;
    private TimeoutConfig timeoutConfig;

    TimeoutConfig getTimeoutConfig() {
      if (timeoutConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (timeoutConfigBuildStage == STAGE_UNINITIALIZED) {
        timeoutConfigBuildStage = STAGE_INITIALIZING;
        this.timeoutConfig = Objects.requireNonNull(getTimeoutConfigInitialize(), "timeoutConfig");
        timeoutConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.timeoutConfig;
    }

    void timeoutConfig(TimeoutConfig timeoutConfig) {
      this.timeoutConfig = timeoutConfig;
      timeoutConfigBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (userAgentBuildStage == STAGE_INITIALIZING) attributes.add("userAgent");
      if (timeoutConfigBuildStage == STAGE_INITIALIZING) attributes.add("timeoutConfig");
      return "Cannot build DownloadConfig, attribute initializers form cycle " + attributes;
    }
  }

  private String getUserAgentInitialize() {
    return DownloadConfig.super.getUserAgent();
  }

  private TimeoutConfig getTimeoutConfigInitialize() {
    return DownloadConfig.super.getTimeoutConfig();
  }

  /**
   * @return The value of the {@code proxyFactory} attribute
   */
  @Override
  public Optional<ProxyFactory> proxyFactory() {
    return Optional.ofNullable(proxyFactory);
  }

  /**
   * @return The value of the {@code userAgent} attribute
   */
  @Override
  public String getUserAgent() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.getUserAgent()
        : this.userAgent;
  }

  /**
   * @return The value of the {@code authorization} attribute
   */
  @Deprecated
  @Override
  public Optional<String> getAuthorization() {
    return Optional.ofNullable(authorization);
  }

  /**
   * @return The value of the {@code timeoutConfig} attribute
   */
  @Override
  public TimeoutConfig getTimeoutConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.getTimeoutConfig()
        : this.timeoutConfig;
  }

  /**
   * Copy the current immutable object by setting a <em>present</em> value for the optional {@link DownloadConfig#proxyFactory() proxyFactory} attribute.
   * @param value The value for proxyFactory
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableDownloadConfig withProxyFactory(ProxyFactory value) {
    ProxyFactory newValue = Objects.requireNonNull(value, "proxyFactory");
    if (this.proxyFactory == newValue) return this;
    return new ImmutableDownloadConfig(newValue, this.userAgent, this.authorization, this.timeoutConfig);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link DownloadConfig#proxyFactory() proxyFactory} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional An optional value for proxyFactory
   * @return A modified copy or {@code this} if not changed
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final ImmutableDownloadConfig withProxyFactory(Optional<? extends ProxyFactory> optional) {
    ProxyFactory value = optional.orElse(null);
    if (this.proxyFactory == value) return this;
    return new ImmutableDownloadConfig(value, this.userAgent, this.authorization, this.timeoutConfig);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadConfig#getUserAgent() userAgent} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for userAgent
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableDownloadConfig withUserAgent(String value) {
    String newValue = Objects.requireNonNull(value, "userAgent");
    if (this.userAgent.equals(newValue)) return this;
    return new ImmutableDownloadConfig(this.proxyFactory, newValue, this.authorization, this.timeoutConfig);
  }

  /**
   * Copy the current immutable object by setting a <em>present</em> value for the optional {@link DownloadConfig#getAuthorization() authorization} attribute.
   * @param value The value for authorization
   * @return A modified copy or {@code this} if not changed
   */
  @Deprecated
  public final ImmutableDownloadConfig withAuthorization(String value) {
    String newValue = Objects.requireNonNull(value, "authorization");
    if (Objects.equals(this.authorization, newValue)) return this;
    return new ImmutableDownloadConfig(this.proxyFactory, this.userAgent, newValue, this.timeoutConfig);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link DownloadConfig#getAuthorization() authorization} attribute.
   * An equality check is used on inner value to prevent copying of the same value by returning {@code this}.
   * @param optional An optional value for authorization
   * @return A modified copy or {@code this} if not changed
   */
  @Deprecated
  public final ImmutableDownloadConfig withAuthorization(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.authorization, value)) return this;
    return new ImmutableDownloadConfig(this.proxyFactory, this.userAgent, value, this.timeoutConfig);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadConfig#getTimeoutConfig() timeoutConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for timeoutConfig
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableDownloadConfig withTimeoutConfig(TimeoutConfig value) {
    if (this.timeoutConfig == value) return this;
    TimeoutConfig newValue = Objects.requireNonNull(value, "timeoutConfig");
    return new ImmutableDownloadConfig(this.proxyFactory, this.userAgent, this.authorization, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableDownloadConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableDownloadConfig
        && equalsByValue((ImmutableDownloadConfig) another);
  }

  private boolean equalsByValue(ImmutableDownloadConfig another) {
    return Objects.equals(proxyFactory, another.proxyFactory)
        && userAgent.equals(another.userAgent)
        && Objects.equals(authorization, another.authorization)
        && timeoutConfig.equals(another.timeoutConfig);
  }

  /**
   * Computes a hash code from attributes: {@code proxyFactory}, {@code userAgent}, {@code authorization}, {@code timeoutConfig}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(proxyFactory);
    h += (h << 5) + userAgent.hashCode();
    h += (h << 5) + Objects.hashCode(authorization);
    h += (h << 5) + timeoutConfig.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code DownloadConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("DownloadConfig{");
    if (proxyFactory != null) {
      builder.append("proxyFactory=").append(proxyFactory);
    }
    if (builder.length() > 15) builder.append(", ");
    builder.append("userAgent=").append(userAgent);
    if (authorization != null) {
      builder.append(", ");
      builder.append("authorization=").append(authorization);
    }
    builder.append(", ");
    builder.append("timeoutConfig=").append(timeoutConfig);
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link DownloadConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable DownloadConfig instance
   */
  public static ImmutableDownloadConfig copyOf(DownloadConfig instance) {
    if (instance instanceof ImmutableDownloadConfig) {
      return (ImmutableDownloadConfig) instance;
    }
    return ImmutableDownloadConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableDownloadConfig ImmutableDownloadConfig}.
   * <pre>
   * ImmutableDownloadConfig.builder()
   *    .proxyFactory(Optional&lt;de.flapdoodle.net.ProxyFactory&gt;) // optional {@link DownloadConfig#proxyFactory() proxyFactory}
   *    .userAgent(String) // optional {@link DownloadConfig#getUserAgent() userAgent}
   *    .authorization(Optional&lt;String&gt;) // optional {@link DownloadConfig#getAuthorization() authorization}
   *    .timeoutConfig(de.flapdoodle.embed.process.config.TimeoutConfig) // optional {@link DownloadConfig#getTimeoutConfig() timeoutConfig}
   *    .build();
   * </pre>
   * @return A new ImmutableDownloadConfig builder
   */
  public static ImmutableDownloadConfig.Builder builder() {
    return new ImmutableDownloadConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableDownloadConfig ImmutableDownloadConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private ProxyFactory proxyFactory;
    private String userAgent;
    private String authorization;
    private TimeoutConfig timeoutConfig;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code DownloadConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(DownloadConfig instance) {
      Objects.requireNonNull(instance, "instance");
      Optional<ProxyFactory> proxyFactoryOptional = instance.proxyFactory();
      if (proxyFactoryOptional.isPresent()) {
        proxyFactory(proxyFactoryOptional);
      }
      this.userAgent(instance.getUserAgent());
      Optional<String> authorizationOptional = instance.getAuthorization();
      if (authorizationOptional.isPresent()) {
        authorization(authorizationOptional);
      }
      this.timeoutConfig(instance.getTimeoutConfig());
      return this;
    }

    /**
     * Initializes the optional value {@link DownloadConfig#proxyFactory() proxyFactory} to proxyFactory.
     * @param proxyFactory The value for proxyFactory
     * @return {@code this} builder for chained invocation
     */
    public final Builder proxyFactory(ProxyFactory proxyFactory) {
      this.proxyFactory = Objects.requireNonNull(proxyFactory, "proxyFactory");
      return this;
    }

    /**
     * Initializes the optional value {@link DownloadConfig#proxyFactory() proxyFactory} to proxyFactory.
     * @param proxyFactory The value for proxyFactory
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder proxyFactory(Optional<? extends ProxyFactory> proxyFactory) {
      this.proxyFactory = proxyFactory.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadConfig#getUserAgent() userAgent} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadConfig#getUserAgent() userAgent}.</em>
     * @param userAgent The value for userAgent 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder userAgent(String userAgent) {
      this.userAgent = Objects.requireNonNull(userAgent, "userAgent");
      return this;
    }

    /**
     * Initializes the optional value {@link DownloadConfig#getAuthorization() authorization} to authorization.
     * @param authorization The value for authorization
     * @return {@code this} builder for chained invocation
     */
    @Deprecated
    public final Builder authorization(String authorization) {
      this.authorization = Objects.requireNonNull(authorization, "authorization");
      return this;
    }

    /**
     * Initializes the optional value {@link DownloadConfig#getAuthorization() authorization} to authorization.
     * @param authorization The value for authorization
     * @return {@code this} builder for use in a chained invocation
     */
    @Deprecated
    public final Builder authorization(Optional<String> authorization) {
      this.authorization = authorization.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadConfig#getTimeoutConfig() timeoutConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadConfig#getTimeoutConfig() timeoutConfig}.</em>
     * @param timeoutConfig The value for timeoutConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder timeoutConfig(TimeoutConfig timeoutConfig) {
      this.timeoutConfig = Objects.requireNonNull(timeoutConfig, "timeoutConfig");
      return this;
    }

    /**
     * Builds a new {@link ImmutableDownloadConfig ImmutableDownloadConfig}.
     * @return An immutable instance of DownloadConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableDownloadConfig build() {
      return new ImmutableDownloadConfig(this);
    }
  }
}
