package de.flapdoodle.embed.mongo.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link UsernamePassword}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableUsernamePassword.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableUsernamePassword.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableUsernamePassword implements UsernamePassword {
  private final String name;
  private final char[] password;

  private ImmutableUsernamePassword(String name, char[] password) {
    this.name = Objects.requireNonNull(name, "name");
    this.password = password.clone();
  }

  private ImmutableUsernamePassword(ImmutableUsernamePassword original, String name, char[] password) {
    this.name = name;
    this.password = password;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @Override
  public String name() {
    return name;
  }

  /**
   * @return A cloned {@code password} array
   */
  @Override
  public char[] password() {
    return password.clone();
  }

  /**
   * Copy the current immutable object by setting a value for the {@link UsernamePassword#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableUsernamePassword withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableUsernamePassword(this, newValue, this.password);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link UsernamePassword#password() password}.
   * The array is cloned before being saved as attribute values.
   * @param elements The non-null elements for password
   * @return A modified copy of {@code this} object
   */
  public final ImmutableUsernamePassword withPassword(char... elements) {
    char[] newValue = elements.clone();
    return new ImmutableUsernamePassword(this, this.name, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableUsernamePassword} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableUsernamePassword
        && equalTo(0, (ImmutableUsernamePassword) another);
  }

  private boolean equalTo(int synthetic, ImmutableUsernamePassword another) {
    return name.equals(another.name)
        && Arrays.equals(password, another.password);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code password}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + name.hashCode();
    h += (h << 5) + Arrays.hashCode(password);
    return h;
  }

  /**
   * Prints the immutable value {@code UsernamePassword} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "UsernamePassword{"
        + "name=" + name
        + ", password=" + Arrays.toString(password)
        + "}";
  }

  /**
   * Construct a new immutable {@code UsernamePassword} instance.
   * @param name The value for the {@code name} attribute
   * @param password The value for the {@code password} attribute
   * @return An immutable UsernamePassword instance
   */
  public static ImmutableUsernamePassword of(String name, char[] password) {
    return new ImmutableUsernamePassword(name, password);
  }

  /**
   * Creates an immutable copy of a {@link UsernamePassword} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable UsernamePassword instance
   */
  public static ImmutableUsernamePassword copyOf(UsernamePassword instance) {
    if (instance instanceof ImmutableUsernamePassword) {
      return (ImmutableUsernamePassword) instance;
    }
    return ImmutableUsernamePassword.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableUsernamePassword ImmutableUsernamePassword}.
   * <pre>
   * ImmutableUsernamePassword.builder()
   *    .name(String) // required {@link UsernamePassword#name() name}
   *    .password(char) // required {@link UsernamePassword#password() password}
   *    .build();
   * </pre>
   * @return A new ImmutableUsernamePassword builder
   */
  public static ImmutableUsernamePassword.Builder builder() {
    return new ImmutableUsernamePassword.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableUsernamePassword ImmutableUsernamePassword}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_NAME = 0x1L;
    private static final long INIT_BIT_PASSWORD = 0x2L;
    private long initBits = 0x3L;

    private String name;
    private char[] password;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code UsernamePassword} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(UsernamePassword instance) {
      Objects.requireNonNull(instance, "instance");
      this.name(instance.name());
      this.password(instance.password());
      return this;
    }

    /**
     * Initializes the value for the {@link UsernamePassword#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Initializes the value for the {@link UsernamePassword#password() password} attribute.
     * @param password The elements for password
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder password(char... password) {
      this.password = password.clone();
      initBits &= ~INIT_BIT_PASSWORD;
      return this;
    }

    /**
     * Builds a new {@link ImmutableUsernamePassword ImmutableUsernamePassword}.
     * @return An immutable instance of UsernamePassword
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableUsernamePassword build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableUsernamePassword(null, name, password);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      if ((initBits & INIT_BIT_PASSWORD) != 0) attributes.add("password");
      return "Cannot build UsernamePassword, some of required attributes are not set " + attributes;
    }
  }
}
