/*
 * Copyright 2014-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.codecentric.boot.admin.server.domain.values;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import lombok.ToString;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

/**
 * Registration info for the instance registers with (including metadata)
 *
 * @author Johannes Edmeier
 */
@lombok.Data
@ToString(exclude = "metadata")
public final class Registration implements Serializable {

	private static final Logger log = LoggerFactory.getLogger(Registration.class);

	private final String name;

	/**
	 * Base URL of the Actuator (management) endpoints. May run on a different port or
	 * context than the service itself. Must be an absolute URL when present. Example:
	 * <code>https://example.com/actuator</code>
	 */
	@Nullable private final String managementUrl;

	/**
	 * Absolute URL of the Actuator health endpoint. Required and used by Spring Boot
	 * Admin to determine the instance status. Example:
	 * <code>https://example.com/actuator/health</code>
	 */
	private final String healthUrl;

	/**
	 * Public base URL of the business application (not the Actuator base). Used by Spring
	 * Boot Admin to link to the running app (e.g., "Open application"). Must be an
	 * absolute URL; can be overridden via metadata keys "service-url" and "service-path".
	 */
	@Nullable private final String serviceUrl;

	private final String source;

	private final Map<String, String> metadata;

	@lombok.Builder(builderClassName = "Builder", toBuilder = true)
	private Registration(String name, @Nullable String managementUrl, String healthUrl, @Nullable String serviceUrl,
			String source, @lombok.Singular("metadata") Map<String, String> metadata) {
		Assert.hasText(name, "'name' must not be empty.");
		Assert.hasText(healthUrl, "'healthUrl' must not be empty.");
		Assert.isTrue(checkUrl(healthUrl), "'healthUrl' is not valid: " + healthUrl);
		Assert.isTrue(!StringUtils.hasText(managementUrl) || checkUrl(managementUrl),
				"'managementUrl' is not valid: " + managementUrl);
		Assert.isTrue(!StringUtils.hasText(serviceUrl) || checkUrl(serviceUrl),
				"'serviceUrl' is not valid: " + serviceUrl);

		this.name = name;
		this.managementUrl = managementUrl;
		this.healthUrl = healthUrl;
		this.serviceUrl = this.getServiceUrl(serviceUrl, metadata);
		this.source = source;
		this.metadata = new LinkedHashMap<>();
		for (Map.Entry<String, String> entry : metadata.entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();
			this.metadata.put(key, value);
		}
	}

	public static Registration.Builder create(String name, String healthUrl) {
		return builder().name(name).healthUrl(healthUrl);
	}

	public static Registration.Builder copyOf(Registration registration) {
		return registration.toBuilder();
	}

	/**
	 * Determines the service url. It might be overriden by metadata entries to override
	 * the service url.
	 * @param serviceUrl original serviceUrl
	 * @param metadata metadata information of registered instance
	 * @return the actual service url
	 */
	@Nullable private String getServiceUrl(@Nullable String serviceUrl, Map<String, String> metadata) {
		if (serviceUrl == null) {
			return null;
		}
		String url = metadata.getOrDefault("service-url", serviceUrl);

		try {
			URI baseUri = new URI(url);
			return baseUri.toString();
		}
		catch (URISyntaxException ex) {
			log.warn("Invalid service url: " + serviceUrl, ex);
		}

		return serviceUrl;
	}

	public Map<String, String> getMetadata() {
		return Collections.unmodifiableMap(this.metadata);
	}

	/**
	 * Checks the syntax of the given URL.
	 * @param url the URL.
	 * @return true, if valid.
	 */
	private boolean checkUrl(String url) {
		try {
			URI uri = new URI(url);
			return uri.isAbsolute();
		}
		catch (URISyntaxException ex) {
			return false;
		}
	}

	public static class Builder {

		// Will be generated by lombok

	}

}
