package com.zoyi.channel.plugin.android.view.video_player;


import android.content.Context;
import android.net.Uri;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.ProgressBar;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.store.GlobalStore;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.com.google.android.exoplayer2.*;
import com.zoyi.com.google.android.exoplayer2.source.ExtractorMediaSource;
import com.zoyi.com.google.android.exoplayer2.trackselection.AdaptiveTrackSelection;
import com.zoyi.com.google.android.exoplayer2.trackselection.DefaultTrackSelector;
import com.zoyi.com.google.android.exoplayer2.ui.AspectRatioFrameLayout;
import com.zoyi.com.google.android.exoplayer2.ui.PlayerView;
import com.zoyi.com.google.android.exoplayer2.upstream.DefaultHttpDataSourceFactory;
import com.zoyi.com.google.android.exoplayer2.util.Util;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.concurrent.TimeUnit;

public abstract class AbsPopUpVideoPlayerView extends AbsVideoPlayerView {

  public AbsPopUpVideoPlayerView(@NonNull Context context) {
    super(context);

    init(context);
  }

  public AbsPopUpVideoPlayerView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context);
  }

  public AbsPopUpVideoPlayerView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context);
  }

  private Context context;

  protected PlayerView playerAttachment;

  @Nullable
  protected SimpleExoPlayer exoPlayer;

  @Nullable
  private ProgressBar progressBar;

  @Nullable
  private Subscription videoProgressSubscription;

  private void init(Context context) {
    this.context = context;

    LayoutInflater.from(context).inflate(getLayoutId(), this);

    playerAttachment = findViewById(R.id.ch_playerAttachment);
  }

  public AbsPopUpVideoPlayerView setUrl(
      @Nullable String messageId,
      File attachment
  ) {
    setId(String.format("%s:%s", messageId, attachment.getId()));

    progressBar = findViewById(R.id.ch_popupPlayerProgress);

    AdaptiveTrackSelection.Factory videoTrackSelectionFactory = new AdaptiveTrackSelection.Factory();
    DefaultTrackSelector trackSelector = new DefaultTrackSelector(videoTrackSelectionFactory);

    exoPlayer = ExoPlayerFactory.newSimpleInstance(context, new DefaultRenderersFactory(context), trackSelector);
    exoPlayer.addListener(new Player.EventListener() {
      @Override
      public void onPlayerStateChanged(boolean playWhenReady, int playbackState) {
        changeVideoState(playWhenReady);

        if (playWhenReady) {
          bindVideoProgress();
        } else {
          unbindVideoProgress();
        }

        switch (playbackState) {
          case Player.STATE_ENDED:
            if (exoPlayer != null) {
              exoPlayer.seekTo(0);
              exoPlayer.setPlayWhenReady(false);
            }
            break;
        }
      }
    });

    playerAttachment.setPlayer(exoPlayer);
    playerAttachment.setResizeMode(AspectRatioFrameLayout.RESIZE_MODE_ZOOM);

    DefaultHttpDataSourceFactory dataSourceFactory = new DefaultHttpDataSourceFactory(
        Util.getUserAgent(context, "ChannelTalkSDK")
    );
    dataSourceFactory.getDefaultRequestProperties().set(Const.X_SESSION, GlobalStore.get().jwt.get());

    ExtractorMediaSource videoSource = new ExtractorMediaSource.Factory(dataSourceFactory)
        .createMediaSource(Uri.parse(attachment.getUrl()));

    exoPlayer.prepare(videoSource);
    exoPlayer.setPlayWhenReady(true);
    exoPlayer.seekTo(getLatestSeconds() * TimeUnit.SECONDS.toMillis(1));

    if (exoPlayer != null) {
      exoPlayer.setVolume(0);
    }
    return this;
  }

  @Override
  public void setOnClickListener(@Nullable OnClickListener l) {
    playerAttachment.setOnClickListener(l);
  }

  @Override
  void onVideoPaused() {
    if (exoPlayer != null) {
      exoPlayer.setPlayWhenReady(false);
    }
  }

  // handle progress

  private void bindVideoProgress() {
    if (videoProgressSubscription == null || videoProgressSubscription.isUnsubscribed()) {
      unbindVideoProgress();

      videoProgressSubscription = Observable.interval(1000, TimeUnit.MILLISECONDS)
          .onBackpressureLatest()
          .subscribeOn(Schedulers.computation())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(aLong -> Optional.ofNullable(exoPlayer)
              .map(SimpleExoPlayer::getCurrentPosition)
              .ifPresent(position -> {
                long seconds = position / TimeUnit.SECONDS.toMillis(1);
                changeVideoPosition(seconds);

                onDurationChange(position);
              }));
    }
  }

  private void onDurationChange(long milliseconds) {
    Long totalMs = exoPlayer != null ? exoPlayer.getDuration() : null;

    if (progressBar != null && milliseconds >= 0 && totalMs != null && totalMs > 0) {
      progressBar.setVisibility(View.VISIBLE);

      if (milliseconds <= totalMs) {
        progressBar.setProgress(Math.max(0, (int) (milliseconds * 100 / totalMs)));
      } else {
        progressBar.setProgress(100);
      }
    }
  }

  private void unbindVideoProgress() {
    if (progressBar != null) {
      progressBar.setVisibility(View.GONE);
    }

    dispose(videoProgressSubscription);
  }

  // private functions

  protected void dispose(@Nullable Subscription subscription) {
    if (subscription != null && !subscription.isUnsubscribed()) {
      subscription.unsubscribe();
    }
  }

  // clear

  @Override
  public void clear() {
    super.clear();

    dispose(videoProgressSubscription);

    videoProgressSubscription = null;

    if (exoPlayer != null) {
      exoPlayer.release();
      exoPlayer = null;
    }
  }

  protected abstract int getLayoutId();
}
