package com.zoyi.channel.plugin.android.view.layout;

import android.content.Context;
import android.content.res.TypedArray;
import androidx.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;

// https://stackoverflow.com/questions/2961777/android-linearlayout-horizontal-with-wrapping-children

public class ChEndFlowLayout extends ViewGroup {

  public int horizontalSpacing;
  public int verticalSpacing;

  public ChEndFlowLayout(Context context) {
    super(context);

    init(context, null);
  }

  public ChEndFlowLayout(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context, attrs);
  }

  public ChEndFlowLayout(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context, attrs);
  }

  private void init(Context context, @Nullable AttributeSet attrs) {
    TypedArray typedArray = context.getTheme().obtainStyledAttributes(attrs, R.styleable.ChEndFlowLayout, 0, 0);
    try {
      horizontalSpacing = (int) typedArray.getDimension(R.styleable.ChEndFlowLayout_ch_fl_horizontalSpacing, 0f);
      verticalSpacing = (int) typedArray.getDimension(R.styleable.ChEndFlowLayout_ch_fl_verticalSpacing, 0f);
    } finally {
      typedArray.recycle();
    }
  }

  @Override
  protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
    final int width = MeasureSpec.getSize(widthMeasureSpec) - getPaddingLeft() - getPaddingRight();
    final int count = getChildCount();

    int height = MeasureSpec.getSize(heightMeasureSpec) - getPaddingTop() - getPaddingBottom();
    int lineHeight = 0;

    int leftX = getPaddingLeft();
    int topY = 0;

    int childHeightMeasureSpec;
    if (MeasureSpec.getMode(heightMeasureSpec) == MeasureSpec.AT_MOST) {
      childHeightMeasureSpec = MeasureSpec.makeMeasureSpec(height, MeasureSpec.AT_MOST);
    } else {
      childHeightMeasureSpec = MeasureSpec.makeMeasureSpec(0, MeasureSpec.UNSPECIFIED);
    }

    // do measuring gravity: start
    for (int i = 0; i < count; i++) {
      final View child = getChildAt(i);
      if (child.getVisibility() != GONE) {
        child.measure(MeasureSpec.makeMeasureSpec(width, MeasureSpec.AT_MOST), childHeightMeasureSpec);
        final int childw = Math.min(width, child.getMeasuredWidth());

        // if child width overflows empty area, append to new line
        if (leftX + childw > width) {
          leftX = childw + horizontalSpacing;
          topY += (lineHeight + verticalSpacing);

          lineHeight = child.getMeasuredHeight();
        }
        // else, just add this line
        else {
          leftX += childw + horizontalSpacing;
          lineHeight = Math.max(lineHeight, child.getMeasuredHeight());
        }
      }
    }

    setMeasuredDimension(
        width + getPaddingLeft() + getPaddingRight(),
        topY + lineHeight + getPaddingTop() + getPaddingBottom()
    );
  }

  // return current line's max height
  private int appendChild(int startIndex, int endIndex, int rightX, int topY) {
    int rightPosition = rightX;
    int lineHeight = 0;

    for (int i = endIndex; i >= startIndex; i--) {
      final View child = getChildAt(i);

      if (child.getVisibility() != GONE) {
        child.layout(
            rightPosition - child.getMeasuredWidth(),
            topY,
            rightPosition,
            topY + child.getMeasuredHeight()
        );

        rightPosition -= (child.getMeasuredWidth() + horizontalSpacing);
        lineHeight = Math.max(lineHeight, child.getMeasuredHeight());
      }
    }

    return lineHeight;
  }

  @Override
  protected void onLayout(boolean changed, int l, int t, int r, int b) {
    final int count = getChildCount();
    final int width = r - l;
    final int availableWidth = r - l - getPaddingLeft() - getPaddingRight();

    int leftX = 0;
    int topY = getPaddingTop();
    int lastQueuedIndex = 0;

    for (int i = 0; i <= count; i++) {
      if (i == count) {
        appendChild(lastQueuedIndex, i - 1, width - getPaddingRight(), topY);
      } else {
        final View child = getChildAt(i);

        if (child.getVisibility() != GONE) {
          final int childWidth = Math.min(child.getMeasuredWidth(), availableWidth);

          if (leftX + childWidth > availableWidth) {
            leftX = childWidth + horizontalSpacing;
            topY += appendChild(lastQueuedIndex, i - 1, width - getPaddingRight(), topY) + verticalSpacing;
            lastQueuedIndex = i;
          } else {
            leftX += (childWidth + horizontalSpacing);
          }
        }
      }
    }
  }
}
