package com.zoyi.channel.plugin.android.view.layout;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.os.Build;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.cardview.widget.CardView;

import android.util.AttributeSet;
import android.view.*;
import android.widget.FrameLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.ListUtils;

import java.util.List;

public class ChBorderLayout extends FrameLayout {

  public ChBorderLayout(@NonNull Context context) {
    super(context);
    init(context, null);
  }

  public ChBorderLayout(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context, attrs);
  }

  public ChBorderLayout(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context, attrs);
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP)
  public ChBorderLayout(Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes) {
    super(context, attrs, defStyleAttr, defStyleRes);
    init(context, attrs);
  }

  private List<Integer> componentIds = ListUtils.newArrayList(
      R.id.ch_rootBorderLayout,
      R.id.ch_cardBorderLayoutBack,
      R.id.ch_cardBorderLayoutContent
  );

  private CardView cardBack;
  private CardView cardContent;

  private Float borderRadius = 0f;
  private int borderWidth = 0;
  private int borderColor = Color.TRANSPARENT;
  private int backColor = Color.WHITE;

  private void init(@NonNull Context context, @Nullable AttributeSet attrs) {
    LayoutInflater.from(context).inflate(R.layout.ch_layout_border, this);

    cardBack = findViewById(R.id.ch_cardBorderLayoutBack);
    cardContent = findViewById(R.id.ch_cardBorderLayoutContent);

    TypedArray typedArray = context.getTheme().obtainStyledAttributes(attrs, R.styleable.ChBorderLayout, 0, 0);
    try {
      borderRadius = typedArray.getDimension(R.styleable.ChBorderLayout_ch_bl_borderRadius, borderRadius);
      borderWidth = (int) typedArray.getDimension(R.styleable.ChBorderLayout_ch_bl_borderWidth, borderWidth);
      borderColor = typedArray.getColor(R.styleable.ChBorderLayout_ch_bl_borderColor, borderColor);
      backColor = typedArray.getColor(R.styleable.ChBorderLayout_ch_bl_backColor, backColor);
    } finally {
      typedArray.recycle();
    }

    resolveStyle();
  }

  public void setBorderRadius(Float borderRadius) {
    this.borderRadius = borderRadius;

    resolveStyle();
  }

  public void setBorderWidth(float borderWidth) {
    this.borderWidth = (int) borderWidth;

    resolveStyle();
  }

  public void setBorderColor(int borderColor) {
    this.borderColor = borderColor;

    resolveStyle();
  }

  public void setBackColor(int backColor) {
    this.backColor = backColor;

    resolveStyle();
  }

  private void resolveStyle() {
    if (cardBack != null) {
      cardBack.setRadius(borderRadius);
      cardBack.setCardBackgroundColor(borderColor);
    }

    if (cardContent != null) {
      ViewGroup.LayoutParams params = cardContent.getLayoutParams();
      if (params instanceof MarginLayoutParams) {
        ((MarginLayoutParams) params).setMargins(borderWidth, borderWidth, borderWidth, borderWidth);
      }

      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
        cardContent.setRadius(Math.max(0, borderRadius - borderWidth));
      } else {
        cardContent.setRadius(Math.max(1, borderRadius - borderWidth));
      }

      cardContent.setCardBackgroundColor(backColor);
    }
  }

  @Override
  public void addView(View child, int index, ViewGroup.LayoutParams params) {
    if (child != null && componentIds.contains(child.getId())) {
      super.addView(child, index, params);
    } else if (cardContent != null && child != null) {
      cardContent.addView(child, index, params);
    }
  }
}
