package com.zoyi.channel.plugin.android.activity.chat2;

import android.app.Activity;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.action.ChatAction;
import com.zoyi.channel.plugin.android.activity.chat.binder.ChatHeaderBinder;
import com.zoyi.channel.plugin.android.activity.chat.binder.ChatInteractionBinder;
import com.zoyi.channel.plugin.android.activity.chat.contract.ChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.chat.enumerate.ChatState;
import com.zoyi.channel.plugin.android.activity.chat.listener.chatmanager.OnMessageSendStateChangeListener;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManagerInterface;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.base.AbstractAdapterPresenter;
import com.zoyi.channel.plugin.android.bind.BindAction;
import com.zoyi.channel.plugin.android.enumerate.*;
import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.model.repo.MessagesRepo;
import com.zoyi.channel.plugin.android.model.repo.UserChatRepo;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.source.photopicker.PhotoItem;
import com.zoyi.channel.plugin.android.open.listener.ChannelPluginListener;
import com.zoyi.channel.plugin.android.selector.*;
import com.zoyi.channel.plugin.android.store.GlobalStore;
import com.zoyi.channel.plugin.android.store.PluginStore;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.com.annimon.stream.Stream;
import com.zoyi.com.annimon.stream.function.IndexedFunction;

import java.util.*;

import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_CLOSE;
import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_SOLVE;

public class ChatPresenter2
    extends AbstractAdapterPresenter<ChatContract2.View, ChatAdapterContract.View, ChatAdapterContract.Model>
    implements ChatContract2.Presenter, OnMessageSendStateChangeListener {

  @Nullable
  private String chatId;

  @Nullable
  private String presetMessage;

  @Nullable
  private Session lastReadSession;

  @Nullable
  private String prev;

  private ChatState chatState = ChatState.DISCONNECTED;

  // queue

  private ArrayList<SendItem> preCreateItems = new ArrayList<>();
  private ArrayList<SendItem> preCreateFailedItems = new ArrayList<>();

  @Nullable
  private ArrayList<Message> tempQueue;

  // binders

  @Nullable
  private ChatHeaderBinder chatHeaderBinder;

  @Nullable
  private ChatInteractionBinder chatInteractionBinder;

  // last message for set input state
  @Nullable
  private Message lastMessage;

  public ChatPresenter2(
      ChatContract2.View view,
      ChatAdapterContract.View adapterView,
      ChatAdapterContract.Model adapterModel,
      @Nullable String chatId,
      @Nullable String presetMessage
  ) {
    super(view, adapterView, adapterModel);

    this.chatId = chatId;
    this.presetMessage = presetMessage;
  }

  private ChatManagerInterface getChatManager() {
    return ChatManager.get(this.chatId);
  }

  @Override
  public void init() {
    view.onFetchStateChange(FetchState.LOADING);

    if (this.chatId != null) {
      initUserChat();
    } else {
      initLocalChat();
    }
  }

  // set local chat logic

  private void initLocalChat() {
    InitMessageItem item = ChatSelector.getLocalMessage(presetMessage);

    if (item == null) {
      view.finish(Transition.NONE);
      return;
    }

    if (item instanceof WelcomeMessageItem) {
      ChannelSelector.bindAcceptInput(acceptInput -> {
        view.onChatInteractionStateChange(acceptInput ? ChatInteractionState.NORMAL : ChatInteractionState.DISABLE);
      }).bind(this, BindAction.BIND_LOCAL_MESSAGE_INPUT_STATE);
    } else if (item instanceof SupportBotMessageItem) {
      view.onChatInteractionStateChange(ChatInteractionState.NONE);
    }

    // need to remove later
    view.switchHeader(false);

    adapterModel.setInitMessage(item);
    view.onFetchStateChange(FetchState.COMPLETE);
  }

  private void createUserChat() {
    if (!isRunning(BindAction.CREATE_CHAT)) {
      Plugin plugin = PluginStore.get().pluginState.get();
      String pluginId = plugin != null ? plugin.getId() : null;

      Activity topActivity = GlobalStore.get().topActivity.get();
      String url = Optional.ofNullable(topActivity).map(Activity::getClass).map(Class::getSimpleName).orElse("");

      if (pluginId != null) {
        Api.createUserChat2(pluginId, url)
            .onError(error -> {
              for (SendItem item : this.preCreateItems) {
                item.doFail();
                adapterModel.addMessageItem(item);
                this.preCreateFailedItems.add(item);
              }

              this.preCreateItems.clear();
            })
            .call(repo -> onChatCreated(repo.getUserChat(), repo.getMessage()))
            .bind(this, BindAction.CREATE_CHAT);
      }
    }
  }

  private void createUserChat(@NonNull String supportBotId, @NonNull String actionType, @NonNull ActionButton actionButton) {
    if (!isRunning(BindAction.CREATE_CHAT)) {
      SendActionItem item = new SendActionItem(null, actionType, actionButton);

      String url = Optional.ofNullable(GlobalStore.get().topActivity.get())
          .map(Activity::getClass)
          .map(Class::getSimpleName)
          .orElse("");

      addLocalSendItem(item);

      Api.createSupportBotUserChat2(supportBotId, url)
          .onError(ex -> {
            this.adapterModel.removeMessageItem(item);
            this.preCreateItems.remove(item);
          })
          .call(repo -> {
            item.updateMessageOnActionInput(repo.getMessage());

            onChatCreated(repo.getUserChat(), repo.getMessage());
          })
          .bind(this, BindAction.CREATE_CHAT);
    }
  }

  private void onChatCreated(UserChat userChat, @Nullable Message message) {
    // clear binder when using in local message
    unbind(BindAction.BIND_LOCAL_MESSAGE_INPUT_STATE);

    // set id
    this.chatId = userChat.getId();

    // attach chat manager listener
    getChatManager().attachListener(this);

    // send created event
    ChannelPluginListener listener = ChannelIO.getListener();

    if (listener != null && userChat.getId() != null) {
      listener.onChatCreated(userChat.getId());
    }

    // replace init message
    if (message != null) {
      adapterModel.replaceInitMessageItem(message);
    }

    // send messages to manager
    for (SendItem item : this.preCreateItems) {
      item.setChatId(this.chatId);
    }
    for (SendItem item : this.preCreateFailedItems) {
      item.setChatId(this.chatId);
    }

    getChatManager().sendMessages(this.preCreateItems);
    getChatManager().addFailedItems(this.preCreateFailedItems);

    // clear pre created send items
    this.preCreateItems.clear();
    this.preCreateFailedItems.clear();

    // bind init
    initUserChat(message);
  }

  @Override
  public void createMarketingSupportBotUserChat() {
    if (this.chatInteractionBinder != null) {
      Message marketingMessage = chatInteractionBinder.getMarketingMessage();

      if (this.chatId != null && marketingMessage != null && marketingMessage.getMarketing() != null && !isRunning(BindAction.CREATE_CHAT)) {
        view.showProgress();

        Api.createMarketingSupportBotUserChat(this.chatId, marketingMessage.getMarketing().getType(), marketingMessage.getMarketing().getId())
            .onComplete(() -> view.hideProgress())
            .call(repo -> {
              adapterModel.addMessage(repo.getMessage());

              if (this.chatInteractionBinder != null) {
                chatInteractionBinder.setMarketingMessage(repo.getMessage());
              }
            })
            .bind(this, BindAction.CREATE_CHAT);
      }
    }
  }

  // Add temporary sending items before chat created

  private void addLocalSendItem(SendItem item) {
    addLocalSendItems(Collections.singleton(item));
  }

  private void addLocalSendItems(Collection<SendItem> items) {
    this.adapterModel.addMessageItems(items);
    this.preCreateItems.addAll(items);
  }

  // user chat logic

  private void initUserChat() {
    getChatManager().attachListener(this);

    initUserChat(null);
  }

  private void initUserChat(@Nullable Message initMessage) {
    // must call only once

    SocketSelector.bindSocket(socketStatus -> {
      if (socketStatus == SocketStatus.READY) {
        fetchMessages();
      }
    }).bind(this, BindAction.BIND_SOCKET);

    if (this.chatId != null) {
      ChatSelector.bindHostTyping(this.chatId, typings -> adapterModel.setTypings(typings)).bind(this);
    }

    this.chatHeaderBinder = new ChatHeaderBinder(showAssignee -> view.switchHeader(showAssignee)).bind(this);

    this.chatInteractionBinder = new ChatInteractionBinder(initMessage, chatInteractionState ->
        view.onChatInteractionStateChange(chatInteractionState)
    ).bind(this);

    RxBus.bind(this::onSocketReceived, this);
  }

  // data fetch

  private void fetchMessages() {
    if (this.chatId != null) {
      this.chatState = ChatState.FETCHING;

      initQueue();

      Api.getMessages(this.chatId)
          .onError(ex -> {
            this.chatState = ChatState.DISCONNECTED;

            if (ex.is4xxClientError()) {
              view.finish();
            } else {
              view.onFetchStateChange(FetchState.FAILED);
            }

            clearQueue();
          })
          .call(repo -> {
            this.chatState = ChatState.NEWEST;

            UserChatRepo userChatRepo = repo.getUserChatRepo();
            MessagesRepo messagesRepo = repo.getMessagesRepo();

            onUserChatChanged(userChatRepo.getUserChat());

            this.lastReadSession = userChatRepo.getSession();
            this.prev = messagesRepo.getNext();

            // resolve messages
            ArrayList<Message> appendMessages = new ArrayList<>(messagesRepo.getMessages());

            if (this.tempQueue != null) {
              appendMessages.addAll(this.tempQueue);

              clearQueue();
            }

            // sort append messages
            Collections.sort(appendMessages, (o1, o2) -> CompareUtils.compare(o1.getCreatedAt(), o2.getCreatedAt()));

            adapterModel.setMessages(appendMessages, getChatManager().getUnsentItems(), this.lastReadSession, messagesRepo.getNext());

            // set last message when repo.prev (next) is null and not empty for show support bot button
            Message lastMessage = messagesRepo.getPrev() != null || appendMessages.isEmpty() ? null : appendMessages.get(appendMessages.size() - 1);
            if (this.chatInteractionBinder != null) {
              this.chatInteractionBinder.setMarketingMessage(lastMessage);
            }

            view.scrollToBottom();
            view.onFetchStateChange(FetchState.COMPLETE);

            ChatAction.read(chatId);

            // need to change postDelayed
            if (messagesRepo.getNext() != null && !view.isScrollable()) {
              fetchPrevMessages();
            }
          })
          .bind(this, BindAction.FETCH_MESSAGES);
    }
  }

  @Override
  public void fetchPrevMessages() {
    if (this.chatId != null && !isRunning(BindAction.FETCH_MESSAGES) && this.prev != null)
      Api.getMessages2(this.chatId, this.prev, Const.MESSAGE_FETCH_LIMIT, Const.DESC)
          .call(repo -> {
            this.prev = repo.getNext();

            adapterModel.addMessages(repo.getMessages(), this.lastReadSession, repo.getNext());

            if (repo.getNext() != null && !view.isScrollable()) {
              unbind(BindAction.FETCH_MESSAGES);

              fetchPrevMessages();
            }
          })
          .bind(this, BindAction.FETCH_MESSAGES);
  }

  // handle action, socket

  private void onSocketReceived(Object object) {
    if (object instanceof UserChat) {
      UserChat userChat = (UserChat) object;

      // TODO : check user chat version
      if (this.chatId != null && this.chatId.equals(userChat.getId())) {
        onUserChatChanged(userChat);
      }
    }

    if (object instanceof Message) {
      Message message = (Message) object;

      if (this.chatId != null && this.chatId.equals(message.getChatId())) {
        onReceiveMessage(message);
      }
    }
  }

  private void onUserChatChanged(UserChat userChat) {
    if (userChat.isStateRemoved()) {
      view.finish(Transition.NONE);
      return;
    }

    if (this.chatHeaderBinder != null) {
      this.chatHeaderBinder.setUserChat(userChat);
    }

    if (this.chatInteractionBinder != null) {
      this.chatInteractionBinder.setUserChat(userChat);
    }

    view.onChatStateChange(userChat);
  }

  @Override
  protected void handleAction(ActionType actionType) {
    switch (actionType) {
      case SOCKET_DISCONNECTED:
      case SHUTDOWN:
        if (isRunning(BindAction.FETCH_MESSAGES)) {
          view.onFetchStateChange(FetchState.FAILED);

          unbind(BindAction.FETCH_MESSAGES);
        }

        clearQueue();

        this.chatState = ChatState.DISCONNECTED;
        this.prev = null;
        break;
    }
  }

  private void onReceiveMessage(Message message) {
    switch (this.chatState) {
      case NEWEST:
        boolean isScrollOnBottom = view.isScrollOnBottom();

        adapterModel.addMessage(message);

        if (isScrollOnBottom) {
          view.scrollToBottom();
        } else if (!CompareUtils.isSame(message.getPersonType(), Const.USER)) {
          // TODO : Show new message alert in bottom
        }

        // to refresh chat interactive state
        if (this.chatInteractionBinder != null) {
          this.chatInteractionBinder.setMarketingMessage(message);
        }
        break;

      case FETCHING:
        if (this.tempQueue != null) {
          this.tempQueue.add(message);
        }
        break;
    }
  }

  // message actions

  @Override
  public void onLocalSupportBotActionClick(@Nullable String supportBotId, @NonNull String actionType, @NonNull ActionButton actionButton) {
    if (this.chatId == null && Const.ACTION_TYPE_SUPPORT_BOT.equals(actionType) && supportBotId != null) {
      createUserChat(supportBotId, actionType, actionButton);
    }

    view.scrollToBottom();
  }

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    if (CompareUtils.exists(actionType, ACTION_TYPE_SOLVE, ACTION_TYPE_CLOSE) &&
        CompareUtils.isSame(actionButton.getKey(), Const.ACTION_KEY_REOPEN)
    ) {
      reopenChat(actionButton);
      return;
    }

    if (this.chatId != null) {
      getChatManager().sendMessage(new SendActionItem(this.chatId, actionType, actionButton));
    }

    view.scrollToBottom();
  }

  private void reopenChat(@NonNull ActionButton actionButton) {
    Message message = actionButton.getMessage();
    message.clearAction();
    adapterModel.addMessage(message);

    if (this.chatInteractionBinder != null) {
      this.chatInteractionBinder.setTemporaryInputOpened(true);
    }

    view.scrollToBottom();
  }

  @Override
  public void deleteMessage(String messageId) {
    if (this.chatId != null) {
      Api.deleteMessage2(this.chatId, messageId)
          .call(repo -> {
            if (repo.getMessage() != null) {
              adapterModel.updateItem(new ChatMessageItem(repo.getMessage()));
            }
          })
          .bind(this);
    }
  }

  // send actions

  @Override
  public void sendText(String message) {
    if (this.chatId == null) {
      addLocalSendItem(new SendTextItem(null, message));
      createUserChat();
    } else {
      getChatManager().sendMessage(new SendTextItem(this.chatId, message));
    }

    view.scrollToBottom();
  }

  @Override
  public void resend(SendItem item) {
    if (this.chatId == null) {
      this.adapterModel.removeMessageItem(item);

      if (this.preCreateFailedItems.remove(item)) {
        item.renew();

        this.adapterModel.addMessageItem(item);
        this.preCreateItems.add(item);

        createUserChat();
      }
    } else {
      getChatManager().resend(item);
    }

    view.scrollToBottom();
  }

  @Override
  public void removeFailedItem(SendItem item) {
    this.adapterModel.removeMessageItem(item);
    this.preCreateFailedItems.remove(item);
    getChatManager().remove(item);
  }

  @Override
  public void uploadFiles(List<PhotoItem> photoItems) {
    if (photoItems != null) {
      List<SendItem> items = Stream.of(photoItems)
          .mapIndexed((IndexedFunction<PhotoItem, SendItem>) (index, item) ->
              new SendFileItem(this.chatId, index, item.getUri(), item.getName(), item.getSize())
          )
          .toList();

      if (this.chatId == null) {
        addLocalSendItems(items);
        createUserChat();
      } else {
        getChatManager().sendMessages(items);
      }

      view.scrollToBottom();
    }
  }

  @Override
  public void cancelSendingFile(SendFileItem item) {
    if (item.getType() == MessageType.FAILED_FILE) {
      removeFailedItem(item);
    } else {
      getChatManager().cancelRecentSendingFile();
    }
  }

  @Override
  public void setTyping(boolean isTyping) {
    if (this.chatInteractionBinder != null) {
      this.chatInteractionBinder.setTyping(isTyping);
    }
  }

  // message send listeners

  @Override
  public void onMessageItemUpsert(MessageItem item) {
    adapterModel.addMessageItem(item);
  }

  @Override
  public void onMessageItemRemove(MessageItem item) {
    adapterModel.removeMessageItem(item);
  }

  @Override
  public void onMessageSendSuccess(Message message) {
    boolean isScrollOnBottom = view.isScrollOnBottom();

    adapterModel.addMessage(message);

    if (isScrollOnBottom) {
      view.scrollToBottom();
    }

    // to refresh chat interactive state
    if (this.chatInteractionBinder != null) {
      this.chatInteractionBinder.setMarketingMessage(message);
    }
  }

  // util functions

  private void initQueue() {
    clearQueue();

    this.tempQueue = new ArrayList<>();
  }

  private void clearQueue() {
    if (this.tempQueue != null) {
      this.tempQueue.clear();
      this.tempQueue = null;
    }
  }
}
