package com.zoyi.channel.plugin.android.activity.chat2;

import android.app.Dialog;
import android.content.Intent;
import android.graphics.Rect;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.appcompat.app.AlertDialog;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.action.MarketingAction;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity2;
import com.zoyi.channel.plugin.android.activity.chat.ChatAdapter;
import com.zoyi.channel.plugin.android.activity.chat.dialog.ReactionsDialog;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.listener.view.ChatInteractionActionListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.viewholder.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.utils.KeyboardUtils;
import com.zoyi.channel.plugin.android.activity.chat.view.chat.ChatInteractionView;
import com.zoyi.channel.plugin.android.activity.chat.view.chat.NewMessageAlertView;
import com.zoyi.channel.plugin.android.activity.chat.view.navigation.HostHeaderView;
import com.zoyi.channel.plugin.android.activity.common.navigation.ChannelInfoNavigationView;
import com.zoyi.channel.plugin.android.activity.common.navigation.NavigationView;
import com.zoyi.channel.plugin.android.activity.download.DownloadActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumStorage;
import com.zoyi.channel.plugin.android.activity.photo_picker.PhotoPickerActivity;
import com.zoyi.channel.plugin.android.enumerate.*;
import com.zoyi.channel.plugin.android.global.Action;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.view.dialog.ChannelDialog;
import com.zoyi.channel.plugin.android.view.dialog.bottom_sheet.IconButtonBottomSheetDialog;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.load_wrapper.LoadWrapperLayout;
import com.zoyi.channel.plugin.android.view.scrollview.RecyclerBottomPlaceholderLayout;
import com.zoyi.channel.plugin.android.view.toast.DateToastView;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.subjects.PublishSubject;

import java.util.List;
import java.util.concurrent.TimeUnit;

public class ChatActivity2
    extends BaseActivity2
    implements
    ChatContract2.View,
    ChatInteractionActionListener,
    OnAttachmentUploadContentActionListener,
    OnMessageActionListener,
    OnSendingActionListener,
    TypingListener {

  private NavigationView navigation;
  private ChannelInfoNavigationView channelHeader;
  private HostHeaderView hostHeader;

  private LoadWrapperLayout loaderChat;

  private ChatInteractionView interactionView;

  private RecyclerView recyclerView;

  private NewMessageAlertView newMessageAlertView;
  private DateToastView dateToastView;

  private RecyclerBottomPlaceholderLayout bottomPlaceholder;

  private ChatAdapter adapter;
  private LinearLayoutManager layoutManager;

  private ChatContract2.Presenter presenter;

  @Nullable
  private KeyboardUtils keyboardUtils;

  @Nullable
  private Subscription toastSubscription;

  private PublishSubject<Integer> toastPublishSubject = PublishSubject.create();


  @Initializer
  @Override
  protected boolean onCreate() {
    init(R.layout.ch_plugin_activity_chat2);

    String contentId = Optional.ofNullable(getIntent()).map(intent -> intent.getStringExtra(Const.EXTRA_CHAT_CONTENT_ID)).orElse(null);
    String presetMessage = Optional.ofNullable(getIntent()).map(intent -> intent.getStringExtra(Const.EXTRA_CHAT_PRESET_MESSAGE)).orElse(null);

    // view initialize

    navigation = findViewById(R.id.ch_navigationChat);
    navigation.addAction(R.drawable.ch_plugin_close_white, v -> Action.invoke(ActionType.EXIT));

    channelHeader = findViewById(R.id.ch_navigationChatChannel);
    hostHeader = findViewById(R.id.ch_navigationChatHost);
    hostHeader.setVisibility(View.GONE);

    loaderChat = findViewById(R.id.ch_loaderChat);

    recyclerView = findViewById(R.id.ch_recyclerViewChat);

    newMessageAlertView = findViewById(R.id.ch_viewNewMessageAlert);
    newMessageAlertView.setListener(() -> {
      scrollToBottom();
      newMessageAlertView.hide();
    });

    dateToastView = findViewById(R.id.ch_viewChatDateToast);

    interactionView = findViewById(R.id.ch_viewChatInteraction);
    interactionView.setChatInteractionActionListener(this);
    interactionView.setTypingListener(this);

    // adapter initialize

    adapter = new ChatAdapter();
    adapter.setOnMessageActionListener(this);
    adapter.setOnSendingActionListener(this);
    adapter.setOnAttachmentUploadContentActionListener(this);

    // presenter initialize

    presenter = new ChatPresenter2(
        this,
        adapter,
        adapter,
        contentId,
        presetMessage
    );
    // init recycler view

    layoutManager = new LinearLayoutManager(this) {
      @Override
      public boolean requestChildRectangleOnScreen(
          @NonNull RecyclerView parent,
          @NonNull View child,
          @NonNull Rect rect,
          boolean immediate,
          boolean focusedChildVisible
      ) {
        return false;
      }
    };

    recyclerView.setLayoutManager(layoutManager);
    recyclerView.setAdapter(adapter);
    recyclerView.setItemAnimator(null);
    recyclerView.addOnScrollListener(new InfiniteScrollListener() {
      @Override
      public void scrollAttachedToBottom() {
        newMessageAlertView.hide();
      }

      @Override
      public void scrollAttachedToTop() {
        presenter.fetchPrevMessages();
      }
    });

    // bottom placeholder

    bottomPlaceholder = findViewById(R.id.ch_bottomLayoutChat);
    bottomPlaceholder.setRecyclerView(recyclerView);
    bottomPlaceholder.setStackFromEnd(true);
    bottomPlaceholder.setOnSizeChangeListener(height -> {
      ViewGroup.LayoutParams params = newMessageAlertView.getLayoutParams();

      if (params instanceof ViewGroup.MarginLayoutParams) {
        ((ViewGroup.MarginLayoutParams) params).setMargins(0, 0, 0, height);
      }
    });

    // keyboard adjust resize utils

    keyboardUtils = new KeyboardUtils(this, loaderChat, recyclerView);
    keyboardUtils.enable();

    toastSubscription = toastPublishSubject
        .throttleLast(250, TimeUnit.MILLISECONDS)
        .onBackpressureLatest()
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(integer -> showFirstMessageDate(adapter.getItem(integer)));

    recyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
      @Override
      public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
        if (dy != 0) {
          if (toastPublishSubject != null && recyclerView.getLayoutManager() != null) {
            toastPublishSubject.onNext(((LinearLayoutManager) recyclerView.getLayoutManager()).findFirstVisibleItemPosition());
          }
        }
      }
    });

    bindPresenter(presenter);

    GlobalStore.get().activeChatActivity2.set(this);

    return true;
  }

  @Override
  protected void onPause() {
    super.onPause();

    KeyboardUtils.hideKeyboard(this);

    if (isFinishing()) {
      ChatStore.get().reset();
      TranslationStore.get().reset();
      ProfileBotStore.get().reset();
      TypingStore.get().reset();

      ChatActivity2 chatActivity2 = GlobalStore.get().activeChatActivity2.get();

      if (this == chatActivity2) {
        GlobalStore.get().activeChatActivity2.set(null);
      }

      Action.invoke(ActionType.CHAT_CLOSED);
    }
  }

  @Override
  protected void onDestroy() {
    recyclerView.setAdapter(null);

    if (keyboardUtils != null) {
      keyboardUtils.disable();
    }
    if (toastSubscription != null && !toastSubscription.isUnsubscribed()) {
      toastSubscription.unsubscribe();
    }
    keyboardUtils = null;
    toastSubscription = null;

    super.onDestroy();
  }

  @Override
  public void onFetchStateChange(FetchState fetchState) {
    loaderChat.setLoadState(fetchState);
  }

  @Override
  public void switchHeader(boolean showAssignee) {
    Views.setVisibility(channelHeader, !showAssignee);
    Views.setVisibility(hostHeader, showAssignee);
  }

  @Override
  public void onChatStateChange(UserChat userChat) {
    interactionView.initUserChat(userChat.getId());
    hostHeader.attachUserChat(userChat.getId());
  }

  @Override
  public void onChatInteractionStateChange(ChatInteractionState inputType) {
    if (interactionView != null) {
      interactionView.setState(inputType);
    }
  }

  @Override
  public boolean isScrollOnBottom() {
    return !recyclerView.canScrollVertically(1);
  }

  @Override
  public void scrollToBottom() {
    layoutManager.scrollToPosition(adapter.getItemCount() - 1);
  }

  @Override
  public boolean isScrollable() {
    return recyclerView.canScrollVertically(1) || recyclerView.canScrollVertically(-1);
  }

  // interaction view listeners

  @Override
  public void onAttachmentButtonClick() {
    IntentUtils.setNextActivity(this, PhotoPickerActivity.class).startActivityForResult(Const.REQUEST_PHOTO);
  }

  @Override
  public void onSendClick(String message) {
    if (presenter != null) {
      presenter.sendText(message);
    }
  }

  @Override
  public void startNewChat(int resultCode, Transition transition) {
    finish(resultCode, transition);
  }

  @Override
  public void startMarketingSupportBot() {
    if (presenter != null) {
      presenter.createMarketingSupportBotUserChat();
    }
  }

  // content action, click listeners

  @Nullable
  @Override
  public Dialog onMessageLongClick(Message message) {
    IconButtonBottomSheetDialog dialog = new IconButtonBottomSheetDialog(this);

    if (message != null) {
      if (!message.getPlainText().isEmpty()) {
        dialog.addButton(
            R.drawable.ch_copy,
            ResUtils.getString("ch.chat.message.actions.copy_message"),
            () -> copyText(message.getPlainText())
        );
      }

      if (
          !message.isDeleted()
              && message.getPersonType() != null
              && message.getPersonType().equals(Const.USER)
              && message.getChatId() != null
              && message.getId() != null
              && message.getLog() == null
      ) {
        dialog.addButton(
            R.drawable.ch_trash,
            ResUtils.getString("ch.chat.message.actions.delete_message"),
            R.color.ch_red400,
            R.color.ch_red400,
            () -> deleteMessage(message.getId())
        );
      }

      if (dialog.getButtons().size() > 0) {
        return dialog;
      }
    }
    return null;
  }

  @Override
  public void onReactionsLongClicked(List<Reaction> reactions) {
    new ReactionsDialog(this, reactions).show();
  }

  @Override
  public void onMarketingAction(Marketing marketing, @Nullable String url) {
    MarketingAction.sendClickEvent(marketing, url);
  }

  @Override
  public void onOpenVideoClick(File attachment, long startAt) {
    Executor.startFullScreenVideo(this, attachment, startAt);
  }

  @Override
  public void onAttachmentClick(File attachment, Message message) {
    if (attachment.isImage()) {
      IntentUtils.setNextActivity(this, PhotoAlbumActivity.class)
          .putExtra(Const.EXTRA_ATTACHMENT_ID, attachment.getId())
          .putExtra(Const.EXTRA_STORAGE_ID, PhotoAlbumStorage.getInstance().save(message.getFiles()))
          .startActivity();
    } else {
      IntentUtils.setNextActivity(this, DownloadActivity.class)
          .putExtra(Const.EXTRA_URL, attachment.getUrl())
          .putExtra(Const.EXTRA_FILE_NAME, attachment.getName())
          .putExtra(Const.EXTRA_TYPE, attachment.getType())
          .setTransition(Transition.NONE)
          .startActivity();
    }
  }

  @Override
  public void onActionButtonClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    if (presenter != null) {
      presenter.onActionClick(actionType, actionButton);
    }
  }

  @Override
  public void onLocalSupportBotActionClick(@Nullable String supportBotId, @NonNull String actionType, @NonNull ActionButton actionButton) {
    if (presenter != null) {
      presenter.onLocalSupportBotActionClick(supportBotId, actionType, actionButton);
    }
  }

  @Override
  public void onResendButtonClick(SendItem sendItem) {
    // TODO : fix to channel dialog
    final AlertDialog dialog = new AlertDialog.Builder(this)
        .setMessage(ResUtils.getString("ch.chat.resend.description"))
        .setPositiveButton(ResUtils.getString("ch.chat.retry_sending_message"), (dialog12, which) -> {
          if (presenter != null) {
            presenter.resend(sendItem);
          }
        })
        .setNegativeButton(ResUtils.getString("ch.chat.resend.cancel"), null)
        .setNeutralButton(ResUtils.getString("ch.chat.delete"), (dialog1, which) -> {
          if (presenter != null) {
            presenter.removeFailedItem(sendItem);
          }
        })
        .setCancelable(true)
        .create();

    dialog.setOnShowListener(args -> {
      int dark = ContextCompat.getColor(ChatActivity2.this, R.color.ch_grey900);
      int cobalt = ContextCompat.getColor(ChatActivity2.this, R.color.ch_cobalt400);

      dialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(cobalt);
      dialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(dark);
      dialog.getButton(AlertDialog.BUTTON_NEUTRAL).setTextColor(dark);
    });
    dialog.show();
  }

  @Override
  public void onCancelClick(@Nullable SendFileItem sendFileItem) {
    if (sendFileItem != null) {
      presenter.cancelSendingFile(sendFileItem);
    }
  }

  @Override
  public void onUrlClick(String url) {
    Executor.executeLinkAction(this, url, LinkType.URL);
  }

  @Override
  public void onTypingStateChange(boolean isTyping) {
    if (presenter != null) {
      presenter.setTyping(isTyping);
    }
  }

  // callbacks

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);

    switch (requestCode) {
      case Const.REQUEST_PHOTO:
        if (resultCode == Const.RESULT_PHOTO) {
          presenter.uploadFiles(data.getParcelableArrayListExtra(Const.PHOTO_INTENT_KEY));
        }
        break;
    }
  }

  // private functions

  private void copyText(String text) {
    if (ClipboardUtils.copyToClipBoard(text)) {
      Toast.makeText(this, ResUtils.getString("ch.copy_message.success"), Toast.LENGTH_SHORT).show();
    }
  }

  private void showFirstMessageDate(@Nullable MessageItem item) {
    if (item != null && dateToastView != null) {
      switch (item.getType()) {
        case HOST:
        case USER:
          ChatMessageItem chatMessageItem = (ChatMessageItem) item;
          if (chatMessageItem.getCreatedAt() != 0L) {
            dateToastView.show(chatMessageItem.getCreatedAt());
          }
          break;

      }
    }
  }

  private void deleteMessage(String messageId) {
    new ChannelDialog(this)
        .setTitle(ResUtils.getString("ch.chat.message.delete_confirm.title"))
        .setDescription(ResUtils.getString("ch.chat.message.delete_confirm.message"))
        .addButton(ButtonType.CANCEL)
        .addButton(
            ResUtils.getString("ch.chat.delete"),
            ResUtils.getColor(R.color.ch_red400),
            (v) -> {
              if (presenter != null) {
                presenter.deleteMessage(messageId);
              }
            }
        )
        .show();
  }
}
