package com.zoyi.channel.plugin.android.activity.chat.view.attachment;

import android.content.Context;
import android.content.res.TypedArray;
import androidx.annotation.IntRange;
import androidx.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.contentview.OnAttachmentContentActionListener;
import com.zoyi.channel.plugin.android.activity.chat.view.viewholder.MessageContentView;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.view.media_thumbnail.MediaThumbnailView;
import com.zoyi.channel.plugin.android.view.media_thumbnail.SquareAttachmentMediaThumbnailView;
import com.zoyi.com.annimon.stream.Stream;

import java.util.List;

public class AttachmentGroupGridLayout extends ViewGroup implements MessageContentView {

  private Context context;
  private int margin = 0;
  @IntRange(from = 1)
  private int columnSize = 1;

  public AttachmentGroupGridLayout(Context context) {
    super(context);

    init(context, null);
  }

  public AttachmentGroupGridLayout(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context, attrs);
  }

  public AttachmentGroupGridLayout(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context, attrs);
  }

  private void init(Context context, @Nullable AttributeSet attrs) {
    this.context = context;

    if (attrs != null) {
      TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.AttachmentGroupGridLayout);
      try {
        margin = typedArray.getDimensionPixelSize(R.styleable.AttachmentGroupGridLayout_ch_aggl_margin, 0);
        columnSize = Math.max(typedArray.getInt(R.styleable.AttachmentGroupGridLayout_ch_aggl_columnSize, columnSize), 1);
      } finally {
        typedArray.recycle();
      }
    }
  }

  public void setAttachments(
      @Nullable String messageId,
      List<File> files,
      @Nullable OnAttachmentContentActionListener listener
  ) {
    clear();

    Stream.ofNullable(files).forEachIndexed((index, file) -> {
      switch (file.getType()) {
        case Const.FILE_TYPE_VIDEO:
        case Const.FILE_TYPE_IMAGE:
          SquareAttachmentMediaThumbnailView view = new SquareAttachmentMediaThumbnailView(context);
          view.setAttachment(messageId, file, listener);

          addView(view);
          break;
      }
    });
  }

  @Override
  public void resetView() {
    if (getChildCount() > 0) {
      clear();
      removeAllViews();
    }
  }

  @Override
  public void clear() {
    if (getChildCount() > 0) {
      for (int i = 0; i < getChildCount(); i++) {
        View view = getChildAt(i);

        if (view instanceof MediaThumbnailView) {
          ((MediaThumbnailView) view).clear();
        }
      }
    }
  }

  @Override
  protected void onMeasure(int widthSpec, int heightSpec) {
    final int width = MeasureSpec.getSize(widthSpec);
    final int availableWidth = MeasureSpec.getSize(widthSpec) - getPaddingLeft() - getPaddingRight();
    final int verticalPadding = getPaddingTop() + getPaddingBottom();
    final int count = getChildCount();
    final int contentSize = Math.max(0, (availableWidth - (columnSize - 1) * margin) / columnSize);
    final int rowCount = (count + 1) / columnSize;

    for (int i = 0; i < count; i++) {
      final View child = getChildAt(i);

      if (child.getVisibility() != GONE) {
        child.measure(
            MeasureSpec.makeMeasureSpec(contentSize, MeasureSpec.EXACTLY),
            MeasureSpec.makeMeasureSpec(contentSize, MeasureSpec.EXACTLY)
        );
      }
    }

    setMeasuredDimension(width, contentSize * rowCount + margin * Math.max(0, rowCount - 1) + verticalPadding);
  }

  @Override
  protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
    final int width = right - left;
    final int availableWidth = width - getPaddingLeft() - getPaddingRight();
    final int count = getChildCount();
    final int contentSize = Math.max(0, (availableWidth - (columnSize - 1) * margin) / columnSize);
    final int appendSize = contentSize + margin;
    final int topPoint = getPaddingTop();
    final int leftPoint = getPaddingLeft();

    for (int i = 0; i < count; i++) {
      final View view = getChildAt(i);

      if (view.getVisibility() != View.GONE) {
        view.layout(
            (i % columnSize) * appendSize + leftPoint,
            (i / columnSize) * appendSize + topPoint,
            (i % columnSize) * appendSize + contentSize + leftPoint,
            (i / columnSize) * appendSize + contentSize + topPoint
        );
      }
    }
  }
}
