package com.zoyi.channel.plugin.android.view.video_player;

import android.content.Context;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnAttachmentContentActionListener;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.store.GlobalStore;
import com.zoyi.channel.plugin.android.util.Views;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.com.google.android.exoplayer2.*;
import com.zoyi.com.google.android.exoplayer2.source.ExtractorMediaSource;
import com.zoyi.com.google.android.exoplayer2.trackselection.AdaptiveTrackSelection;
import com.zoyi.com.google.android.exoplayer2.trackselection.DefaultTrackSelector;
import com.zoyi.com.google.android.exoplayer2.ui.PlayerView;
import com.zoyi.com.google.android.exoplayer2.upstream.DefaultHttpDataSourceFactory;
import com.zoyi.com.google.android.exoplayer2.util.Util;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.concurrent.TimeUnit;

public class AttachmentVideoPlayerView extends AbsVideoPlayerView {

  public AttachmentVideoPlayerView(@NonNull Context context) {
    super(context);

    init(context);
  }

  public AttachmentVideoPlayerView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context);
  }

  public AttachmentVideoPlayerView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context);
  }

  private Context context;

  private PlayerView playerAttachment;
  private View buttonVideoVolumeOn;
  private View buttonVideoVolumeOff;
  private View buttonVideoFullscreen;

  @Nullable
  private SimpleExoPlayer exoPlayer;

  @Nullable
  private Subscription hideControllerSubscription;
  @Nullable
  private Subscription videoProgressSubscription;

  @Nullable
  private File attachment;
  @Nullable
  private OnAttachmentContentActionListener listener;

  private void init(Context context) {
    this.context = context;

    LayoutInflater.from(context).inflate(R.layout.ch_view_attachment_video_player, this);

    playerAttachment = findViewById(R.id.ch_playerAttachment);
    buttonVideoVolumeOn = findViewById(R.id.ch_buttonVideoVolumeOn);
    buttonVideoVolumeOff = findViewById(R.id.ch_buttonVideoVolumeOff);
    buttonVideoFullscreen = findViewById(R.id.ch_buttonVideoFullscreen);

    buttonVideoVolumeOn.setOnClickListener(v -> {
      applyVolumeButton(false);

      if (listener != null) {
        listener.onAttachmentActionClick();
      }
    });
    buttonVideoVolumeOff.setOnClickListener(v -> {
      applyVolumeButton(true);

      if (listener != null) {
        listener.onAttachmentActionClick();
      }
    });
    buttonVideoFullscreen.setOnClickListener(v -> {
      if (exoPlayer != null) {
        exoPlayer.setPlayWhenReady(false);
      }

      if (listener != null && attachment != null) {
        listener.onOpenVideoClick(attachment, getLatestSeconds() * TimeUnit.SECONDS.toMillis(1));
      }
    });
  }

  public AttachmentVideoPlayerView setUrl(
      @Nullable String messageId,
      File attachment,
      @Nullable OnAttachmentContentActionListener listener,
      @Nullable Boolean isVolumeOn
  ) {
    setId(String.format("%s:%s", messageId, attachment.getId()));

    this.attachment = attachment;
    this.listener = listener;

    AdaptiveTrackSelection.Factory videoTrackSelectionFactory = new AdaptiveTrackSelection.Factory();
    DefaultTrackSelector trackSelector = new DefaultTrackSelector(videoTrackSelectionFactory);
    boolean volumeState = Optional.ofNullable(isVolumeOn).orElse(getLatestVolumeState());

    exoPlayer = ExoPlayerFactory.newSimpleInstance(context, new DefaultRenderersFactory(context), trackSelector);
    exoPlayer.addListener(new Player.EventListener() {
      @Override
      public void onPlayerStateChanged(boolean playWhenReady, int playbackState) {
        changeVideoState(playWhenReady);

        if (playWhenReady) {
          bindVideoProgress();
        } else {
          unbindVideoProgress();
        }

        switch (playbackState) {
          case Player.STATE_ENDED:
            if (exoPlayer != null) {
              exoPlayer.seekTo(0);
              exoPlayer.setPlayWhenReady(false);
            }
            break;
          default:
            if (listener != null) {
              listener.onAttachmentActionClick();
            }
            break;
        }
      }
    });

    playerAttachment.setPlayer(exoPlayer);

    DefaultHttpDataSourceFactory dataSourceFactory = new DefaultHttpDataSourceFactory(
        Util.getUserAgent(context, "ChannelTalkSDK")
    );
    dataSourceFactory.getDefaultRequestProperties().set(Const.X_SESSION, GlobalStore.get().jwt.get());

    ExtractorMediaSource videoSource = new ExtractorMediaSource.Factory(dataSourceFactory)
        .createMediaSource(Uri.parse(attachment.getUrl()));

    exoPlayer.prepare(videoSource);
    exoPlayer.setPlayWhenReady(true);
    exoPlayer.seekTo(getLatestSeconds() * TimeUnit.SECONDS.toMillis(1));

    applyVolumeButton(volumeState);

    return this;
  }

  private void applyVolumeButton(boolean isVolumeOn) {
    if (exoPlayer != null) {
      exoPlayer.setVolume(isVolumeOn ? 1f : 0f);
    }

    Views.setVisibility(buttonVideoVolumeOn, isVolumeOn);
    Views.setVisibility(buttonVideoVolumeOff, !isVolumeOn);

    changeVideoVolumeState(isVolumeOn);
  }

  @Override
  void onVideoPaused() {
    if (exoPlayer != null) {
      exoPlayer.setPlayWhenReady(false);
    }
  }

  // handle progress

  private void bindVideoProgress() {
    if (videoProgressSubscription == null || videoProgressSubscription.isUnsubscribed()) {
      unbindVideoProgress();

      videoProgressSubscription = Observable.interval(1000, TimeUnit.MILLISECONDS)
          .onBackpressureLatest()
          .subscribeOn(Schedulers.computation())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(aLong -> Optional.ofNullable(exoPlayer)
              .map(SimpleExoPlayer::getCurrentPosition)
              .map(position -> position / TimeUnit.SECONDS.toMillis(1))
              .ifPresent(this::changeVideoPosition)
          );
    }
  }

  private void unbindVideoProgress() {
    dispose(videoProgressSubscription);
  }

  // private functions

  private void dispose(@Nullable Subscription subscription) {
    if (subscription != null && !subscription.isUnsubscribed()) {
      subscription.unsubscribe();
    }
  }

  // clear


  @Override
  public void clear() {
    super.clear();

    this.attachment = null;
    this.listener = null;

    dispose(hideControllerSubscription);
    dispose(videoProgressSubscription);

    hideControllerSubscription = null;
    videoProgressSubscription = null;

    if (exoPlayer != null) {
      exoPlayer.release();
      exoPlayer = null;
    }
  }
}
