package com.zoyi.channel.plugin.android;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.support.annotation.*;
import android.text.TextUtils;
import android.util.Log;

import com.zoyi.channel.plugin.android.action.*;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.global.Action;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Event;
import com.zoyi.channel.plugin.android.open.callback.BootCallback;
import com.zoyi.channel.plugin.android.open.callback.UserUpdateCallback;
import com.zoyi.channel.plugin.android.open.config.BootConfig;
import com.zoyi.channel.plugin.android.open.enumerate.BootStatus;
import com.zoyi.channel.plugin.android.open.exception.ChannelException;
import com.zoyi.channel.plugin.android.open.model.UserData;
import com.zoyi.channel.plugin.android.push.ChannelPushClient;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.selector.GlobalSelector;
import com.zoyi.channel.plugin.android.store.PluginStore;
import com.zoyi.channel.plugin.android.store.SettingsStore;
import com.zoyi.channel.plugin.android.util.ListUtils;

import java.util.List;
import java.util.Map;

public class ChannelIO {

  private static boolean isDebugMode = !BuildConfig.IS_PRODUCTION_DEPLOYMENT;

  private static boolean attachChannelView = true;

  @Nullable
  private static RealChannelIO realChannelIO;

  // Main Method

  public static void initialize(@Nullable Application application) {
    initialize(application, true);
  }

  public static void initialize(@Nullable Application application, boolean attachView) {
    if (application == null) {
      Log.e("ChannelIO", "Fail to 'initialize', Application can't be NULL");
    } else if (realChannelIO != null) {
      Log.e("ChannelIO", "Fail to 'initialize', Channel plugin already initialized");
    } else {
      attachChannelView = attachView;
      realChannelIO = new RealChannelIO(application);
    }
  }

  public static boolean isAttachChannelView() {
    return attachChannelView;
  }

  public static void setDebugMode(boolean enable) {
    isDebugMode = enable;
  }

  // boot functions

  @Deprecated
  public static void boot(@Nullable ChannelPluginSettings pluginSettings) {
    boot(pluginSettings, null, null);
  }

  @Deprecated
  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable OnBootListener completion) {
    boot(pluginSettings, null, completion);
  }

  @Deprecated
  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile) {
    boot(pluginSettings, profile, null);
  }

  @Deprecated
  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile, @Nullable OnBootListener completion) {
    Log.w("ChannelIO", "This boot function is deprecated. Please use ChannelIO.boot(BootConfig) or ChannelIO.boot(BootConfig, BootCallback)");

    // Set debug mode to console
    if (pluginSettings != null) {
      ChannelIO.setDebugMode(pluginSettings.isDebugMode());
    }

    boot(BootConfig.fromLegacy(pluginSettings, profile), (bootStatus, user) -> {
      User legacyUser = user != null ? user.toLegacy() : null;

      if (completion != null) {
        switch (bootStatus) {
          case SUCCESS:
            completion.onCompletion(ChannelPluginCompletionStatus.SUCCESS, legacyUser);
            break;
          case NOT_INITIALIZED:
            completion.onCompletion(ChannelPluginCompletionStatus.NOT_INITIALIZED, legacyUser);
            break;
          case NETWORK_TIMEOUT:
            completion.onCompletion(ChannelPluginCompletionStatus.NETWORK_TIMEOUT, legacyUser);
            break;
          case NOT_AVAILABLE_VERSION:
            completion.onCompletion(ChannelPluginCompletionStatus.NOT_AVAILABLE_VERSION, legacyUser);
            break;
          case SERVICE_UNDER_CONSTRUCTION:
            completion.onCompletion(ChannelPluginCompletionStatus.SERVICE_UNDER_CONSTRUCTION, legacyUser);
            break;
          case REQUIRE_PAYMENT:
            completion.onCompletion(ChannelPluginCompletionStatus.REQUIRE_PAYMENT, legacyUser);
            break;
          case ACCESS_DENIED:
            completion.onCompletion(ChannelPluginCompletionStatus.ACCESS_DENIED, legacyUser);
            break;
          default:
            completion.onCompletion(ChannelPluginCompletionStatus.UNKNOWN_ERROR, legacyUser);
            break;
        }
      }
    });
  }

  public static void boot(@Nullable BootConfig bootConfig) {
    boot(bootConfig, null);
  }

  public static void boot(@Nullable BootConfig bootConfig, @Nullable BootCallback bootCallback) {
    if (realChannelIO == null) {
      Log.e("ChannelIO", "Fail to boot, Initialize first");

      if (bootCallback != null) {
        bootCallback.onComplete(BootStatus.NOT_INITIALIZED, null);
      }
    } else if (bootConfig == null || bootConfig.getPluginKey() == null) {
      Log.e("ChannelIO", "Fail to boot, Check boot configuration");

      if (bootCallback != null) {
        bootCallback.onComplete(BootStatus.NOT_INITIALIZED, null);
      }
    } else {
      realChannelIO.boot(bootConfig, bootCallback);
    }
  }

  // sleep

  public static void sleep() {
    if (realChannelIO != null) {
      realChannelIO.sleep();
    }
  }

  // shutdown

  public static void shutdown() {
    if (realChannelIO != null) {
      realChannelIO.shutdown();
    }
  }

  // showMessenger

  public static void showMessenger(@Nullable Activity activity) {
    if (activity != null && realChannelIO != null) {
      realChannelIO.showMessenger(activity);
    }
  }


  /**
   * @deprecated Use {@link #showMessenger(Activity)}
   */
  @Deprecated
  public static boolean open(@Nullable Context context) {
    Log.w("ChannelIO", "ChannelIO.open(Context) is deprecated. Please use ChannelIO.showMessenger(Activity)");
    return open(context, true);
  }

  /**
   * @deprecated Use {@link #showMessenger(Activity)}
   */
  @Deprecated
  public static boolean open(@Nullable Context context, boolean animate) {
    Log.w("ChannelIO", "ChannelIO.open(Context, boolean) is deprecated. Please use ChannelIO.showMessenger(Activity)");
    if (context instanceof Activity && realChannelIO != null) {
      showMessenger((Activity) context);
      return true;
    }
    return false;
  }

  // hideMessenger

  public static void hideMessenger() {
    Action.invoke(ActionType.EXIT);
  }

  /**
   * @deprecated Use {@link #hideMessenger()}
   */
  @Deprecated
  public static void close() {
    Log.w("ChannelIO", "ChannelIO.close() is deprecated. Please use ChannelIO.hideMessenger()");
    hideMessenger();
  }

  /**
   * @deprecated Use {@link #hideMessenger()}
   */
  @Deprecated
  public static void close(boolean animated) {
    Log.w("ChannelIO", "ChannelIO.close(boolean) is deprecated. Please use ChannelIO.hideMessenger()");
    hideMessenger();
  }

  // open chat

  public static void openChat(@Nullable Activity activity, @Nullable String chatId, @Nullable String message) {
    if (activity != null && realChannelIO != null) {
      realChannelIO.openChat(activity, chatId, message);
    }
  }

  /**
   * @deprecated Use {@link #openChat(Activity, String, String)}
   */
  @Deprecated
  public static boolean openChat(@Nullable Context context) {
    Log.w("ChannelIO", "ChannelIO.openChat(Context) is deprecated. Please use ChannelIO.openChat(Activity, String, String)");
    return openChat(context, null);
  }

  /**
   * @deprecated Use {@link #openChat(Activity, String, String)}
   */
  @Deprecated
  public static boolean openChat(@Nullable Context context, @Nullable String chatId) {
    Log.w("ChannelIO", "ChannelIO.openChat(Context, String) is deprecated. Please use ChannelIO.openChat(Activity, String, String)");
    return openChat(context, chatId, true);
  }

  /**
   * @deprecated Use {@link #openChat(Activity, String, String)}
   */
  @Deprecated
  public static boolean openChat(@Nullable Context context, @Nullable String chatId, boolean animate) {
    Log.w("ChannelIO",
        "ChannelIO.openChat(Context, String, boolean) is deprecated. Please use ChannelIO.openChat(Activity, String, String)");
    if (context instanceof Activity && realChannelIO != null) {
      openChat((Activity) context, chatId, null);
      return true;
    }
    return false;
  }

  // show

  public static void showChannelButton() {
    if (!SettingsStore.get().showLauncher.get()) {
      SettingsStore.get().showLauncher.set(true);

      if (isBooted() && SettingsStore.get().trackDefaultEvent.get()) {
        EventAction.trackPageView();
      }
    }
  }

  /**
   * @deprecated Use {@link #showChannelButton()}
   */
  @Deprecated
  public static void show() {
    Log.w("ChannelIO", "ChannelIO.show() is deprecated. Please use ChannelIO.showChannelButton()");
    showChannelButton();
  }

  // hide

  public static void hideChannelButton() {
    SettingsStore.get().showLauncher.set(false);
  }

  /**
   * @deprecated Use {@link #hideChannelButton()}
   */
  @Deprecated
  public static void hide() {
    Log.w("ChannelIO", "ChannelIO.hide() is deprecated. Please use ChannelIO.hideChannelButton()");
    hideChannelButton();
  }

  // canShowLauncher

  @Deprecated
  public static boolean canShowLauncher() {
    Log.w("ChannelIO", "ChannelIO.canShowLauncher() is deprecated.");

    return GlobalSelector.getLauncherVisibility();
  }

  // setChannelPluginListener

  public static void setListener(com.zoyi.channel.plugin.android.open.listener.ChannelPluginListener listener) {
    if (realChannelIO != null) {
      realChannelIO.setListener(listener);
    }
  }

  /**
   * @param channelPluginListener listener
   * @deprecated Use {@link #setListener(com.zoyi.channel.plugin.android.open.listener.ChannelPluginListener)}
   */
  @Deprecated
  public static void setChannelPluginListener(ChannelPluginListener channelPluginListener) {
    Log.w("ChannelIO", "ChannelIO.setChannelPluginListener is deprecated. Please use ChannelIO.setListener");

    if (realChannelIO != null) {
      realChannelIO.setListener(channelPluginListener);
    }
  }

  // clear listener

  public static void clearListener() {
    if (realChannelIO != null) {
      realChannelIO.clearListener();
    }
  }

  /**
   * @deprecated Use {@link #clearListener()}
   */
  @Deprecated
  public static void clearChannelPluginListener() {
    clearListener();
  }

  @Nullable
  public static ChannelPluginListener getLegacyListener() {
    if (realChannelIO != null) {
      return realChannelIO.getLegacyListener();
    }
    return null;
  }

  @Nullable
  public static com.zoyi.channel.plugin.android.open.listener.ChannelPluginListener getListener() {
    if (realChannelIO != null) {
      return realChannelIO.getListener();
    }
    return null;
  }

  // track

  public static void track(@NonNull @Size(min = 1L, max = 30L) String eventName) {
    track(eventName, null);
  }

  public static void track(@NonNull @Size(min = 1L, max = 30L) String eventName, @Nullable Map<String, Object> eventProperty) {
    if (TextUtils.isEmpty(eventName)) {
      Log.e("ChannelIO", "Fail to track event. Event name can't be blank or null.");
      return;
    }

    if (eventName.length() > 30) {
      Log.e("ChannelIO", "Fail to track event. Event name must be 30 characters or less.");
      return;
    }

    String jwt = PrefSupervisor.getJwt(getAppContext());

    if (jwt == null) {
      Log.e("ChannelIO", "Fail to track event. Unauthorized access.");
      return;
    }

    EventAction.track(new Event(eventName, eventProperty));
  }

  // initPushToken

  public static void initPushToken(String token) {
    Context context = getAppContext();

    if (context != null) {
      PrefSupervisor.setDeviceToken(context, token);
    }
    if (realChannelIO != null) {
      realChannelIO.registerPushToken();
    }
  }

  // isChannelPushNotification

  public static boolean isChannelPushNotification(Map<String, String> message) {
    return ChannelPushManager.isChannelPushNotification(message);
  }

  // hasStoredPushNotification

  public static boolean hasStoredPushNotification(@Nullable Activity activity) {
    if (activity != null) {
      return ChannelPushClient.hasStoredPushNotification(activity);
    }
    return false;
  }

  // openStoredPushNotification

  public static void openStoredPushNotification(@Nullable Activity activity) {
    if (activity != null) {
      ChannelPushClient.openStoredPushNotification(activity);
    }
  }

  /**
   * @deprecated Use {@link #openStoredPushNotification(Activity)}
   */
  @Deprecated
  public static void handlePushNotification(@Nullable Activity activity) {
    Log.w(
        "ChannelIO",
        "ChannelIO.handlePushNotification(Activity) is deprecated. Please use ChannelIO.openStoredPushNotification(Activity)"
    );
    openStoredPushNotification(activity);
  }

  // showPushNotification

  public static void receivePushNotification(Context context, Map<String, String> message) {
    if (context != null) {
      ChannelPushManager.receivePushNotification(context, message);
    }
  }

  /**
   * @deprecated Use {@link #receivePushNotification(Context, Map)}
   */
  @Deprecated
  public static void showPushNotification(Context context, Map<String, String> message) {
    Log.w(
        "ChannelIO",
        "ChannelIO.showPushNotification(Context, Map) is deprecated. Please use ChannelIO.receivePushNotification(Context, Map)"
    );
    receivePushNotification(context, message);
  }

  // updateUser

  public static void updateUser(@Nullable UserData userData, @Nullable UserUpdateCallback callback) {
    if (isBooted()) {
      UserAction.updateUser(userData, callback);
    } else if (callback != null) {
      callback.onComplete(ChannelException.newInstance("Please boot first"), null);
    }
  }

  /**
   * @deprecated Use {@link #updateUser(UserData, UserUpdateCallback)}
   */
  @Deprecated
  public static void updateProfile(Map<String, Object> profileMap) {
    updateUser(new UserData.Builder().setProfileMap(profileMap).build(), null);
  }

  // Add tags

  public static void addTags(String... tags) {
    if (tags != null) {
      addTags(ListUtils.newArrayList(tags), null);
    }
  }

  public static void addTags(List<String> tags, @Nullable UserUpdateCallback callback) {
    if (isBooted()) {
      TagAction.addTags(tags, callback);
    } else if (callback != null) {
      callback.onComplete(ChannelException.newInstance("Please boot first"), null);
    }
  }

  // Remove tags

  public static void removeTags(String... tags) {
    if (tags != null) {
      removeTags(ListUtils.newArrayList(tags), null);
    }
  }

  public static void removeTags(List<String> tags, @Nullable UserUpdateCallback callback) {
    if (isBooted()) {
      TagAction.removeTags(tags, callback);
    } else if (callback != null) {
      callback.onComplete(ChannelException.newInstance("Please boot first"), null);
    }
  }

  // library inner functions

  public static boolean isInitialized() {
    return realChannelIO != null;
  }

  @Nullable
  public static Context getAppContext() {
    if (realChannelIO != null) {
      return realChannelIO.getApplication();
    }
    return null;
  }

  public static boolean isDebugMode() {
    return isDebugMode;
  }

  public static boolean isBooted() {
    return PluginStore.get().pluginState.get() != null;
  }
}
