package com.zoyi.channel.plugin.android.activity.chat.manager;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import com.zoyi.channel.plugin.android.action.ChatAction;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.ContentType;
import com.zoyi.channel.plugin.android.activity.chat.listener.MessageSendListener;
import com.zoyi.channel.plugin.android.enumerate.SendingState;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.ActionButton;
import com.zoyi.channel.plugin.android.model.repo.MessageRepo;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.com.annimon.stream.Stream;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;
import com.zoyi.rx.subjects.PublishSubject;

import java.util.*;

class RealChatManager implements ChatManagerInterface {

  private String chatId;

  private Map<ContentType, Queue<SendItem>> queue;

  private Map<ContentType, SendItem> sendingItems;

  private PublishSubject<Integer> waitingCountPublishSubject = PublishSubject.create();
  private PublishSubject<Integer> uploadProgressPublishSubject = PublishSubject.create();

  @Nullable
  private Subscription fileSubscription;

  private int currentFileUploadProgress = 0;

  // key is request id
  private Map<String, SendItem> failedItems;

  private boolean isReleased = false;

  @Nullable
  private MessageSendListener listener;

  RealChatManager(String chatId) {
    this.chatId = chatId;
    this.queue = new HashMap<>();
    this.sendingItems = new HashMap<>();
    this.failedItems = new HashMap<>();

    for (ContentType contentType : ContentType.values()) {
      queue.put(contentType, new LinkedList<>());
    }
  }

  // manage listeners

  @Override
  public void attachListener(@Nullable MessageSendListener listener) {
    this.listener = listener;
  }

  @Override
  public void clearListener() {
    this.listener = null;
  }

  // send

  @Override
  public void sendMessage(SendItem item) {
    if (listener != null && item.shouldShowTempMessage()) {
      listener.onMessageItemUpsert(item);
    }

    if (item.getState() == SendingState.FAIL) {
      failedItems.put(item.getRequestId(), item);
    } else {
      enqueue(item.getContentType(), item);
    }
  }

  @Override
  public void sendMessages(List<SendItem> items) {
    Stream.ofNullable(items).forEach(this::sendMessage);
  }

  private void enqueue(ContentType contentType, SendItem item) {
    Optional.ofNullable(queue.get(contentType)).ifPresent(q -> {
      q.add(item);

      if (contentType == ContentType.FILE) {
        notifyWaitingFileCount();
      }

      dequeue(contentType);
    });
  }

  private void dequeue(ContentType contentType) {
    if (!isReleased && sendingItems.get(contentType) == null) {
      SendItem nextItem = Optional.ofNullable(queue.get(contentType)).map(Queue::poll).orElse(null);

      switch (contentType) {
        case TEXT:
          if (nextItem instanceof SendTextItem) {
            sendingItems.put(ContentType.TEXT, nextItem);
            sendText((SendTextItem) nextItem);
          }
          break;

        case FILE:
          if (nextItem instanceof SendFileItem) {
            if (listener != null) {
              listener.onMessageItemUpsert(nextItem);
            }
            sendingItems.put(ContentType.FILE, nextItem);
            sendFile((SendFileItem) nextItem);
          } else if (nextItem == null && listener != null) {
            listener.onMessageItemRemove(DummyItem.createSendFileItem());
          }
          break;

        case ACTION:
          if (nextItem instanceof SendActionItem) {
            sendingItems.put(ContentType.ACTION, nextItem);
            sendAction((SendActionItem) nextItem);
          }
          break;
      }
    }
  }

  private void sendText(SendTextItem item) {
    ChatAction.sendTextMessage(chatId, item, createCommonSubscriber(ContentType.TEXT, item));
  }

  private void sendFile(SendFileItem item) {
    this.fileSubscription = ChatAction
        .sendFileMessage(chatId, item, this::updateFileUploadProgress, createCommonSubscriber(ContentType.FILE, item));
  }

  private void sendAction(SendActionItem item) {
    String actionType = item.getActionType();
    ActionButton actionButton = item.getActionButton();

    if (actionType != null && actionButton != null) {
      switch (actionType) {
        case Const.ACTION_TYPE_SOLVE:
        case Const.ACTION_TYPE_CLOSE:
          switch (actionButton.getKey()) {
            case Const.ACTION_KEY_CLOSE:
              closeChat(item);
              break;

            case Const.ACTION_KEY_LIKE:
              reviewChat(item, Const.REVIEW_LIKE);
              break;

            case Const.ACTION_KEY_DISLIKE:
              reviewChat(item, Const.REVIEW_DISLIKE);
              break;

            default:
              doAction(item);
              break;
          }
          break;

        case Const.ACTION_TYPE_SUPPORT_BOT:
          selectSupportBotStep(item);
          break;

        default:
          doAction(item);
          break;
      }
    }
  }

  // actions

  private void closeChat(SendActionItem item) {
    ChatAction.closeChat(chatId, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  private void reviewChat(SendActionItem item, @NonNull String review) {
    ChatAction.reviewChat(chatId, review, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  private void selectSupportBotStep(SendActionItem item) {
    ChatAction.selectSupportBotStep(chatId, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  private void doAction(SendActionItem item) {
    ChatAction.selectForm(chatId, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  // resend

  @Override
  public void resend(SendItem item) {
    remove(item);

    item.renew();

    sendMessage(item);
  }

  // remove

  @Override
  public void remove(SendItem item) {
    if (listener != null) {
      listener.onMessageItemRemove(item);
    }

    failedItems.remove(item.getRequestId());
  }

  // cancel

  @Override
  public void cancelRecentSendingFile() {
    SendItem item = sendingItems.get(ContentType.FILE);

    if (item != null && this.fileSubscription != null && !this.fileSubscription.isUnsubscribed()) {
      this.fileSubscription.unsubscribe();
      this.fileSubscription = null;

      sendingItems.remove(ContentType.FILE);

      if (listener != null) {
        listener.onMessageItemRemove(item);
      }

      updateFileUploadProgress(0);

      dequeue(ContentType.FILE);
    }
  }

  // public actions

  @Override
  public int getWaitingFileCount() {
    return Optional.ofNullable(queue.get(ContentType.FILE)).map(Queue::size).orElse(0);
  }

  @Override
  public int getCurrentFileUploadProgress() {
    return currentFileUploadProgress;
  }

  @NonNull
  @Override
  public List<SendItem> getUnsentItems() {
    List<SendItem> unsentItems = new ArrayList<>();

    for (ContentType contentType : ContentType.values()) {
      switch (contentType) {
        case TEXT:
        case ACTION:
          Optional.ofNullable(queue.get(contentType)).ifPresent(unsentItems::addAll);
          break;
      }
    }

    Stream.ofNullable(ContentType.values())
        .filter(ContentType::isValid)
        .map(type -> sendingItems.get(type))
        .forEach(item -> {
          if (item != null) {
            unsentItems.add(item);
          }
        });

    unsentItems.addAll(failedItems.values());

    return unsentItems;
  }

  @Override
  public Observable<Integer> getUploadProgressObservable() {
    return uploadProgressPublishSubject
        .onBackpressureLatest()
        .subscribeOn(Schedulers.io())
        .observeOn(AndroidSchedulers.mainThread());
  }

  @Override
  public Observable<Integer> getWaitingCountObservable() {
    return waitingCountPublishSubject
        .onBackpressureLatest()
        .subscribeOn(Schedulers.io())
        .observeOn(AndroidSchedulers.mainThread());
  }

  // private actions

  private void notifyWaitingFileCount() {
    Optional.ofNullable(queue.get(ContentType.FILE)).map(Queue::size).ifPresent(count -> waitingCountPublishSubject.onNext(count));
  }

  private RestSubscriber<MessageRepo> createCommonSubscriber(ContentType contentType, SendItem item) {
    return new RestSubscriber<MessageRepo>() {
      @Override
      public void onError(Throwable e) {
        item.doFail();

        sendingItems.remove(contentType);

        failedItems.put(item.getRequestId(), item);

        if (listener != null) {
          listener.onMessageItemUpsert(item);
        }

        dequeue(contentType);
      }

      @Override
      public void onSuccess(@NonNull MessageRepo repo) {
        sendingItems.remove(contentType);

        if (listener != null) {
          listener.onMessageItemRemove(item);
          listener.onMessageSendSuccess(repo.getMessage());
        }

        dequeue(contentType);
      }
    };
  }

  private void updateFileUploadProgress(int progress) {
    this.currentFileUploadProgress = progress;
    this.uploadProgressPublishSubject.onNext(progress);
  }

  @Override
  public void release() {
    this.isReleased = true;

    this.queue.clear();
    this.sendingItems.clear();
    this.failedItems.clear();
    this.listener = null;
  }
}
