package com.zoyi.channel.plugin.android.view.media_thumbnail;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.*;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.FormatUtils;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.view.layout.ChBorderLayout;
import com.zoyi.channel.plugin.android.view.video_player.AbsVideoPlayerView;
import com.zoyi.com.bumptech.glide.Glide;
import com.zoyi.com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.zoyi.com.bumptech.glide.load.resource.drawable.GlideDrawable;
import com.zoyi.com.bumptech.glide.request.animation.GlideAnimation;
import com.zoyi.com.bumptech.glide.request.target.SimpleTarget;

public abstract class MediaThumbnailView extends ChBorderLayout {

  public MediaThumbnailView(@NonNull Context context) {
    super(context);

    init(context);
  }

  public MediaThumbnailView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context);
  }

  public MediaThumbnailView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context);
  }

  private Context context;

  private View viewPlayerController;

  private View layoutMediaThumbnailPreview;
  private ImageView imageMediaThumbnail;
  private View layoutPlaceholder;
  private View imagePlaceholder;

  private FrameLayout layoutMediaThumbnailVideos;
  private View viewPlayerDuration;
  private TextView textPlayerDuration;

  @Initializer
  private void init(@NonNull Context context) {
    this.context = context;

    LayoutInflater.from(context).inflate(R.layout.ch_view_media_thumbnail, this);

    viewPlayerController = findViewById(R.id.ch_viewPlayerController);

    layoutMediaThumbnailPreview = findViewById(R.id.ch_layoutMediaThumbnailPreview);
    imageMediaThumbnail = findViewById(R.id.ch_imageMediaThumbnail);
    layoutPlaceholder = findViewById(R.id.ch_layoutPlaceholder);
    imagePlaceholder = findViewById(R.id.ch_imagePlaceholder);

    layoutMediaThumbnailVideos = findViewById(R.id.ch_layoutMediaThumbnailVideos);
    viewPlayerDuration = findViewById(R.id.ch_viewPlayerDuration);
    textPlayerDuration = findViewById(R.id.ch_textPlayerDuration);

    viewPlayerController.setOnClickListener(v -> onPlayClick());
  }

  @Override
  public void setOnClickListener(@Nullable View.OnClickListener l) {
    layoutMediaThumbnailPreview.setOnClickListener(l);
    imageMediaThumbnail.setOnClickListener(l);
    layoutPlaceholder.setOnClickListener(l);
  }

  @Override
  public void setOnLongClickListener(@Nullable View.OnLongClickListener l) {
    viewPlayerController.setOnLongClickListener(l);

    layoutMediaThumbnailPreview.setOnLongClickListener(l);
    imageMediaThumbnail.setOnLongClickListener(l);
    layoutPlaceholder.setOnLongClickListener(l);
  }

  protected void setImage(@Nullable String url) {
    setImage(url, ImageView.ScaleType.CENTER_CROP);
  }

  protected void setImage(@Nullable String url, ImageView.ScaleType scaleType) {
    imageMediaThumbnail.setScaleType(scaleType);

    try {
      Glide.with(context)
          .load(url)
          .diskCacheStrategy(DiskCacheStrategy.SOURCE)
          .into(new SimpleTarget<GlideDrawable>() {
            @Override
            public void onResourceReady(GlideDrawable glideDrawable, GlideAnimation<? super GlideDrawable> glideAnimation) {
              layoutPlaceholder.setVisibility(View.GONE);
              imageMediaThumbnail.setVisibility(View.VISIBLE);
              imageMediaThumbnail.setImageDrawable(glideDrawable);
            }

            @Override
            public void onLoadFailed(Exception e, Drawable errorDrawable) {
              imagePlaceholder.animate().alpha(1).setDuration(200).start();
            }
          });
    } catch (Exception e) {
    }
  }

  protected void setImage(@Nullable String url, int width, int height) {
    try {
      Glide.with(context)
          .load(url)
          .centerCrop()
          .override(width, height)
          .diskCacheStrategy(DiskCacheStrategy.SOURCE)
          .into(new SimpleTarget<GlideDrawable>() {
            @Override
            public void onResourceReady(GlideDrawable glideDrawable, GlideAnimation<? super GlideDrawable> glideAnimation) {
              layoutPlaceholder.setVisibility(View.GONE);
              imageMediaThumbnail.setVisibility(View.VISIBLE);
              imageMediaThumbnail.setImageDrawable(glideDrawable);
            }

            @Override
            public void onLoadFailed(Exception e, Drawable errorDrawable) {
              imagePlaceholder.animate().alpha(1).setDuration(200).start();
            }
          });
    } catch (Exception e) {
    }
  }

  protected <E extends AbsVideoPlayerView> E appendVideoView(E view) {
    if (layoutMediaThumbnailVideos != null) {
      layoutMediaThumbnailVideos.setVisibility(View.VISIBLE);
      layoutMediaThumbnailVideos.addView(view);
    }

    return view;
  }

  protected void activatePlayButton() {
    viewPlayerController.setVisibility(View.VISIBLE);
  }

  protected void activateDuration(@Nullable Double duration) {
    Long durationMilliseconds = duration != null ? duration.longValue() : null;
    String formattedDuration = FormatUtils.formatDuration(durationMilliseconds);

    if (formattedDuration != null && viewPlayerDuration != null) {
      viewPlayerDuration.setVisibility(View.VISIBLE);
      textPlayerDuration.setText(formattedDuration);
    } else if (viewPlayerDuration != null) {
      viewPlayerDuration.setVisibility(View.GONE);
    }
  }

  public void onPlayClick() {
    layoutPlaceholder.setVisibility(View.GONE);
    layoutMediaThumbnailPreview.setVisibility(View.INVISIBLE);
    viewPlayerController.setVisibility(View.GONE);
    viewPlayerDuration.setVisibility(View.GONE);
  }

  public void clear() {
    if (layoutMediaThumbnailPreview != null) {
      layoutMediaThumbnailPreview.setVisibility(View.VISIBLE);
    }

    if (layoutPlaceholder != null) {
      layoutPlaceholder.setVisibility(View.VISIBLE);
    }

    if (imagePlaceholder != null) {
      imagePlaceholder.animate().alpha(0).setDuration(0).start();
    }

    if (imageMediaThumbnail != null) {
      imageMediaThumbnail.setVisibility(View.INVISIBLE);

      Glide.clear(imageMediaThumbnail);
    }

    if (layoutMediaThumbnailVideos != null) {
      for (int i = 0; i < layoutMediaThumbnailVideos.getChildCount(); i++) {
        View child = layoutMediaThumbnailVideos.getChildAt(i);

        if (child instanceof AbsVideoPlayerView) {
          ((AbsVideoPlayerView) child).clear();
        }
      }

      layoutMediaThumbnailVideos.removeAllViews();
      layoutMediaThumbnailVideos.setVisibility(View.GONE);
    }

    // video controller: gone
    if (viewPlayerController != null) {
      viewPlayerController.setVisibility(View.GONE);
    }

    // video duration: gone
    if (viewPlayerDuration != null) {
      viewPlayerDuration.setVisibility(View.GONE);
    }
  }
}
