package com.zoyi.channel.plugin.android.global;

import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.BuildConfig;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.model.rest.AppMessengerUri;
import com.zoyi.channel.plugin.android.model.rest.Country;
import com.zoyi.channel.plugin.android.model.wrapper.*;
import com.zoyi.channel.plugin.android.network.*;
import com.zoyi.channel.plugin.android.open.callback.BootCallback;
import com.zoyi.channel.plugin.android.open.config.BootConfig;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.retrofit2.http.Body;
import com.zoyi.retrofit2.http.Header;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.Collections;
import java.util.List;

import static com.zoyi.channel.plugin.android.util.BootManager.bootPlugin;
import static com.zoyi.channel.plugin.android.util.BootManager.isValidPlugin;
import static com.zoyi.channel.plugin.android.util.BootManager.isValidVersion;

public class Api<E> {

  @Nullable
  private static ChannelApi channelApi;

  @Nullable
  private static SimpleChannelApi simpleChannelApi;

  @Nullable
  private static MediaChannelApi mediaChannelApi;

  /* instance fields */

  private Observable<E> observable;

  @Nullable
  private ActionType actionType;

  @Nullable
  private ActionType[] cancelTypes;

  /* static methods */

  public static ChannelApi getApi() {
    if (channelApi == null) {
      channelApi = ServiceFactory.create();
    }
    return channelApi;
  }

  public static SimpleChannelApi getSimpleApi() {
    if (simpleChannelApi == null) {
      simpleChannelApi = ServiceFactory.simpleCreate();
    }
    return simpleChannelApi;
  }

  public static MediaChannelApi getMediaApi() {
    if (mediaChannelApi == null) {
      mediaChannelApi = ServiceFactory.mediaCreate();
    }
    return mediaChannelApi;
  }

  public static boolean isRunning(ActionType actionType) {
    return Action.isRunning(actionType);
  }

  /* instance methods */

  private Api(Observable<E> observable) {
    this.observable = observable.onBackpressureBuffer()
        .subscribeOn(Schedulers.io())
        .observeOn(AndroidSchedulers.mainThread());
  }

  public Api<E> runBy(ActionType actionType) {
    this.actionType = actionType;

    return this;
  }

  public Api<E> cancelBy(ActionType... actionTypes) {
    this.cancelTypes = actionTypes;

    return this;
  }

  public Subscription run(RestSubscriber<E> subscriber) {
    return Action.apiSubscribe(observable, subscriber, actionType, cancelTypes);
  }

  public Subscription run() {
    return Action.apiSubscribe(observable, new RestSubscriber<>(), actionType, cancelTypes);
  }

  /* API Definition */

  public static Api<PluginWrapper> boot(
      @Nullable BootConfig bootConfig,
      @Nullable BootCallback bootCallback
  ) {
    if (BuildConfig.IS_PRODUCTION_DEPLOYMENT
        && CompareUtils.isSame(BuildConfig.FLAVOR, "production")
        && CompareUtils.isSame(BuildConfig.BUILD_TYPE, "release")
    ) {
      return new Api<>(getApi().getLastestPackage(Const.PACKAGE_NAME, BuildConfig.CHANNEL_IO_VERSION)
          .filter(packageInfo -> isValidVersion(packageInfo, bootCallback))
          .flatMap(__ -> bootPlugin(bootConfig))
          .filter(pluginInfo -> isValidPlugin(pluginInfo, bootCallback)));
    } else {
      return new Api<>(bootPlugin(bootConfig).filter(pluginInfo -> isValidPlugin(pluginInfo, bootCallback)));
    }
  }

  public static Api<EmptyWrapper> sendToken(@Body RequestBody body, @Header(Const.X_SESSION) String jwt) {
    return new Api<>(getApi().sendToken(body, jwt));
  }

  public static Api<Void> trackEvent(
      String pluginId,
      String name,
      String property,
      String sessionJWT
  ) {
    return new Api<>(getSimpleApi().trackEvent(pluginId, name, property, sessionJWT));
  }

  public static Api<UserChatsWrapper> getUserChats(String sortField, String sortOrder, Integer limit, boolean includeClosed) {
    return new Api<>(getApi().getUserChats(sortField, sortOrder, limit, includeClosed));
  }

  public static Api<UserChatsWrapper> getUserChats(String sortField, String sortOrder, Integer limit, boolean includeClosed, String since) {
    return new Api<>(getApi().getUserChatsNext(sortField, sortOrder, limit, includeClosed, since));
  }

  public static Api<UserWrapper> touch(String key, String jwt) {
    return new Api<>(getSimpleApi().touch(key, jwt));
  }

  public static Api<UserChatWrapper> getUserChat(String userChatId) {
    return new Api<>(getApi().getUserChat(userChatId));
  }

  public static Api<UserChatWrapper> removeUserChat(String userChatId) {
    return new Api<>(getApi().removeUserChat(userChatId));
  }

  public static Api<MessagesWrapper> getMessages(String userChatId, @Nullable String since, Integer limit, String sortOrder) {
    return new Api<>(getApi().getMessages(userChatId, since, limit, sortOrder));
  }

  public static Api<UserChatWrapper> createUserChat(String pluginId, String url) {
    return new Api<>(getApi().createUserChat(pluginId, url));
  }

  public static Api<UserChatWrapper> createSupportBotUserChat(String supportBotId, String url) {
    return new Api<>(getApi().createSupportBotUserChat(supportBotId, url));
  }

  public static Api<MessageWrapper> sendMessage(String userChatId, RequestBody body) {
    return new Api<>(getApi().sendMessage(userChatId, body));
  }

  public static Api<MessageWrapper> uploadFile(String channelId, String chatId, String requestId, ProgressRequestBody requestBody) {
    return new Api<>(getMediaApi().uploadFile(channelId, requestBody.getName(), requestBody)
        .flatMap((fileObject) -> getApi().sendMessage(
            chatId,
            RequestUtils.form()
                .set("requestId", requestId)
                .set("files", Collections.singletonList(fileObject))
                .create())
        )
    );
  }

  public static Api<MessageWrapper> closeUserChat(String userChatId, @Nullable String actionId, String requestId) {
    return new Api<>(getApi().closeUserChat(userChatId, actionId, requestId));
  }

  public static Api<MessageWrapper> reviewUserChat(String userChatId, @Nullable String actionId, String requestId, String review) {
    return new Api<>(getApi().reviewUserChat(userChatId, actionId, requestId, review));
  }

  public static Api<MessageWrapper> submitButtonAction(
      String userChatId,
      @Nullable String buttonId,
      @Nullable String messageId,
      @Nullable String requestId
  ) {
    return new Api<>(getApi().submitButtonAction(userChatId, buttonId, messageId, requestId));
  }

  public static Api<Void> read(String chatId) {
    return new Api<>(getApi().read(chatId));
  }

  public static Api<TranslationRepo> getTranslatedMessage(String userChatId, String messageId, String language) {
    return new Api<>(getApi().getTranslatedMessage(userChatId, messageId, language));
  }

  public static Api<List<Country>> getCountries() {
    return new Api<>(getApi().getCountries());
  }

  public static Api<MessageWrapper> updateProfileBot(String userChatId, String messageId, RequestBody requestBody) {
    return new Api<>(getApi().updateProfileBot(userChatId, messageId, requestBody));
  }

  public static Api<ProfileBotSchemasRepo> getProfileBotSchemas(String pluginId) {
    return new Api<>(getApi().getProfileBotSchemas(pluginId));
  }

  public static Api<UserWrapper> updateUser(RequestBody body) {
    return new Api<>(getApi().updateUser(body));
  }

  public static Api<Void> closePopUp() {
    return new Api<>(getApi().closePopUp());
  }

  public static Api<LoungeChatsWrapper> getLoungeData(String pluginId, @Nullable String url, boolean showClosedChats) {
    return new Api<>(
        Observable.zip(
            getApi().getLounge(pluginId, url),
            getApi().getUserChats(Const.UPDATED_AT, Const.DESC, Const.USER_CHATS_LIMIT, showClosedChats),
            LoungeChatsWrapper::new
        ));
  }

  public static Api<AppMessengerUri> getMessengerConnect(String appMessengerName) {
    return new Api<>(getApi().getMessengerConnect(appMessengerName));
  }

  public static Api<MessageWrapper> createMarketingSupportBotUserChat(String userChatId, String supportBotId) {
    return new Api<>(getApi().createMarketingSupportBotUserChat(userChatId, supportBotId));
  }

  public static Api<Void> sendCampaignClickEvent(String marketingId, String userId, @Nullable String url) {
    return new Api<>(getApi().campaignClick(marketingId, userId, url));
  }

  public static Api<Void> sendCampaignViewEvent(String marketingId) {
    return new Api<>(getApi().campaignView(marketingId));
  }

  public static Api<Void> oneTimeMsgClick(String marketingId, String userId, @Nullable String url) {
    return new Api<>(getApi().oneTimeMsgClick(marketingId, userId, url));
  }

  public static Api<Void> oneTimeMsgView(String marketingId) {
    return new Api<>(getApi().oneTimeMsgView(marketingId));
  }

  public static Api<UserWrapper> addTags(@Nullable List<String> tags) {
    return new Api<>(getApi().addTags(tags));
  }

  public static Api<UserWrapper> removeTags(@Nullable List<String> tags) {
    return new Api<>(getApi().removeTags(tags));
  }
}
