package com.zoyi.channel.plugin.android.activity.chat;

import android.content.Intent;
import android.graphics.Rect;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.action.MarketingAction;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity2;
import com.zoyi.channel.plugin.android.activity.chat.contract.ChatContract;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.utils.KeyboardUtils;
import com.zoyi.channel.plugin.android.activity.chat.view.chat.ChatInputView;
import com.zoyi.channel.plugin.android.activity.chat.view.chat.NewMessageAlertView;
import com.zoyi.channel.plugin.android.activity.chat.view.navigation.HostHeaderView;
import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.activity.common.error.ErrorRefreshView;
import com.zoyi.channel.plugin.android.activity.common.navigation.ChannelInfoNavigationView;
import com.zoyi.channel.plugin.android.activity.common.navigation.NavigationView;
import com.zoyi.channel.plugin.android.activity.download.DownloadActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumStorage;
import com.zoyi.channel.plugin.android.activity.photo_picker.PhotoPickerActivity;
import com.zoyi.channel.plugin.android.enumerate.*;
import com.zoyi.channel.plugin.android.global.Action;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.entity.ProfileEntity;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.util.io.Keyboard;
import com.zoyi.channel.plugin.android.view.button.ChatStartButton;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.scrollview.RecyclerBottomPlaceholderLayout;

public class ChatActivity
    extends BaseActivity2
    implements
    ChatContract.View,
    OnMessageContentClickListener,
    MessageActionListener,
    OnProfileUpdateRequestListener,
    OnChatActionListener {

  private NavigationView navigation;
  private ChannelInfoNavigationView channelHeader;
  private HostHeaderView hostHeader;

  private View layoutContent;
  private View layoutProgress;
  private ErrorRefreshView layoutError;

  private ChatInputView inputView;

  private RecyclerView recyclerView;

  private NewMessageAlertView newMessageAlertView;

  private RecyclerBottomPlaceholderLayout bottomPlaceholder;
  private ChatStartButton buttonNewChat;

  private ChatAdapter adapter;
  private LinearLayoutManager layoutManager;

  private ChatContract.Presenter presenter;

  @Nullable
  private KeyboardUtils keyboardUtils;

  @Initializer
  @Override
  protected boolean onCreate() {
    init(R.layout.ch_plugin_activity_chat);

    // intent initialize

    Intent intent = getIntent();
    ChatContentType contentType = ChatContentType.NONE;
    String contentId = null;
    String presetMessage = null;

    if (intent != null) {
      contentType = ChatContentType.fromString(intent.getStringExtra(Const.EXTRA_CHAT_CONTENT_TYPE));
      contentId = intent.getStringExtra(Const.EXTRA_CHAT_CONTENT_ID);
      presetMessage = intent.getStringExtra(Const.EXTRA_CHAT_PRESET_MESSAGE);
    }

    // view initialize

    navigation = findViewById(R.id.ch_navigationChat);
    navigation.addAction(R.drawable.ch_plugin_close_white, v -> Action.invoke(ActionType.EXIT));

    channelHeader = findViewById(R.id.ch_navigationChatChannel);
    hostHeader = findViewById(R.id.ch_navigationChatHost);
    hostHeader.setVisibility(View.GONE);

    layoutContent = findViewById(R.id.ch_layoutChatContent);
    layoutProgress = findViewById(R.id.ch_layoutChatProgress);
    layoutError = findViewById(R.id.ch_layoutChatError);

    recyclerView = findViewById(R.id.ch_recyclerViewChat);

    newMessageAlertView = findViewById(R.id.ch_viewNewMessageAlert);
    newMessageAlertView.setListener(() -> {
      scrollToBottom();
      newMessageAlertView.hide();
    });

    inputView = findViewById(R.id.ch_viewChatInput);
    inputView.setMessageActionListener(this);
    inputView.setStartSupportBotClickListener(view -> presenter.onCreateSupportBotClick());

    // adapter initialize

    adapter = new ChatAdapter(this);
    adapter.setOnMessageContentClickListener(this);
    adapter.setOnProfileUpdateRequestListener(this);

    // presenter initialize

    presenter = new ChatPresenter(this, adapter, adapter, contentType, contentId, presetMessage);

    // init recycler view

    layoutManager = new LinearLayoutManager(this) {
      @Override
      public boolean requestChildRectangleOnScreen(
          @NonNull RecyclerView parent,
          @NonNull View child,
          @NonNull Rect rect,
          boolean immediate,
          boolean focusedChildVisible
      ) {
        return false;
      }
    };

    recyclerView.setLayoutManager(layoutManager);
    recyclerView.setAdapter(adapter);
    recyclerView.setItemAnimator(null);
    recyclerView.addOnScrollListener(new InfiniteScrollListener() {
      @Override
      public void scrollAttachedToBottom() {
        newMessageAlertView.hide();
      }

      @Override
      public void scrollAttachedToTop() {
        presenter.fetchBackwardMessages();
      }
    });

    // bottom placeholder

    bottomPlaceholder = findViewById(R.id.ch_bottomLayoutChat);
    bottomPlaceholder.setRecyclerView(recyclerView);
    bottomPlaceholder.setStackFromEnd(true);
    bottomPlaceholder.setOnSizeChangeListener(height -> {
      ViewGroup.LayoutParams params = newMessageAlertView.getLayoutParams();

      if (params instanceof ViewGroup.MarginLayoutParams) {
        ((ViewGroup.MarginLayoutParams) params).setMargins(0, 0, 0, height);
      }
    });

    buttonNewChat = findViewById(R.id.ch_buttonChatStartNewChat);
    buttonNewChat.setOnClickListener(v -> finish(Const.RESULT_CHAT_OPEN_NEW_CHAT, Transition.NONE));

    // keyboard adjust resize utils

    keyboardUtils = new KeyboardUtils(this, findViewById(R.id.rootChatCoordinatorView), recyclerView);
    keyboardUtils.enable();

    bindPresenter(presenter);

    GlobalStore.get().activeChatActivity.set(this);

    return true;
  }

  @Override
  protected void onPause() {
    super.onPause();

    KeyboardUtils.hideKeyboard(this);

    if (isFinishing()) {
      recyclerView.setAdapter(null);

      ChatStore.get().reset();
      TranslationStore.get().reset();
      ProfileBotStore.get().reset();

      ChatActivity chatActivity = GlobalStore.get().activeChatActivity.get();

      if (this == chatActivity) {
        GlobalStore.get().activeChatActivity.set(null);
      }

      Action.invoke(ActionType.CHAT_CLOSED);
    }
  }

  @Override
  protected void onStop() {
    super.onStop();
  }

  @Override
  protected void onDestroy() {
    super.onDestroy();

    if (keyboardUtils != null) {
      keyboardUtils.disable();
      keyboardUtils = null;
    }
  }

  @Override
  public void onFetchStateChange(FetchState fetchState) {
    layoutContent.setVisibility(View.GONE);
    layoutProgress.setVisibility(View.GONE);
    layoutError.setVisibility(View.GONE);

    switch (fetchState) {
      case LOADING:
        layoutProgress.setVisibility(View.VISIBLE);
        break;

      case FAILED:
        layoutError.setVisibility(View.VISIBLE);
        break;

      case COMPLETE:
        layoutContent.setVisibility(View.VISIBLE);
        break;
    }
  }

  @Override
  public void onWelcomeStateChange(ChatContentType contentType, boolean acceptInput) {
    channelHeader.setVisibility(View.VISIBLE);
    hostHeader.setVisibility(View.GONE);

    // input & bottom placeholder
    // input view should set visibility case by case to keep focus

    buttonNewChat.setVisibility(View.GONE);

    switch (contentType) {
      case NONE:
        inputView.setVisibility(View.VISIBLE);
        inputView.setState(acceptInput ? ChatInputType.NORMAL : ChatInputType.DISABLE);
        break;

      case SUPPORT_BOT_CHAT:
        inputView.setVisibility(View.GONE);
        break;
    }
  }

  @Override
  public void onChatStateChange(
      UserChat userChat,
      Message recentMessage,
      boolean acceptInput,
      boolean temporaryInputOpened,
      boolean isWorking
  ) {
    if (userChat.isStateRemoved()) {
      finish(Transition.NONE);
      return;
    }

    inputView.initUserChat(userChat.getId());

    // navigation

    if (userChat.getAssigneeId() != null && isWorking) {
      channelHeader.setVisibility(View.GONE);
      hostHeader.setVisibility(View.VISIBLE);
      hostHeader.attachUserChat(userChat.getId());
    } else {
      channelHeader.setVisibility(View.VISIBLE);
      hostHeader.setVisibility(View.GONE);
    }

    // input & bottom placeholder
    // input view should set visibility case by case to keep focus

    if (recentMessage != null
        && recentMessage.getMarketing() != null
        && recentMessage.getMarketing().getEnableSupportBot()
        && userChat.getState() == null
        && !userChat.isSupporting()
        && SupportBotStore.get().supportBotState.get() != null) {
      Keyboard.close(this, inputView.getEditText());

      inputView.setState(ChatInputType.MARKETING_SUPPORT);
      buttonNewChat.setVisibility(View.GONE);
    } else if (userChat.isSupporting()) {
      Keyboard.close(this, inputView.getEditText());

      inputView.setVisibility(View.GONE);
      buttonNewChat.setVisibility(View.GONE);
    } else if (userChat.isStateClosed()) {
      Keyboard.close(this, inputView.getEditText());

      inputView.setVisibility(View.GONE);
      buttonNewChat.setVisibility(View.VISIBLE);
    } else if (userChat.isStateSolved()) {
      if (temporaryInputOpened) {
        inputView.setVisibility(View.VISIBLE);
        inputView.setState(acceptInput ? ChatInputType.NORMAL : ChatInputType.DISABLE);
        buttonNewChat.setVisibility(View.GONE);
      } else {
        Keyboard.close(this, inputView.getEditText());

        inputView.setVisibility(View.GONE);
        buttonNewChat.setVisibility(View.GONE);
      }
    } else {
      inputView.setVisibility(View.VISIBLE);
      inputView.setState(acceptInput ? ChatInputType.NORMAL : ChatInputType.DISABLE);
      buttonNewChat.setVisibility(View.GONE);
    }
  }

  @Override
  public boolean isScrollOnBottom() {
    return !recyclerView.canScrollVertically(1);
  }

  @Override
  public void scrollToBottom() {
    layoutManager.scrollToPosition(adapter.getItemCount() - 1);
  }

  @Override
  public void showNewMessageAlert(ProfileEntity profileEntity) {
    newMessageAlertView.show(profileEntity);
  }

  // Handle message content action

  @Override
  public void onOpenVideoClick(File attachment, long startAt) {
    Executor.startFullScreenVideo(this, attachment, startAt);
  }

  @Override
  public void onAttachmentClick(File attachment, Message message) {
    if (attachment.isImage()) {
      IntentUtils.setNextActivity(this, PhotoAlbumActivity.class)
          .putExtra(Const.EXTRA_ATTACHMENT_ID, attachment.getId())
          .putExtra(Const.EXTRA_STORAGE_ID, PhotoAlbumStorage.getInstance().save(message.getFiles()))
          .startActivityForResult(Const.REQUEST_PHOTO_ALBUM);
    } else {
      IntentUtils.setNextActivity(this, DownloadActivity.class)
          .putExtra(Const.EXTRA_URL, attachment.getUrl())
          .putExtra(Const.EXTRA_FILE_NAME, attachment.getName())
          .putExtra(Const.EXTRA_IMAGE, false)
          .setTransition(Transition.NONE)
          .startActivity();
    }
  }

  @Override
  public void onUrlClick(String url) {
    Executor.executeLinkAction(this, url, LinkType.URL);
  }

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    presenter.onActionClick(actionType, actionButton);
  }

  @Override
  public void onResendButtonClick(SendItem sendItem) {
    final AlertDialog dialog = new AlertDialog.Builder(this)
        .setMessage(ResUtils.getString("ch.chat.resend.description"))
        .setPositiveButton(ResUtils.getString("ch.chat.retry_sending_message"), (dialog12, which) -> presenter.resend(sendItem))
        .setNegativeButton(ResUtils.getString("ch.chat.resend.cancel"), null)
        .setNeutralButton(ResUtils.getString("ch.chat.delete"), (dialog1, which) -> presenter.removeFailedItem(sendItem))
        .setCancelable(true)
        .create();

    dialog.setOnShowListener(args -> {
      int dark = ContextCompat.getColor(ChatActivity.this, R.color.ch_grey900);
      int cobalt = ContextCompat.getColor(ChatActivity.this, R.color.ch_cobalt400);

      dialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(cobalt);
      dialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(dark);
      dialog.getButton(AlertDialog.BUTTON_NEUTRAL).setTextColor(dark);
    });
    dialog.show();
  }

  @Override
  public void onCancelClick(@Nullable SendFileItem sendFileItem) {
    if (sendFileItem != null) {
      presenter.cancelSendingFile(sendFileItem);
    }
  }

  @Override
  public void onAttachmentButtonClick() {
    IntentUtils.setNextActivity(this, PhotoPickerActivity.class).startActivityForResult(Const.PHOTO_REQUEST_CODE);
  }

  @Override
  public void onSendClick(String message) {
    presenter.sendText(message);
  }

  @Override
  public void onMarketingAction(Marketing marketing, @Nullable String url) {
    MarketingAction.sendClickEvent(marketing, url);
  }

  // profile bot listeners

  @Override
  public void onProfileRequest(ProfileBotMessageItem item, String key, @Nullable Object value) {
    presenter.updateProfile(item, key, value);
  }

  @Override
  public void onProfileUpdateComplete() {
    inputView.setFocus();
  }

  @Override
  public void setInputDim(boolean enabled) {
    inputView.setInputDim(enabled);
  }

  @Override
  public void setTextInput(@Nullable String message) {
    if (message != null) {
      inputView.getEditText().setText(message);
    }
  }

  // callbacks

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);

    switch (requestCode) {
      case Const.PHOTO_REQUEST_CODE:
        if (resultCode == Const.PHOTO_RESULT_CODE) {
          presenter.uploadFiles(data.getParcelableArrayListExtra(Const.PHOTO_INTENT_KEY));
        }
        break;
    }
  }
}
