package com.zoyi.channel.plugin.android;

import android.app.Activity;
import android.app.Application;
import android.arch.lifecycle.ProcessLifecycleOwner;
import android.content.Intent;
import android.os.Looper;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.action.*;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManager;
import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.activity.lounge.LoungeActivity;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.manager.ChatVideoManager;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.open.callback.BootCallback;
import com.zoyi.channel.plugin.android.open.config.BootConfig;
import com.zoyi.channel.plugin.android.open.enumerate.BootStatus;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.selector.GlobalSelector;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.store.base.Store;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.com.annimon.stream.Optional;

import java.util.concurrent.RejectedExecutionException;

class RealChannelIO {

  private Application application;

  private Thread.UncaughtExceptionHandler uncaughtExceptionHandler;

  private ChannelActionHandler handler;

  private ActivityInterceptor activityInterceptor;

  private LifecycleController lifecycleController;

  RealChannelIO(Application application) {
    this.application = application;
    this.uncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    this.handler = new ChannelActionHandler();

    SocketManager.create(application);

    activityInterceptor = new ActivityInterceptor();
    application.registerActivityLifecycleCallbacks(activityInterceptor);

    lifecycleController = new LifecycleController();
    ProcessLifecycleOwner.get().getLifecycle().addObserver(lifecycleController);

    Thread.setDefaultUncaughtExceptionHandler((thread, ex) -> {
      if (uncaughtExceptionHandler != null) {
        uncaughtExceptionHandler.uncaughtException(thread, ex);
      }

      if (!SocketManager.isReady() && Looper.getMainLooper().getThread() != thread && ex instanceof RejectedExecutionException) {
        SocketManager.reconnect();
      }
    });
  }

  Application getApplication() {
    return application;
  }

  void boot(BootConfig bootConfig, @Nullable BootCallback bootCallback) {
    shutdown(true);

    BootAction.boot(bootConfig, bootCallback, new RestSubscriber<PluginWrapper>() {
      @Override
      public void onError(RetrofitException e) {
        BootManager.sendNetworkError(bootCallback, e);
      }

      @Override
      public void onNext(PluginWrapper repo) {
        repo.set();

        onBoot(bootConfig, repo.getChannel(), repo.getUser(), bootCallback);

        Optional.ofNullable(repo)
            .map(PluginWrapper::getUser)
            .map(User::getPopUpChatId)
            .ifPresent(chatId -> ChatAction.fetchPopUpChat(repo.getUser().getPopUpChatId()));
      }
    });
  }

  private void onBoot(
      BootConfig bootConfig,
      Channel channel,
      User user,
      @Nullable BootCallback bootCallback
  ) {
    // Set latest boot data
    PrefSupervisor.setLatestBootData(application, bootConfig.getPluginKey(), channel.getId(), user.getId());

    // set prefs
    SettingsStore.get().language.set(user.getLanguage());
    SettingsStore.get().channelButtonOptionState.set(bootConfig.getChannelButtonOption());
    SettingsStore.get().hidePopup.set(bootConfig.isHidePopup());
    SettingsStore.get().trackDefaultEvent.set(bootConfig.isTrackDefaultEvent());

    // Set settings
    SettingsStore.get().showTranslation.set(PrefSupervisor.canTranslateMessage(application));
    SettingsStore.get().showClosedChat.set(PrefSupervisor.isShownClosedChat(application));
    SettingsStore.get().raiseSoundVibrate.set(PrefSupervisor.isEnabledPushAlarm(application));

    // Enable chat
    ChatManager.initialize();

    // when jwt is not initialized, fetch from preference
    if (GlobalStore.get().jwt.get() == null) {
      GlobalStore.get().jwt.set(PrefSupervisor.getJwt(application));
    }

    registerPushToken();

    SocketManager.setChannelId(channel.getId());

    // if already call ChannelIO.show(), send page view event
    if (SettingsStore.get().showLauncher.get() && SettingsStore.get().trackDefaultEvent.get()) {
      EventAction.trackPageView();
    }

    // attach handler
    handler.handle();

    // turn on socket and timer
    if (lifecycleController.isForeground()) {
      lifecycleController.doOnActivated();
    }

    // update now force
    TimerStore.get().now.set(TimeUtils.getCurrentTime());

    // raise event to initialize
    GlobalStore.get().bootState.set(true);

    // send to bootCallback

    if (bootCallback != null) {
      bootCallback.onComplete(
          BootStatus.SUCCESS,
          com.zoyi.channel.plugin.android.open.model.User.newInstance(user)
      );
    }

    // clear push data. when need to handle push after boot, do it in bootCallback.onCompletion
    PrefSupervisor.clearLatestPushData(application);
  }

  void registerPushToken() {
    ChannelPushManager.registerPushToken(application);
  }

  void deregisterPushToken() {
    ChannelPushManager.deregisterPushToken(application, PrefSupervisor.getJwt(application));
  }

  void shutdown(boolean unregisterPushToken) {
    handler.unHandle();

    lifecycleController.doOnDeactivated();

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    Action.invoke(ActionType.SHUTDOWN);
    Action.release();

    ChatManager.release();
    ChatVideoManager.get().clear();

    Store.destroy();

    if (unregisterPushToken) {
      deregisterPushToken();
    }

    GlobalStore.get().bootState.set(false);
  }

  // start messenger

  public void openChat(Activity activity, @Nullable String chatId, @Nullable String message) {
    if (!GlobalStore.get().bootState.get()) {
      L.e("Fail to start messenger, please 'Boot' first");
      return;
    }

    IntentUtils.setNextActivity(activity, LoungeActivity.class)
        .putExtra(Const.EXTRA_CHAT_CONTENT_TYPE, ChatContentType.USER_CHAT.toString())
        .putExtra(Const.EXTRA_CHAT_CONTENT_ID, chatId)
        .putExtra(Const.EXTRA_TOP_ACTIVITY_NAME, GlobalSelector.getTopActivityName())
        .setFlag(Intent.FLAG_ACTIVITY_CLEAR_TOP)
        .setTransition(Transition.SLIDE_FROM_BOTTOM)
        .startActivity();
  }

  // handle listener

  void setListener(@Nullable com.zoyi.channel.plugin.android.open.listener.ChannelPluginListener listener) {
    handler.setListener(listener);
  }

  void setListener(@Nullable ChannelPluginListener listener) {
    handler.setLegacyListener(listener);
  }

  void clearListener() {
    handler.clearListener();
  }

  @Nullable
  ChannelPluginListener getLegacyListener() {
    return handler.getLegacyListener();
  }

  @Nullable
  com.zoyi.channel.plugin.android.open.listener.ChannelPluginListener getListener() {
    return handler.getListener();
  }
}
