package com.zoyi.channel.plugin.android.view.youtube.ui.views;


import android.content.Context;
import android.content.res.TypedArray;
import android.support.annotation.ColorInt;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v4.graphics.drawable.DrawableCompat;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.Gravity;
import android.widget.LinearLayout;
import android.widget.SeekBar;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.view.youtube.player.PlayerConstants;
import com.zoyi.channel.plugin.android.view.youtube.player.YouTubePlayer;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerListener;
import com.zoyi.channel.plugin.android.view.youtube.ui.utils.TimeUtilities;
import com.zoyi.com.annimon.stream.Optional;

public class CHYouTubePlayerSeekBar extends LinearLayout implements SeekBar.OnSeekBarChangeListener, YouTubePlayerListener {


  private boolean seekBarTouchStarted = false;
  // I need this variable because onCurrentSecond gets called every 100 mils, so without the proper checks on this variable in onCurrentSeconds the seek bar glitches when touched.
  private int newSeekBarProgress = -1;

  private boolean isPlaying = false;

  private boolean showBufferingProgress = true;

  @Nullable
  private YouTubePlayerSeekBarListener youtubePlayerSeekBarListener;

  private TextView videoCurrentTimeTextView;
  private TextView videoDurationTextView;
  private SeekBar seekBar;

  public CHYouTubePlayerSeekBar(Context context) {
    super(context);
    init(context, null);
  }

  public CHYouTubePlayerSeekBar(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context, attrs);
  }

  public void setYoutubePlayerSeekBarListener(
      @Nullable YouTubePlayerSeekBarListener youtubePlayerSeekBarListener) {
    this.youtubePlayerSeekBarListener = youtubePlayerSeekBarListener;
  }

  @Initializer
  public void init(Context context, @Nullable AttributeSet attrs) {
    videoCurrentTimeTextView = new TextView(context);
    videoDurationTextView = new TextView(context);
    seekBar = new SeekBar(context);

    int fontSize = getResources().getDimensionPixelSize(R.dimen.ch_ayp_12sp);
    int color = ContextCompat.getColor(context, R.color.ch_ayp_red);
    if (attrs != null) {
      TypedArray typedArray = context.getTheme()
          .obtainStyledAttributes(attrs, R.styleable.ChYouTubePlayerSeekBar, 0, 0);

      fontSize = typedArray
          .getDimensionPixelSize(R.styleable.ChYouTubePlayerSeekBar_ch_fontSize, fontSize);
      color = typedArray.getColor(R.styleable.ChYouTubePlayerSeekBar_ch_color, color);

      typedArray.recycle();
    }

    int padding = getResources().getDimensionPixelSize(R.dimen.ch_ayp_8dp);

    videoCurrentTimeTextView.setText(getResources().getString(R.string.ch_ayp_null_time));
    videoCurrentTimeTextView.setPadding(padding, padding, 0, padding);
    videoCurrentTimeTextView.setTextColor(ContextCompat.getColor(context, android.R.color.white));
    videoCurrentTimeTextView.setGravity(Gravity.CENTER_VERTICAL);

    videoDurationTextView.setText(getResources().getString(R.string.ch_ayp_null_time));
    videoDurationTextView.setPadding(0, padding, padding, padding);
    videoDurationTextView.setTextColor(ContextCompat.getColor(context, android.R.color.white));
    videoDurationTextView.setGravity(Gravity.CENTER_VERTICAL);

    setFontSize(fontSize);

    seekBar.setPadding(padding * 2, padding, padding * 2, padding);
    setColor(color);

    addView(videoCurrentTimeTextView,
        new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.WRAP_CONTENT));
    addView(seekBar, new LinearLayout.LayoutParams(0, LinearLayout.LayoutParams.WRAP_CONTENT, 1f));
    addView(videoDurationTextView,
        new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.WRAP_CONTENT));

    setGravity(Gravity.CENTER_VERTICAL);

    seekBar.setOnSeekBarChangeListener(this);
  }

  public TextView getVideoDurationTextView() {
    return videoDurationTextView;
  }

  public SeekBar getSeekBar() {
    return seekBar;
  }

  public void setShowBufferingProgress(boolean showBufferingProgress) {
    this.showBufferingProgress = showBufferingProgress;
  }

  /**
   * @param fontSize in pixels.
   */
  public void setFontSize(float fontSize) {
    videoCurrentTimeTextView.setTextSize(TypedValue.COMPLEX_UNIT_PX, fontSize);
    videoDurationTextView.setTextSize(TypedValue.COMPLEX_UNIT_PX, fontSize);
  }

  public void setColor(@ColorInt int color) {
    DrawableCompat.setTint(seekBar.getProgressDrawable(), color);
  }

  private void updateState(PlayerConstants.PlayerState state) {
    switch (state) {
      case ENDED:
      case PAUSED:
        isPlaying = false;
        break;
      case PLAYING:
        isPlaying = true;
        break;
      case UNSTARTED:
        resetUi();
        break;
      default:
        break;
    }
  }

  private void resetUi() {
    seekBar.setProgress(0);
    seekBar.setMax(0);
    videoDurationTextView.post(() -> videoDurationTextView.setText(""));
  }

  // Seekbar

  @Override
  public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
    videoCurrentTimeTextView.setText(TimeUtilities.formatTime(progress));
  }

  @Override
  public void onStartTrackingTouch(SeekBar seekBar) {
    seekBarTouchStarted = true;
  }

  @Override
  public void onStopTrackingTouch(SeekBar seekBar) {
    if (isPlaying) {
      newSeekBarProgress = seekBar.getProgress();
    }

    Optional.ofNullable(youtubePlayerSeekBarListener).get().seekTo(seekBar.getProgress());
    seekBarTouchStarted = false;
  }

  // YouTubePlayerListener

  @Override
  public void onStateChange(YouTubePlayer youTubePlayer, PlayerConstants.PlayerState state) {
    newSeekBarProgress = -1;
    updateState(state);
  }

  @Override
  public void onCurrentSecond(YouTubePlayer youTubePlayer, float second) {
    // ignore if the user is currently moving the SeekBar
    if (seekBarTouchStarted) {
      return;
    }
    // ignore if the current time is older than what the user selected with the SeekBar
    if (newSeekBarProgress > 0 && !TimeUtilities.formatTime(second).equals(TimeUtilities.formatTime(newSeekBarProgress))) {
      return;
    }
    newSeekBarProgress = -1;

    seekBar.setProgress((int) second);
  }

  @Override
  public void onVideoDuration(YouTubePlayer youTubePlayer, float duration) {
    videoDurationTextView.setText(TimeUtilities.formatTime(duration));
    seekBar.setMax((int) duration);
  }

  @Override
  public void onVideoLoadedFraction(YouTubePlayer youTubePlayer, float loadedFraction) {
    seekBar.setSecondaryProgress(showBufferingProgress ? (int) (loadedFraction * seekBar.getMax()) : 0);
  }

  @Override
  public void onReady(YouTubePlayer youTubePlayer) {

  }

  @Override
  public void onPlaybackQualityChange(YouTubePlayer youTubePlayer, PlayerConstants.PlaybackQuality playbackQuality) {

  }

  @Override
  public void onPlaybackRateChange(YouTubePlayer youTubePlayer, PlayerConstants.PlaybackRate playbackRate) {

  }

  @Override
  public void onError(YouTubePlayer youTubePlayer, PlayerConstants.PlayerError error) {

  }

  @Override
  public void onVideoId(YouTubePlayer youTubePlayer, String videoId) {

  }

  @Override
  public void onApiChange(YouTubePlayer youTubePlayer) {

  }

  public interface YouTubePlayerSeekBarListener {

    void seekTo(float time);
  }
}
