package com.zoyi.channel.plugin.android.global;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.com.annimon.stream.Stream;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.subjects.PublishSubject;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public class Action {

  /* static fields */

  private static PublishSubject<ActionType> publishSubject = PublishSubject.create();

  private static Map<ActionType, RestSubscriber<?>> apiSubscribers = new ConcurrentHashMap<>();

  private static Map<ActionType, Set<RestSubscriber>> canceller = new ConcurrentHashMap<>();

  static <E> Subscription apiSubscribe(
      Observable<E> observable,
      RestSubscriber<E> subscriber,
      @Nullable ActionType targetActionType,
      @Nullable ActionType[] cancelActionTypes
  ) {
    Subscription subscription = observable.subscribe(subscriber);

    subscriber.setActionSubscription(targetActionType, cancelActionTypes);

    if (targetActionType != null) {
      cancelInternal(targetActionType);

      apiSubscribers.put(targetActionType, subscriber);
    }

    if (cancelActionTypes != null) {
      for (ActionType cancelType : cancelActionTypes) {
        if (canceller.get(cancelType) == null) {
          canceller.put(cancelType, new HashSet<>());
        }

        Set<RestSubscriber> cancelSet = canceller.containsKey(cancelType) ? canceller.get(cancelType) : null;

        if (cancelSet != null) {
          cancelSet.add(subscriber);
        }
      }
    }

    return subscription;
  }

  private static void cancelInternal(ActionType actionType) {
    // cancel target action

    if (apiSubscribers != null) {
      RestSubscriber subscriber = apiSubscribers.get(actionType);

      if (subscriber != null) {
        if (!subscriber.isUnsubscribed()) {
          subscriber.unsubscribe();
        }

        if (canceller != null && subscriber.getCancelTypes().length > 0) {
          for (ActionType cancelType : subscriber.getCancelTypes()) {
            Set<RestSubscriber> cancelSet = canceller.get(cancelType);

            if (cancelSet != null) {
              cancelSet.remove(subscriber);
            }
          }
        }

        apiSubscribers.remove(actionType);
      }
    }

    // cancel by cancel action types

    if (canceller != null) {
      Set<RestSubscriber> cancelSet = canceller.get(actionType);

      if (cancelSet != null) {
        for (RestSubscriber subscriber : cancelSet) {
          if (subscriber != null) {
            if (!subscriber.isUnsubscribed()) {
              subscriber.unsubscribe();
            }
            if (subscriber.getActionType() != null) {
              apiSubscribers.remove(subscriber.getActionType());
            }
          }
        }
        cancelSet.clear();
      }
    }
  }

  private void invokeInternal(ActionType actionType) {
    cancelInternal(actionType);

    if (publishSubject != null) {
      publishSubject.onNext(actionType);
    }
  }

  private boolean isRunningInternal(ActionType actionType) {
    if (apiSubscribers.containsKey(actionType)) {
      Subscription subscription = apiSubscribers.get(actionType);

      return subscription != null && !subscription.isUnsubscribed();
    }
    return false;
  }

  private Observable<ActionType> observableInternal() {
    return publishSubject
        .onBackpressureBuffer()
        .observeOn(AndroidSchedulers.mainThread());
  }

  private void removeSubscriberInternal(RestSubscriber<?> subscriber) {
    if (subscriber.getActionType() != null) {
      apiSubscribers.remove(subscriber.getActionType());
    }

    if (subscriber.getCancelTypes().length > 0) {
      Stream.of(subscriber.getCancelTypes()).forEach(actionType -> {
        Set<RestSubscriber> set = canceller.get(actionType);

        if (set != null) {
          set.remove(subscriber);
        }
      });
    }
  }

  private void releaseInternal() {
    for (Subscription subscription : apiSubscribers.values()) {
      if (subscription != null && !subscription.isUnsubscribed()) {
        subscription.unsubscribe();
      }
    }
    apiSubscribers.clear();
    canceller.clear();
  }

  // static area

  @Nullable
  private static Action action;

  @NonNull
  private synchronized static Action getAction() {
    if (action == null) {
      action = new Action();
    }
    return action;
  }

  public static void invoke(ActionType actionType) {
    getAction().invokeInternal(actionType);
  }

  static boolean isRunning(ActionType actionType) {
    return getAction().isRunningInternal(actionType);
  }

  public static Observable<ActionType> observable() {
    return getAction().observableInternal();
  }

  public static void removeSubscriber(RestSubscriber<?> subscriber) {
    getAction().removeSubscriberInternal(subscriber);
  }

  public static void release() {
    getAction().releaseInternal();
  }
}
