package com.zoyi.channel.plugin.android.view.layout.message;

import android.content.Context;
import android.content.res.TypedArray;
import android.support.annotation.*;
import android.text.SpannableStringBuilder;
import android.util.AttributeSet;
import android.widget.LinearLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.Block;
import com.zoyi.channel.plugin.android.selector.StyleSelector;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.com.annimon.stream.Stream;

import java.util.List;

public class TextMessageView extends LinearLayout {

  private static final int MODE_NORMAL = 0;
  private static final int MODE_SINGLE_VIEW = 1;
  private static final int MODE_PLAIN = 2;

  private int renderMode = 0;
  private int maxLines = Integer.MAX_VALUE;

  private int textSize = 15;
  private int bigTextSize = 54;
  private int lineSpacingExtra = 5;

  @ColorInt
  private int textColor;

  @ColorInt
  private int textLinkColor;

  public TextMessageView(Context context) {
    super(context);
    init(context, null);
  }

  public TextMessageView(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context, attrs);
  }

  public TextMessageView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context, attrs);
  }

  @Initializer
  private void init(Context context, @Nullable AttributeSet attrs) {
    setOrientation(VERTICAL);

    if (attrs != null) {
      TypedArray typedArray = context.getTheme().obtainStyledAttributes(
          attrs,
          R.styleable.TextMessageView,
          0,
          0);

      try {
        textColor = typedArray.getColor(
            R.styleable.TextMessageView_ch_tmv_textColor,
            StyleSelector.getTextColor()
        );

        textLinkColor = typedArray.getColor(
            R.styleable.TextMessageView_ch_tmv_textLinkColor,
            StyleSelector.getTextColor()
        );

        textSize = typedArray.getDimensionPixelSize(
            R.styleable.TextMessageView_ch_tmv_textSize,
            DimenUtils.spToPx(context, textSize)
        );

        bigTextSize = typedArray.getDimensionPixelSize(
            R.styleable.TextMessageView_ch_tmv_bigTextSize,
            DimenUtils.spToPx(context, bigTextSize)
        );

        lineSpacingExtra = typedArray.getDimensionPixelSize(
            R.styleable.TextMessageView_ch_tmv_lineSpacingExtra,
            lineSpacingExtra
        );

        renderMode = typedArray.getInt(
            R.styleable.TextMessageView_ch_tmv_renderMode,
            renderMode
        );

        maxLines = typedArray.getInt(
            R.styleable.TextMessageView_ch_tmv_maxLines,
            maxLines
        );
      } finally {
        typedArray.recycle();
      }
    }
  }

  public void setMaxLines(int maxLines) {
    this.maxLines = maxLines;
  }

  public void setBlocks(@NonNull List<Block> blocks) {
    removeAllViews();

    switch (renderMode) {
      case MODE_NORMAL:
        renderNormal(blocks);
        break;
      case MODE_SINGLE_VIEW:
      case MODE_PLAIN:
        renderSingle(blocks);
        break;
    }
  }

  private void renderNormal(@NonNull List<Block> blocks) {
    for (Block block : blocks) {
      Optional.ofNullable(block.getType()).ifPresent(type -> {
        switch (block.getType()) {
          case Const.BLOCK_TYPE_TEXT:
          case Const.BLOCK_TYPE_CODE:
            addTextBlock(block);
            break;
          case Const.BLOCK_TYPE_BULLETS:
            addBulletsBlock(block);
            break;
        }
      });
    }
  }

  private void renderSingle(@NonNull List<Block> blocks) {
    SpannableStringBuilder spannableStringBuilder = new SpannableStringBuilder();

    for (Block block : blocks) {
      if (spannableStringBuilder.length() > 0) {
        spannableStringBuilder.append("\n");
      }

      Optional.ofNullable(block.getType()).ifPresent(type -> {
        switch (block.getType()) {
          case Const.BLOCK_TYPE_TEXT:
          case Const.BLOCK_TYPE_CODE:
            Optional.ofNullable(block.getFormattedSpanMessage())
                .ifPresentOrElse(spannableStringBuilder::append, () -> spannableStringBuilder.append("\n"));
            break;
          case Const.BLOCK_TYPE_BULLETS:
            Stream.ofNullable(block.getBlocks()).forEach(textBlock -> {
              spannableStringBuilder.append("• ");
              spannableStringBuilder.append(textBlock.getFormattedSpanMessage());
              spannableStringBuilder.append("\n");
            });
            break;
        }
      });
    }
    addTextBlock(null, spannableStringBuilder, false);
  }

  public void setText(@NonNull CharSequence charSequence) {
    removeAllViews();

    addTextBlock(null, charSequence, false);
  }

  private void addTextBlock(Block block) {
    CharSequence charSequence = Optional.ofNullable(block.getFormattedSpanMessage()).orElse(null);

    addTextBlock(null, charSequence, block.hasOnlyEmoji());
  }

  private void addBulletsBlock(Block block) {
    Stream.ofNullable(block.getBlocks())
        .filter(textBlock -> Const.BLOCK_TYPE_TEXT.equals(textBlock.getType()) && textBlock.getFormattedSpanMessage() != null)
        .map(Block::getFormattedSpanMessage)
        .forEach(formattedSpanMessage -> addTextBlock("• ", formattedSpanMessage, false));
  }

  private void addTextBlock(@Nullable String prefix, CharSequence charSequence, boolean bigText) {
    addView(new TextBlockView(getContext())
        .setPrefix(prefix)
        .setTextSize(bigText ? bigTextSize : textSize)
        .setLineSpacingExtra(lineSpacingExtra)
        .setTextColor(textColor)
        .setTextLinkColor(textLinkColor)
        .setPlainText(this.renderMode == MODE_PLAIN)
        .setMaxLines(maxLines)
        .build(charSequence));
  }
}
