package com.zoyi.channel.plugin.android.view.youtube.player.views;


import android.arch.lifecycle.Lifecycle;
import android.arch.lifecycle.LifecycleObserver;
import android.arch.lifecycle.OnLifecycleEvent;
import android.content.Context;
import android.content.res.TypedArray;
import android.support.annotation.LayoutRes;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.view.youtube.player.YouTubePlayer;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.AbstractYouTubePlayerListener;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerCallback;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerFullScreenListener;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerListener;
import com.zoyi.channel.plugin.android.view.youtube.player.option.IFramePlayerOptions;
import com.zoyi.channel.plugin.android.view.youtube.player.util.FullScreenHelper;
import com.zoyi.channel.plugin.android.view.youtube.player.util.YouTubePlayerUtils;
import com.zoyi.channel.plugin.android.view.youtube.ui.PlayerUiController;


public class CHYouTubePlayerView extends SixteenByNineFrameLayout implements LifecycleObserver {

  private LegacyYouTubePlayerView legacyTubePlayerView;
  private FullScreenHelper fullScreenHelper;

  private boolean enableAutomaticInitialization;

  public CHYouTubePlayerView(Context context) {
    this(context, null);
  }

  public CHYouTubePlayerView(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context, attrs);
  }

  public CHYouTubePlayerView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context, attrs);
  }

  private void init(Context context, @Nullable AttributeSet attrs) {
    legacyTubePlayerView = new LegacyYouTubePlayerView(context);
    fullScreenHelper = new FullScreenHelper(this);

    addView(legacyTubePlayerView, new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));

    boolean autoPlay = false;
    boolean handleNetworkEvents = true;
    String videoId = null;

    boolean useWebUi = false;
    boolean enableLiveVideoUi = false;
    boolean showYouTubeButton = true;
    boolean showFullScreenButton = true;
    boolean showVideoCurrentTime = true;
    boolean showVideoDuration = true;
    boolean showSeekBar = true;

    if (attrs != null) {
      TypedArray typedArray = context.getTheme().obtainStyledAttributes(attrs, R.styleable.CHYouTubePlayerView, 0, 0);

      enableAutomaticInitialization = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_enableAutomaticInitialization, true);
      autoPlay = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_autoPlay, autoPlay);
      handleNetworkEvents = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_handleNetworkEvents, handleNetworkEvents);
      videoId = typedArray.getString(R.styleable.CHYouTubePlayerView_ch_videoId);

      useWebUi = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_useWebUi, useWebUi);
      enableLiveVideoUi = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_enableLiveVideoUi, enableLiveVideoUi);
      showYouTubeButton = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_showYouTubeButton, showYouTubeButton);
      showFullScreenButton = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_showFullScreenButton, showFullScreenButton);
      showVideoCurrentTime = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_showVideoCurrentTime, showVideoCurrentTime);
      showVideoDuration = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_showVideoDuration, showVideoDuration);
      showSeekBar = typedArray.getBoolean(R.styleable.CHYouTubePlayerView_ch_showSeekBar, showSeekBar);

      typedArray.recycle();
    }

    if (!enableAutomaticInitialization && useWebUi) {
      throw new IllegalStateException("YouTubePlayerView: 'enableAutomaticInitialization' is false and 'useWebUi' is set to true. " +
          "This is not possible, if you want to manually initialize YouTubePlayerView and use the web ui, " +
          "you should manually initialize the YouTubePlayerView using 'initializeWithWebUi'");
    }

    if (videoId == null && autoPlay) {
      throw new IllegalStateException(
          "YouTubePlayerView: videoId is not set but autoPlay is set to true. This combination is not possible.");
    }

    if (!useWebUi) {
      legacyTubePlayerView.getPlayerUiController()
          .enableLiveVideoUi(enableLiveVideoUi)
          .showYouTubeButton(showYouTubeButton)
          .showFullscreenButton(showFullScreenButton)
          .showCurrentTime(showVideoCurrentTime)
          .showDuration(showVideoDuration)
          .showSeekBar(showSeekBar);
    }

    final String finalVideoId = videoId;
    final boolean finalAutoPlay = autoPlay;

    AbstractYouTubePlayerListener youTubePlayerListener = new AbstractYouTubePlayerListener() {

      @Override
      public void onReady(YouTubePlayer youTubePlayer) {
        if (finalVideoId != null) {
          YouTubePlayerUtils.loadOrCueVideo(youTubePlayer, legacyTubePlayerView.isCanPlay() && finalAutoPlay, finalVideoId, 0f);
          youTubePlayer.removeListener(this);

        }
      }
    };

    if (enableAutomaticInitialization) {
      if (useWebUi) {
        legacyTubePlayerView.initializeWithWebUi(context, youTubePlayerListener, handleNetworkEvents);
      } else {
        legacyTubePlayerView.initialize(context, youTubePlayerListener, handleNetworkEvents);
      }
    }

    legacyTubePlayerView.addFullScreenListener(new YouTubePlayerFullScreenListener() {
      @Override
      public void onYouTubePlayerEnterFullScreen() {
        fullScreenHelper.enterFullScreen();
      }

      @Override
      public void onYouTubePlayerExitFullScreen() {
        fullScreenHelper.exitFullScreen();
      }
    });
  }


  /**
   * Initialize the player. You must call this method before using the player.
   *
   * @param youTubePlayerListener listener for player events
   * @param handleNetworkEvents   if set to true a broadcast receiver will be registered and network events will be handled automatically.
   *                              If set to false, you should handle network events with your own broadcast receiver.
   * @param playerOptions         customizable options for the embedded video player, can be null.
   */
  public void initialize(Context context, YouTubePlayerListener youTubePlayerListener, boolean handleNetworkEvents,
      IFramePlayerOptions playerOptions) {
    if (enableAutomaticInitialization) {
      throw new IllegalStateException(
          "YouTubePlayerView: If you want to initialize this view manually, you need to set 'enableAutomaticInitialization' to false"
      );
    } else {
      legacyTubePlayerView.initialize(context, youTubePlayerListener, handleNetworkEvents, playerOptions);
    }
  }

  /**
   * Initialize the player.
   *
   * @param handleNetworkEvents if set to true a broadcast receiver will be registered and network events will be handled automatically. If
   *                            set to false, you should handle network events with your own broadcast receiver.
   * @see CHYouTubePlayerView.initialize
   */
  public void initialize(Context context, YouTubePlayerListener youTubePlayerListener, boolean handleNetworkEvents) {
    if (enableAutomaticInitialization) {
      throw new IllegalStateException(
          "YouTubePlayerView: If you want to initialize this view manually, you need to set 'enableAutomaticInitialization' to false"
      );
    } else {
      legacyTubePlayerView.initialize(context, youTubePlayerListener, handleNetworkEvents, null);
    }
  }

  /**
   * Initialize the player. Network events are automatically handled by the player.
   *
   * @param youTubePlayerListener listener for player events
   * @see CHYouTubePlayerView.initialize
   */
  public void initialize(Context context, YouTubePlayerListener youTubePlayerListener) {
    if (enableAutomaticInitialization) {
      throw new IllegalStateException(
          "YouTubePlayerView: If you want to initialize this view manually, you need to set 'enableAutomaticInitialization' to false"
      );
    } else {
      legacyTubePlayerView.initialize(context, youTubePlayerListener, true);
    }
  }

  /**
   * Initialize a player using the web-base Ui instead pf the native Ui. The default PlayerUiController will be removed and
   * [YouTubePlayerView.getPlayerUiController] will throw exception.
   *
   * @see CHYouTubePlayerView.initialize
   */
  public void initializeWithWebUi(Context context, YouTubePlayerListener youTubePlayerListener, boolean handleNetworkEvents) {
    if (enableAutomaticInitialization) {
      throw new IllegalStateException(
          "YouTubePlayerView: If you want to initialize this view manually, you need to set 'enableAutomaticInitialization' to false"
      );
    } else {
      legacyTubePlayerView.initializeWithWebUi(context, youTubePlayerListener, handleNetworkEvents);
    }
  }

  /**
   * @param youTubePlayerCallback A callback that will be called when the YouTubePlayer is ready. If the player is ready when the function
   *                              is called, the callback is called immediately. This function is called only once.
   */
  public void getYouTubePlayerWhenReady(YouTubePlayerCallback youTubePlayerCallback) {
    legacyTubePlayerView.getYouTubePlayerWhenReady(youTubePlayerCallback);
  }

  /**
   * Use this method to replace the default Ui of the player with a custom Ui.
   * <p>
   * You will be responsible to manage the custom Ui from your application, the default controller obtained through
   * [YouTubePlayerView.getPlayerUiController] won't be available anymore.
   *
   * @param layoutId the ID of the layout defining the custom Ui.
   * @return The inflated View
   */
  public View inflateCustomPlayerUi(Context context, @LayoutRes int layoutId) {
    return legacyTubePlayerView.inflateCustomPlayerUi(context, layoutId);
  }

  public PlayerUiController getPlayerUiController() {
    return legacyTubePlayerView.getPlayerUiController();
  }

  /**
   * Don't use this method if you want to publish your app on the PlayStore. Background playback is against YouTube terms of service.
   */
  public void enableBackgroundPlayback(boolean enable) {
    legacyTubePlayerView.enableBackgroundPlayback(enable);
  }

  /**
   * Call this method before destroying the host Fragment/Activity, or register this View as an observer of its host lifecycle
   */
  @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
  public void release() {
    legacyTubePlayerView.release();
  }

  @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
  private void onResume() {
    legacyTubePlayerView.onResume();
  }

  @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
  private void onStop() {
    legacyTubePlayerView.onStop();
  }

  public void addYouTubePlayerListener(YouTubePlayerListener youTubePlayerListener) {
    legacyTubePlayerView.getYouTubePlayer().addListener(youTubePlayerListener);

  }

  public void removeYouTubePlayerListener(YouTubePlayerListener youTubePlayerListener) {
    legacyTubePlayerView.getYouTubePlayer().removeListener(youTubePlayerListener);
  }

  public void enterFullScreen() {
    legacyTubePlayerView.enterFullScreen();
  }

  public void exitFullScreen() {
    legacyTubePlayerView.exitFullScreen();
  }

  public void toggleFullScreen() {
    legacyTubePlayerView.toggleFullScreen();
  }

  public boolean isFullScreen() {
    return fullScreenHelper.isFullScreen();
  }

  public boolean addFullScreenListener(YouTubePlayerFullScreenListener fullScreenListener) {
    return fullScreenHelper.addFullScreenListener(fullScreenListener);
  }

  public boolean removeFullScreenListener(YouTubePlayerFullScreenListener fullScreenListener) {
    return fullScreenHelper.removeFullScreenListener(fullScreenListener);
  }
}