package com.zoyi.channel.plugin.android.view.youtube.player.option;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.json.JSONException;
import org.json.JSONObject;

public class IFramePlayerOptions {

  private JSONObject playerOptions;

  public static IFramePlayerOptions DEFAULT = new Builder().build();

  public IFramePlayerOptions(JSONObject playerOptions) {
    this.playerOptions = playerOptions;
  }

  @NonNull
  @Override
  public String toString() {
    return playerOptions.toString();
  }

  @Nullable
  public String getOrigin() {
    try {
      return playerOptions.getString(Builder.ORIGIN);
    } catch (JSONException e) {
      e.printStackTrace();
      return null;
    }
  }

  public static class Builder {

    private static final String AUTO_PLAY = "autoplay";
    private static final String CONTROLS = "controls";
    private static final String ENABLE_JS_API = "enablejsapi";
    private static final String FS = "fs";
    private static final String REL = "rel";
    private static final String SHOW_INFO = "showinfo";
    private static final String IV_LOAD_POLICY = "iv_load_policy";
    private static final String MODEST_BRANDING = "modestbranding";
    private static final String CC_LOAD_POLICY = "cc_load_policy";
    private static final String CC_LANG_PREF = "cc_lang_pref";

    public static final String ORIGIN = "origin";

    private JSONObject builderOptions = new JSONObject();

    public Builder() {
      addInt(AUTO_PLAY, 0);
      addInt(CONTROLS, 0);
      addInt(ENABLE_JS_API, 1);
      addInt(FS, 0);
      addString(ORIGIN, "https://www.youtube.com");
      addInt(REL, 0);
      addInt(SHOW_INFO, 0);
      addInt(IV_LOAD_POLICY, 3);
      addInt(MODEST_BRANDING, 1);
      addInt(CC_LOAD_POLICY, 0);
    }

    public IFramePlayerOptions build() {
      return new IFramePlayerOptions(builderOptions);
    }

    /**
     * Controls whether the web-based UI of the IFrame player is used or not.
     *
     * @param controls If set to 0: web UI is not used. If set to 1: web UI is used.
     */
    public Builder controls(Integer controls) {
      addInt(CONTROLS, controls);
      return this;
    }

    /**
     * Controls the related videos shown at the end of a video.
     *
     * @param rel If set to 0: related videos will come from the same channel as the video that was just played. If set to 1: related videos
     *            will come from multiple channels.
     */
    public Builder rel(Integer rel) {
      addInt(REL, rel);
      return this;
    }

    /**
     * Controls video annotations.
     *
     * @param ivLoadPolicy if set to 1: the player will show video annotations. If set to 3: they player won't show video annotations.
     */
    public Builder ivLoadPolicy(Integer ivLoadPolicy) {
      addInt(IV_LOAD_POLICY, ivLoadPolicy);
      return this;
    }

    /**
     * This parameter specifies the default language that the player will use to display captions. If you use this parameter and also set
     * the cc_load_policy parameter to 1, then the player will show captions in the specified language when the player loads. If you do not
     * also set the cc_load_policy parameter, then captions will not display by default, but will display in the specified language if the
     * user opts to turn captions on.
     *
     * @param languageCode ISO 639-1 two-letter language code
     */
    public Builder langPref(String languageCode) {
      addString(CC_LANG_PREF, languageCode);
      return this;
    }


    /**
     * Controls video captions. It doesn't work with automatically generated captions.
     *
     * @param ccLoadPolicy if set to 1: the player will show captions. If set to 0: the player won't show captions.
     */
    public Builder ccLoadPolicy(Integer ccLoadPolicy) {
      addInt(CC_LOAD_POLICY, ccLoadPolicy);
      return this;
    }

    /**
     * Controls domain as the origin parameter value.
     *
     * @param origin your domain
     */
    public Builder origin(String origin) {
      addString(ORIGIN, origin);
      return this;
    }

    private void addString(String key, String value) {
      try {
        builderOptions.put(key, value);
      } catch (JSONException e) {
        throw new RuntimeException(String.format("Illegal JSON value %s: %s", key, value));
      }
    }

    private void addInt(String key, Integer value) {
      try {
        builderOptions.put(key, value);
      } catch (JSONException e) {
        throw new RuntimeException(String.format("Illegal JSON value %s: %s", key, value));
      }
    }
  }
}
