package com.zoyi.channel.plugin.android.activity.base;

import android.app.Dialog;
import android.content.Intent;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.AnimRes;
import android.support.annotation.ColorInt;
import android.support.annotation.LayoutRes;
import android.support.annotation.Nullable;
import android.support.v7.app.AppCompatActivity;
import android.view.View;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.contract.BasePresenter;
import com.zoyi.channel.plugin.android.contract.BaseView;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.global.Action;
import com.zoyi.channel.plugin.android.selector2.PluginSelector;
import com.zoyi.channel.plugin.android.store2.binder.Binder;
import com.zoyi.channel.plugin.android.util.ProgressHelper;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.rx.Subscription;

public abstract class BaseActivity2 extends AppCompatActivity implements BaseView {

  @Nullable
  private Binder pluginBinder;

  @Nullable
  private Subscription actionSubscription;

  @Nullable
  private BasePresenter presenter;

  private Transition outTransition = Transition.SLIDE_FROM_RIGHT;

  @ColorInt
  private int backColor = Color.BLACK;

  private boolean backColorFixed = false;

  @Nullable
  private Dialog dialog;

  @Override
  protected final void onCreate(@Nullable Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);

    if (savedInstanceState != null) {
      finish();
      return;
    }

    if (onCreate()) {
      handlePluginStyle();

      actionSubscription = Action.observable().subscribe(actionType -> {
        if (actionType == ActionType.SHUTDOWN) {
          finish(Transition.NONE);
        }
      });
    } else {
      finish();
    }
  }

  @Override
  protected void onDestroy() {
    if (pluginBinder != null) {
      pluginBinder.unbind();
      pluginBinder = null;
    }
    if (presenter != null) {
      presenter.release();
      presenter = null;
    }
    if (actionSubscription != null) {
      if (!actionSubscription.isUnsubscribed()) {
        actionSubscription.unsubscribe();
      }
      actionSubscription = null;
    }
    super.onDestroy();
  }

  protected void bindPresenter(BasePresenter presenter) {
    this.presenter = presenter;
    this.presenter.init();
  }

  private void handlePluginStyle() {
    if (!backColorFixed) {
      pluginBinder = PluginSelector.bindPlugin(plugin -> {
        if (plugin != null && !backColorFixed) {
          setStatusBarColor(Color.TRANSPARENT, plugin.getTextColor() == Color.BLACK);
        }
      });
    }
  }

  protected void init(@LayoutRes int layoutResId) {
    setContentView(layoutResId);
  }

  protected void init(@LayoutRes int layoutResId, @ColorInt int backColor, boolean lightMode) {
    init(layoutResId);

    this.backColor = backColor;
    this.backColorFixed = true;

    setStatusBarColor(backColor, lightMode);
  }

  private void setStatusBarColor(@ColorInt int backColor, boolean lightMode) {
    try {
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
        getWindow().setStatusBarColor(backColor);

        int systemUiVisibility = getWindow().getDecorView().getSystemUiVisibility();

        if (lightMode) {
          getWindow().getDecorView().setSystemUiVisibility(systemUiVisibility | View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
        } else {
          getWindow().getDecorView().setSystemUiVisibility(systemUiVisibility & ~View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
        }
      }
    } catch (Exception ignored) {
    }
  }

  protected abstract boolean onCreate();

  public void setOutTransition(Transition outTransition) {
    this.outTransition = outTransition;
  }

  @Override
  public void finish() {
    super.finish();
    overridePendingTransition(getEnterAnimOfFinish(), getExistAnimOfFinish());
  }

  @Override
  public void finish(Transition transition) {
    this.outTransition = transition;
    finish();
  }

  @Override
  public void finish(int resultCode) {
    setResult(resultCode);
    finish();
  }

  @Override
  public void finish(int resultCode, Transition transition) {
    setResult(resultCode);
    this.outTransition = transition;
    finish();
  }

  @AnimRes
  private int getEnterAnimOfFinish() {
    return R.anim.ch_plugin_idle;
  }

  @AnimRes
  private int getExistAnimOfFinish() {
    switch (outTransition) {
      case NONE:
        return R.anim.ch_plugin_idle;

      case SLIDE_FROM_BOTTOM:
        return R.anim.ch_plugin_slide_out_bottom;

      default:
        return R.anim.ch_plugin_slide_out_right;
    }
  }

  // intent getter

  @Nullable
  public String getString(String key) {
    Intent intent = getIntent();

    if (intent != null) {
      return intent.getStringExtra(key);
    }
    return null;
  }

  // progress

  @Override
  public void showProgress() {
    showProgress(ResUtils.getString(this, "ch.settings.changing_message"));
  }

  @Override
  public void showProgress(String message) {
    hideProgress();

    dialog = ProgressHelper.show(
        this,
        message,
        false
    );
  }

  @Override
  public void hideProgress() {
    if (dialog != null && dialog.isShowing()) {
      dialog.dismiss();
      dialog = null;
    }
  }
}
