package com.zoyi.channel.plugin.android.activity.lounge.view;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.Toast;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.action.ChannelAction;
import com.zoyi.channel.plugin.android.enumerate.FetchState;
import com.zoyi.channel.plugin.android.selector2.ChannelSelector;
import com.zoyi.channel.plugin.android.store2.binder.Binder;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.util.rxpermission.RxPermissions;
import com.zoyi.channel.plugin.android.view.layout.FetchLayout;

public class LoungeIntegrationsView extends FetchLayout {

  private Context context;

  private View buttonKakao;
  private View buttonLine;
  private View buttonLink;
  private View buttonPhone;

  private RxPermissions rxPermissions;

  @Nullable
  private String kakaoLink;

  @Nullable
  private String lineLink;

  @Nullable
  private String defaultPluginLink;

  @Nullable
  private String phoneNumber;

  @Nullable
  private Binder binder;

  public LoungeIntegrationsView(@NonNull Context context) {
    super(context);
    init(context);
  }

  public LoungeIntegrationsView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public LoungeIntegrationsView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  private void init(Context context) {
    this.context = context;
    this.rxPermissions = new RxPermissions((Activity) context);

    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_view_lounge_integrations, this, true);

    buttonKakao = view.findViewById(R.id.buttonIntegrationsKakao);
    buttonKakao.setOnClickListener(v -> openUrl(kakaoLink));

    buttonLine = view.findViewById(R.id.buttonIntegrationsLine);
    buttonLine.setOnClickListener(v -> openUrl(lineLink));

    buttonLink = view.findViewById(R.id.buttonIntegrationsLink);
    buttonLink.setOnClickListener(v -> copyText(defaultPluginLink));

    buttonPhone = view.findViewById(R.id.buttonIntegrationsPhone);
    buttonPhone.setOnClickListener(v -> call(phoneNumber));
  }

  @Override
  protected void onAttachedToWindow() {
    super.onAttachedToWindow();

    binder = ChannelSelector.bindIntegrations((fetchState, channel, plugin, repo) -> {
      switch (fetchState) {
        case LOADING:
        case FAILED:
          setVisibility(View.VISIBLE);
          setFetchState(fetchState);
          break;

        case COMPLETE:
          int activeCount = 0;

          // kakao
          if (repo != null && repo.getAppKakaoLink() != null) {
            this.kakaoLink = repo.getAppKakaoLink();
            this.buttonKakao.setVisibility(View.VISIBLE);

            activeCount++;
          } else {
            this.kakaoLink = null;
            this.buttonKakao.setVisibility(View.GONE);
          }

          // line
          if (repo != null && repo.getAppLineLink() != null) {
            this.lineLink = repo.getAppLineLink();
            this.buttonLine.setVisibility(View.VISIBLE);

            activeCount++;
          } else {
            this.lineLink = null;
            this.buttonLine.setVisibility(View.GONE);
          }

          // default link
          if (channel != null &&
              plugin != null &&
              channel.getDomain() != null &&
              channel.getDefaultPluginId() != null &&
              channel.getDefaultPluginId().equals(plugin.getId())
          ) {
            this.defaultPluginLink = String.format("https://%s.channel.io", channel.getDomain());
            this.buttonLink.setVisibility(View.VISIBLE);

            activeCount++;
          } else {
            this.defaultPluginLink = null;
            this.buttonLink.setVisibility(View.GONE);
          }

          // phone
          if (channel != null && channel.getPhoneNumber() != null) {
            this.phoneNumber = channel.getPhoneNumber();
            this.buttonPhone.setVisibility(View.VISIBLE);

            activeCount++;
          } else {
            this.phoneNumber = null;
            this.buttonPhone.setVisibility(View.GONE);
          }

          Views.setVisibility(this, activeCount > 0);
          setFetchState(FetchState.COMPLETE);
          break;
      }
    });
  }

  private void openUrl(@Nullable String url) {
    if (url != null) {
      IntentUtils.setUrl(context, url).startActivity();
    }
  }

  private void copyText(@Nullable String text) {
    if (text != null) {
      if (ClipboardUtils.copyToClipBoard(text)) {
        Toast.makeText(context, ResUtils.getString("ch.integrations.copy_link.success"), Toast.LENGTH_SHORT).show();
      }
    }
  }

  private void call(@Nullable String number) {
    if (number != null && context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_TELEPHONY)) {
      rxPermissions.request(Manifest.permission.CALL_PHONE).subscribe(granted -> {
        if (granted) {
          Executor.call(context, number);
        } else {
          showPermissionDeniedToast();
        }
      });
    } else {
      showPermissionDeniedToast();
    }
  }

  private void showPermissionDeniedToast() {
    Toast.makeText(
        context,
        ResUtils.getString("ch.permission.denied"),
        Toast.LENGTH_LONG
    ).show();
  }

  @Override
  protected void onDetachedFromWindow() {
    if (binder != null) {
      binder.unbind();
      binder = null;
    }
    super.onDetachedFromWindow();
  }

  @Override
  protected void onRefreshClick() {
    ChannelAction.fetchMessengers();
  }
}
