package com.zoyi.channel.plugin.android.activity.chat3;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.view.ViewGroup;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageContentClickListener;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.*;
import com.zoyi.channel.plugin.android.activity.chat3.contract.ChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.chat3.listener.OnProfileUpdateRequestListener;
import com.zoyi.channel.plugin.android.activity.chat3.model.ProfileBotMessageItem;
import com.zoyi.channel.plugin.android.activity.chat3.viewholder.ProfileBotMessageHolder;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.util.CompareUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class ChatAdapter
    extends RecyclerView.Adapter<RecyclerView.ViewHolder>
    implements ChatAdapterContract.View, ChatAdapterContract.Model {

  private SortedList<MessageItem> items;

  @Nullable
  private OnProfileUpdateRequestListener onProfileUpdateRequestListener;
  @Nullable
  private OnMessageContentClickListener onMessageContentClickListener;

  @NonNull
  private TypingItem typingItem;

  @Nullable
  private InitMessageItem initMessageItem = null;
  @Nullable
  private DateItem initDateItem = null;

  public ChatAdapter() {
    items = new SortedList<>(MessageItem.class, new SortedListCallback<>(this, false));

    // Typing layout must always exists
    typingItem = new TypingItem();
    items.add(typingItem);
  }

  public void setOnProfileUpdateRequestListener(
      @Nullable OnProfileUpdateRequestListener onProfileUpdateRequestListener
  ) {
    this.onProfileUpdateRequestListener = onProfileUpdateRequestListener;
  }

  public void setOnMessageContentClickListener(@Nullable OnMessageContentClickListener onMessageContentClickListener) {
    this.onMessageContentClickListener = onMessageContentClickListener;
  }

  @Override
  public void setInitMessage(InitMessageItem item) {
    if (item != null && initMessageItem == null) {
      initMessageItem = item;
      initDateItem = new DateItem(item.getCreatedAt());

      items.beginBatchedUpdates();

      items.add(item);
      items.add(initDateItem);

      items.endBatchedUpdates();
    }
  }

  @Override
  public void setMessages(Collection<Message> messages, Collection<SendingMessageItem> unsentItems, @Nullable Session lastReadSession) {
    List<MessageItem> newItems = purifyMessages(messages, lastReadSession);
    newItems.add(typingItem);
    newItems.addAll(unsentItems);

    items.replaceAll(newItems);
  }

  @Override
  public void replaceInitMessageItem(Message message) {
    if (initMessageItem != null) {
      items.remove(initMessageItem);
    }
    if (initDateItem != null) {
      items.remove(initDateItem);
    }
    if (message != null) {
      addMessages(Collections.singleton(message), null);
    }
  }

  @Override
  public void addMessages(Collection<Message> messages, @Nullable Session lastReadSession) {
    List<MessageItem> newItems = purifyMessages(messages, lastReadSession);

    items.beginBatchedUpdates();

    for (MessageItem item : newItems) {
      items.add(item);
    }

    for (Message message : messages) {
      if (message == null) {
        continue;
      }
      if (CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
        SendingMessageItem item = SendingMessageItem.createDummy(message.getRequestId());
        if (item != null) {
          items.remove(item);
        }
      }
    }

    items.endBatchedUpdates();
  }

  @Override
  public void addMessage(Message message) {
    addMessages(Collections.singleton(message), null);
  }

  @NonNull
  private List<MessageItem> purifyMessages(Collection<Message> messages, @Nullable Session session) {
    List<MessageItem> newItems = new ArrayList<>();

    long minTimestamp = Long.MAX_VALUE;
    long maxTimestamp = Long.MIN_VALUE;

    for (Message message : messages) {
      if (message == null) {
        continue;
      }

      minTimestamp = Math.min(minTimestamp, message.getCreatedAt());
      maxTimestamp = Math.max(maxTimestamp, message.getCreatedAt());

      if (message.getLog() != null) {
        newItems.add(new LogMessageItem(message));
        continue;
      }

      newItems.add(new ChatMessageItem(message));
      newItems.add(new DateItem(message.getCreatedAt()));

      if (message.getProfileBot() != null) {
        newItems.add(new ProfileBotMessageItem(message));
      }
    }

    if (session != null && session.getReadAt() != null && minTimestamp < session.getReadAt() && session.getReadAt() < maxTimestamp) {
      newItems.add(new NewMessageItem(session.getReadAt()));
    }

    return newItems;
  }

  @Override
  public void addMessageItem(MessageItem item) {
    items.add(item);
  }

  @Override
  public void addMessageItems(Collection<? extends MessageItem> items) {
    this.items.addAll((Collection<MessageItem>) items);
  }

  @Override
  public void removeMessageItem(MessageItem item) {
    items.remove(item);
  }

  @Override
  public void setTypings(List<Typing> typings) {
    addMessageItem(typingItem.setTypings(typings));
  }

  @Override
  public void replaceMessageItem(MessageItem newItem, MessageItem oldItem) {
    items.beginBatchedUpdates();

    items.remove(oldItem);
    items.add(newItem);

    items.endBatchedUpdates();
  }

  @Nullable
  @Override
  public RecyclerView.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
    MessageType type = MessageType.fromId(viewType);

    switch (type) {
      case NEW_MESSAGE_DIVIDER:
        return NewMessageHolder.newInstance(parent);

      case TYPING:
        return TypingHolder.newInstance(parent);

      case DATE:
        return DateDividerHolder.newInstance(parent);

      case LOG:
        return LogMessageHolder.newInstance(parent);

      case SENDING:
        return SendingMessageHolder.newInstance(parent, type, onMessageContentClickListener);

      case HOST:
      case GUEST:
        return ChatMessageHolder.newInstance(parent, type, onMessageContentClickListener);

      case PROFILE_BOT:
        return ProfileBotMessageHolder.newInstance(parent, onProfileUpdateRequestListener);

      case WELCOME:
        return WelcomeMessageHolder.newInstance(parent);

      case SUPPORT_BOT:
        return SupportBotMessageHolder.newInstance(parent, onMessageContentClickListener);

      case PUSH_BOT:
        return PushBotMessageHolder.newInstance(parent, onMessageContentClickListener);

      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case TYPING:
        TypingHolder typingHolder = (TypingHolder) holder;
        typingHolder.bind((TypingItem) item);
        break;

      case DATE:
        DateDividerHolder dateDividerHolder = (DateDividerHolder) holder;
        dateDividerHolder.bind((DateItem) item);
        break;

      case LOG:
        LogMessageHolder logMessageHolder = (LogMessageHolder) holder;
        logMessageHolder.bind((LogMessageItem) item);
        break;

      case SENDING:
        SendingMessageItem sendingMessageItem = (SendingMessageItem) item;
        SendingMessageHolder sendingHolder = (SendingMessageHolder) holder;
        sendingHolder.bind((SendingMessageItem) item, sendingMessageItem.isConnected(getItem(position - 1)));
        break;

      case HOST:
      case GUEST:
        chatMessageItem = (ChatMessageItem) item;
        ChatMessageHolder chatMessageHolder = (ChatMessageHolder) holder;
        chatMessageHolder.bind(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)),
            isLastPosition(position)
        );
        break;

      case PROFILE_BOT:
        ProfileBotMessageItem profileItem = (ProfileBotMessageItem) item;
        ProfileBotMessageHolder profileBotMessageHolder = (ProfileBotMessageHolder) holder;
        profileBotMessageHolder.bind(profileItem);
        break;

      case WELCOME:
        WelcomeMessageItem welcomeMessageItem = (WelcomeMessageItem) item;
        WelcomeMessageHolder welcomeMessageHolder = (WelcomeMessageHolder) holder;
        welcomeMessageHolder.bind(welcomeMessageItem);
        break;

      case SUPPORT_BOT:
        SupportBotMessageItem supportBotMessageItem = (SupportBotMessageItem) item;
        SupportBotMessageHolder supportBotMessageHolder = (SupportBotMessageHolder) holder;
        supportBotMessageHolder.bind(supportBotMessageItem, isLastPosition(position));
        break;

      case PUSH_BOT:
        PushBotMessageItem pushBotMessageItem = (PushBotMessageItem) item;
        PushBotMessageHolder pushBotMessageHolder = (PushBotMessageHolder) holder;
        pushBotMessageHolder.bind(pushBotMessageItem);
        break;
    }
  }

  @Nullable
  private MessageItem getItem(int position) {
    if (position >= 0 && position < items.size()) {
      return items.get(position);
    }
    return null;
  }

  private boolean isLastPosition(int position) {
    return position == getItemCount() - 2;
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Nullable
  @Override
  public Long getLastMessageItemKey() {
    MessageItem item = getItem(getItemCount() - 2);

    if (item != null) {
      return item.getPrimaryKey();
    }
    return null;
  }

  @Nullable
  @Override
  public Message getLastMessage() {
    for (int i = getItemCount(); i >= 0; i--) {
      MessageItem item = getItem(i);

      if (item != null && item.getType() == MessageType.HOST) {
        return ((ChatMessageItem) item).getMessage();
      }
    }
    return null;
  }
}
