package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.os.AsyncTask;
import android.os.Looper;
import android.support.annotation.Nullable;
import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.zoyi.channel.plugin.android.action.BootAction;
import com.zoyi.channel.plugin.android.action.EventAction;
import com.zoyi.channel.plugin.android.activity.chat3.manager.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.activity.lounge.LoungeActivity;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.global.Action;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store2.GlobalStore;
import com.zoyi.channel.plugin.android.store2.SettingsStore;
import com.zoyi.channel.plugin.android.store2.base.Store2;
import com.zoyi.channel.plugin.android.util.BootManager;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.L;

import java.util.concurrent.RejectedExecutionException;

import static com.zoyi.channel.plugin.android.ChannelIO.isInitializedChannelIO;

class RealChannelIO {

  private Application application;

  private Thread.UncaughtExceptionHandler uncaughtExceptionHandler;

  private ChannelActionHandler handler;

  RealChannelIO(Application application) {
    this.application = application;
    this.uncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    this.handler = new ChannelActionHandler();

    ActivityLifecycleManager activityLifecycleManager = new ActivityLifecycleManager();

    SocketManager.create(application);
    application.registerActivityLifecycleCallbacks(activityLifecycleManager);

    Thread.setDefaultUncaughtExceptionHandler((thread, ex) -> {
      if (Looper.getMainLooper().getThread() == thread) {
        if (uncaughtExceptionHandler != null) {
          uncaughtExceptionHandler.uncaughtException(thread, ex);
        }
      } else if (ex instanceof RejectedExecutionException) {
        SocketManager.reconnect();
      }
    });

    new GoogleAdIdTask(application).execute();
  }

  Application getApplication() {
    return application;
  }

  void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile, @Nullable OnBootListener listener) {
    shutdown();

    BootAction.boot(pluginSettings, profile, listener, new RestSubscriber<PluginWrapper>() {
      @Override
      public void onError(RetrofitException e) {
        BootManager.sendNetworkError(listener, e);
      }

      @Override
      public void onNext(PluginWrapper repo) {
        repo.set();

        onBoot(pluginSettings, repo.getChannel(), repo.getPlugin(), repo.getGuest(), listener);
      }
    });
  }

  private void onBoot(
      ChannelPluginSettings pluginSettings,
      Channel channel,
      Plugin plugin,
      com.zoyi.channel.plugin.android.model.entity.Guest guest,
      @Nullable OnBootListener listener
  ) {
    // Save plugin settings for push notification click route
    PrefSupervisor.setPluginSetting(application, pluginSettings);

    // Set guest data
    PrefSupervisor.setBootData(application, guest);

    // set prefs
    if (pluginSettings != null) {
      PrefSupervisor.setPluginLanguage(application, pluginSettings.getLocale());

      SettingsStore.get().launcherConfig.set(pluginSettings.getLauncherConfig());
    }

    // Set settings
    SettingsStore.get().locale.set(PrefSupervisor.getPluginLocale(application));
    SettingsStore.get().showTranslation.set(PrefSupervisor.canTranslateMessage(application));
    SettingsStore.get().showClosedChat.set(PrefSupervisor.isShownClosedChat(application));
    SettingsStore.get().raiseSoundVibrate.set(PrefSupervisor.isEnabledPushAlarm(application));

    // Enable chat
    ChatManager.initialize();

    // when jwt is not initialized, fetch from preference
    if (GlobalStore.get().jwt.get() == null) {
      GlobalStore.get().jwt.set(PrefSupervisor.getJwt(application));
    }

    ChannelPushManager.sendTokenToChannelIO(application);

    SocketManager.setChannelId(channel.getId());

    if (GlobalStore.get().topActivity.get() != null) {
      SocketManager.connect();
    }

    // if already call ChannelIO.show(), send page view event
    if (SettingsStore.get().showLauncher.get()) {
      EventAction.trackPageView();
    }

    // attach handler
    handler.handle();

    // raise event to initialize
    GlobalStore.get().bootState.set(true);

    // send to listener

    if (listener != null) {
      listener.onCompletion(
          ChannelPluginCompletionStatus.SUCCESS,
          Guest.newInstance(guest)
      );
    }
  }

  void shutdown() {
    GlobalStore.get().bootState.set(false);

    handler.unHandle();

    Action.release();

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    Action.invoke(ActionType.SHUTDOWN);

    ChatManager.release();

    Store2.destroy();

    PrefSupervisor.clearBootData(application);

    String jwt = GlobalStore.get().jwt.get();

    ChannelPushManager.deleteToken(application, jwt);
    PrefSupervisor.setJwtToken(application, null);
  }

  // start messenger

  public boolean startMessenger(@Nullable Context context, @Nullable String chatId, boolean isAnimated) {
    if (!isInitializedChannelIO()) {
      L.e("Fail to start messenger, please initialize ChannelIO first");
      return false;
    }

    if (!GlobalStore.get().bootState.get()) {
      L.e("Fail to start messenger, please 'Boot' first");
      return false;
    }

    if (context == null) {
      L.e("Fail to start messenger, context can't be NULL");
      return false;
    }

    IntentUtils.setNextActivity(context, LoungeActivity.class)
        .putExtra(Const.EXTRA_CHAT_CONTENT_TYPE, ChatContentType.USER_CHAT.toString())
        .putExtra(Const.EXTRA_CHAT_CONTENT_ID, chatId)
        .putExtra(Const.EXTRA_REDIRECT_ANIMATED, isAnimated)
        .setFlag(Intent.FLAG_ACTIVITY_CLEAR_TOP)
        .setTransition(isAnimated ? Transition.SLIDE_FROM_BOTTOM : Transition.NONE)
        .startActivity();

    return true;
  }

  // handle listener

  void setListener(@Nullable ChannelPluginListener listener) {
    handler.setListener(listener);
  }

  void clearListener() {
    handler.clearListener();
  }

  @Nullable
  ChannelPluginListener getListener() {
    return handler.getListener();
  }

  // Google ad id getter

  private static class GoogleAdIdTask extends AsyncTask<Void, Void, String> {

    private Application application;

    GoogleAdIdTask(Application application) {
      this.application = application;
    }

    @Nullable
    protected String doInBackground(final Void... params) {
      try {
        if (application != null) {
          AdvertisingIdClient.Info advertisingIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(application);
          if (advertisingIdInfo != null && !advertisingIdInfo.isLimitAdTrackingEnabled()) {
            return advertisingIdInfo.getId();
          }
        }
      } catch (Exception ignored) {
      }
      return null;
    }

    protected void onPostExecute(String adId) {
      GlobalStore.get().adId.set(adId);
    }
  }
}
