package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.util.LongSparseArray;

import com.zoyi.channel.plugin.android.ChannelApiManager;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat.model.MessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.type.ContentType;
import com.zoyi.channel.plugin.android.enumerate.SendingState;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.ActionButton;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.wrapper.MessageWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.okhttp3.MediaType;
import com.zoyi.okhttp3.MultipartBody;
import com.zoyi.okhttp3.RequestBody;

import java.util.*;

/**
 * Created by mika on 2017. 3. 2..
 */
public class ChatManager {

  @Nullable
  private static ChatManager chatManager;

  public static ChatManager get() {
    if (chatManager == null) {
      synchronized (ChatManager.class) {
        if (chatManager == null) {
          chatManager = new ChatManager();
        }
      }
    }
    return chatManager;
  }

  private boolean isMessageSending;
  private boolean released;

  private ArrayList<OnMessageSendListener> listeners;

  private Deque<SendingMessageItem> sendingItems;
  private LongSparseArray<SendingMessageItem> failedItems;

  private ChatManager() {
    released = false;
    isMessageSending = false;
    sendingItems = new LinkedList<>();
    failedItems = new LongSparseArray<>();
    listeners = new ArrayList<>();
  }

  public void addListener(OnMessageSendListener listener) {
    if (listeners != null && listener != null && listeners.indexOf(listener) < 0) {
      listeners.add(listener);
    }
  }

  public void removeListener(OnMessageSendListener listener) {
    if (listeners != null && listener != null) {
      listeners.remove(listener);
    }
  }

  public void send(SendingMessageItem item) {
    send(ListUtils.newArrayList(item));
  }

  public void send(List<SendingMessageItem> items) {
    if (!released) {
      sendingItems.addAll(items);
      sendWaitingMessage();
    }
  }

  public void sendImmediately(SendingMessageItem item) {
    if (!released) {
      sendingItems.addFirst(item);
      sendWaitingMessage();
    }
  }

  private void sendWaitingMessage() {
    if (!isMessageSending) {
      while (sendingItems.size() > 0) {
        SendingMessageItem item = sendingItems.poll();
        if (item != null && item.getContentType() != ContentType.NONE) {
          isMessageSending = true;
          routeItem(item);

          break;
        }
      }
    }
  }

  public void updateFailedMessages(List<SendingMessageItem> items) {
    for (SendingMessageItem item : items) {
      failedItems.put(item.getPrimaryKey(), item);
    }
  }

  public void read(@Nullable String chatId) {
    if (chatId != null) {
      ChannelApiManager.call(ChannelApiManager.get().read(chatId), new RestSubscriber<Void>());
    }
  }

  // Message item router

  private void routeItem(SendingMessageItem item) {
    switch (item.getContentType()) {
      case FILE:
        sendMediaFileMessage(item);
        break;

      case TEXT:
        sendTextMessage(item);
        break;

      case ACTION:
        processAction(item);
        break;
    }
  }

  private void processAction(SendingMessageItem item) {
    String actionType = item.getActionType();
    ActionButton actionButton = item.getActionButton();

    if (actionType != null && actionButton != null) {
      switch (actionType) {
        case Const.ACTION_TYPE_SOLVE:
        case Const.ACTION_TYPE_CLOSE:
          switch (actionButton.getKey()) {
            case Const.ACTION_KEY_CLOSE:
              closeChat(item);
              break;

            case Const.ACTION_KEY_LIKE:
              reviewChat(item, Const.REVIEW_LIKE);
              break;

            case Const.ACTION_KEY_DISLIKE:
              reviewChat(item, Const.REVIEW_DISLIKE);
              break;

            default:
              submitMessageAction(item);
              break;
          }
          break;

        case Const.ACTION_TYPE_SUPPORT_BOT:
          submitSupportBotAction(item);
          break;

        case Const.ACTION_TYPE_SELECT:
          submitMessageAction(item);
          break;

        default:
          submitMessageAction(item);
          break;
      }
    }
  }

  // Api functions

  private void sendTextMessage(SendingMessageItem item) {
    String chatId = item.getChatId();

    if (chatId != null) {
      RequestBody body = RequestUtils.form()
          .set("message", item.getText())
          .set("requestId", item.getRequestId())
          .create();

      ChannelApiManager.call(
          ChannelApiManager.get().sendMessage(chatId, body),
          createMessageSubscriber(item)
      );
    }
  }

  private void sendMediaFileMessage(SendingMessageItem item) {
    String chatId = item.getChatId();

    if (chatId != null) {
      MultipartBody.Part body = RequestUtils.makeMultipart(item.getFilePath());
      RequestBody timestamp = RequestBody.create(MediaType.parse("text/plane"), item.getRequestId());

      if (body == null) {
        onSendFail(new Exception("Not supported type"), item);
        return;
      }

      ChannelApiManager.call(
          ChannelApiManager.get().uploadFile(chatId, body, timestamp),
          createMessageSubscriber(item)
      );
    }
  }

  private void submitMessageAction(SendingMessageItem item) {
    String chatId = item.getChatId();
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Map<String, String> submit = new HashMap<>();
      submit.put("id", actionButton.getMessage().getId());
      submit.put("key", actionButton.getKey());

      RequestBody body = RequestUtils.form()
          .set("message", item.getText())
          .set("requestId", item.getRequestId())
          .set("submit", submit)
          .create();

      ChannelApiManager.call(
          ChannelApiManager.get().sendMessage(chatId, body),
          createMessageSubscriber(item)
      );
    }
  }

  private void closeChat(SendingMessageItem item) {
    String chatId = item.getChatId();
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      ChannelApiManager.call(
          ChannelApiManager.get().closeUserChat(chatId, actionButton.getMessage().getId(), item.getRequestId()),
          createMessageSubscriber(item)
      );
    }
  }

  private void reviewChat(SendingMessageItem item, @NonNull String review) {
    String chatId = item.getChatId();
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      ChannelApiManager.call(
          ChannelApiManager.get().reviewUserChat(chatId, actionButton.getMessage().getId(), review, item.getRequestId()),
          createMessageSubscriber(item)
      );
    }
  }

  private void submitSupportBotAction(SendingMessageItem item) {
    String chatId = item.getChatId();
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      ChannelApiManager.call(
          ChannelApiManager.get().submitButtonAction(chatId, actionButton.getKey(), actionButton.getMessage().getId(), item.getRequestId()),
          createMessageSubscriber(item)
      );
    }
  }

  // Finish callbacks

  private void onSendSuccess(Message message) {
    if (!released) {
      if (listeners != null) {
        for (OnMessageSendListener listener : listeners) {
          listener.onSendSuccess(message);
        }
      }
      isMessageSending = false;
      sendWaitingMessage();
    }
  }

  private void onSendFail(Exception ex, SendingMessageItem item) {
    if (ex != null) {
      L.e(ex.getMessage());
    }

    if (!released) {
      item.setSendingState(SendingState.FAIL);
      failedItems.put(item.getPrimaryKey(), item);

      if (listeners != null) {
        for (OnMessageSendListener listener : listeners) {
          listener.onSendFail(item);
        }
      }

      isMessageSending = false;
      sendWaitingMessage();
    }
  }

  // Default subscriber callback generator

  private RestSubscriber<MessageWrapper> createMessageSubscriber(final SendingMessageItem item) {
    return new RestSubscriber<MessageWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        onSendFail(error, item);
      }

      @Override
      public void onSuccess(@NonNull MessageWrapper repo) {
        repo.update();
        onSendSuccess(repo.getMessage());
      }
    };
  }

  // Failed item logic

  public void removeSendingFailedItem(Long createdAt) {
    if (createdAt != null) {
      failedItems.remove(createdAt);
    }
  }

  public List<MessageItem> getSendingItems(@Nullable String chatId) {
    List<MessageItem> items = new ArrayList<>();

    for (SendingMessageItem item : sendingItems) {
      if (CompareUtils.isSame(chatId, item.getChatId())) {
        items.add(item);
      }
    }
    return items;
  }

  public List<MessageItem> getSendingFailedItems(@Nullable String chatId) {
    List<MessageItem> items = new ArrayList<>();

    for (int i = 0; i < failedItems.size(); i++) {
      SendingMessageItem item = failedItems.valueAt(i);

      if (CompareUtils.isSame(chatId, item.getChatId())) {
        items.add(item);
      }
    }
    return items;
  }

  // Release function
  public static void release() {
    if (chatManager != null) {
      chatManager.released = true;
      chatManager.sendingItems.clear();
      chatManager.failedItems.clear();
      chatManager = null;
    }
  }
}
