/*
 * Copyright (C) 2016 Francisco José Montiel Navarro.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.zoyi.channel.plugin.android.util.persistentcookiejar;

import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.persistentcookiejar.cache.CookieCache;
import com.zoyi.channel.plugin.android.util.persistentcookiejar.persistence.CookiePersistor;

import java.util.*;

import com.zoyi.okhttp3.Cookie;
import com.zoyi.okhttp3.HttpUrl;

public class PersistentCookieJar implements ClearableCookieJar {

    private CookieCache cache;
    private CookiePersistor persistor;

    public PersistentCookieJar(CookieCache cache, CookiePersistor persistor) {
        this.cache = cache;
        this.persistor = persistor;

        this.cache.addAll(persistor.loadAll());
    }

    @Override
    synchronized public void saveFromResponse(HttpUrl url, List<Cookie> cookies) {
        cache.addAll(cookies);
        persistor.saveAll(filterPersistentCookies(cookies));
    }

    private static List<Cookie> filterPersistentCookies(List<Cookie> cookies) {
        List<Cookie> persistentCookies = new ArrayList<>();

        for (Cookie cookie : cookies) {
            if (cookie.persistent()) {
                persistentCookies.add(cookie);
            }
        }
        return persistentCookies;
    }

    @Override
    synchronized public List<Cookie> loadForRequest(HttpUrl url) {
        List<Cookie> cookiesToRemove = new ArrayList<>();
        Map<String, Cookie> validCookies = new HashMap<>();

        for (Iterator<Cookie> it = cache.iterator(); it.hasNext(); ) {
            Cookie currentCookie = it.next();

            if (isCookieExpired(currentCookie)) {
                cookiesToRemove.add(currentCookie);
                it.remove();

            } else if (currentCookie.matches(url)) {
                Cookie oldbie = validCookies.containsKey(currentCookie.name()) ? validCookies.get(currentCookie.name()) : null;
                if (canUpsert(oldbie, currentCookie)) {
                    validCookies.put(currentCookie.name(), currentCookie);
                }
            }
        }

        persistor.removeAll(cookiesToRemove);

        return new ArrayList<>(validCookies.values());
    }

    private static boolean isCookieExpired(Cookie cookie) {
        return cookie.expiresAt() < System.currentTimeMillis();
    }

    private static boolean canUpsert(Cookie oldbie, Cookie newbie) {
        if (newbie == null) {
            return false;
        }
        if (oldbie == null) {
            return true;
        }

        String oldDomain = oldbie.domain();
        String newDomain = newbie.domain();

        if (newDomain == null) {
            return false;
        }
        if (oldDomain == null) {
            return true;
        }

        return oldDomain.length() < newDomain.length();
    }

    @Override
    synchronized public void clearSession() {
        cache.clear();
        cache.addAll(persistor.loadAll());
    }

    @Override
    synchronized public void clear() {
        cache.clear();
        persistor.clear();
    }
}
