package com.zoyi.channel.plugin.android.evaluation;

import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.evaluation.EnvironmentInfo;
import com.zoyi.channel.plugin.android.model.rest.TargetCondition;
import com.zoyi.channel.plugin.android.util.L;

import java.math.BigDecimal;
import java.util.List;
import java.util.regex.Pattern;

import static com.zoyi.channel.plugin.android.evaluation.Operator.EQUAL;
import static com.zoyi.channel.plugin.android.evaluation.Operator.EXIST;
import static com.zoyi.channel.plugin.android.evaluation.Operator.GT;
import static com.zoyi.channel.plugin.android.evaluation.Operator.GTE;
import static com.zoyi.channel.plugin.android.evaluation.Operator.IN;
import static com.zoyi.channel.plugin.android.evaluation.Operator.LT;
import static com.zoyi.channel.plugin.android.evaluation.Operator.LTE;
import static com.zoyi.channel.plugin.android.evaluation.Operator.NOT_EQUAL;
import static com.zoyi.channel.plugin.android.evaluation.Operator.NOT_EXIST;
import static com.zoyi.channel.plugin.android.evaluation.Operator.NOT_IN;
import static com.zoyi.channel.plugin.android.evaluation.Operator.NOT_PREFIX;
import static com.zoyi.channel.plugin.android.evaluation.Operator.PREFIX;
import static com.zoyi.channel.plugin.android.evaluation.Operator.REGEX;

/**
 * Created by jerry on 2018. 11. 20..
 */

public class EvaluationUtils {

  public static boolean evaluateTarget(List<List<TargetCondition>> target, EnvironmentInfo environmentInfo) {
    for (List<TargetCondition> conditions : target) {
      if (!evaluateConditions(conditions, environmentInfo)) {
        return false;
      }
    }
    return true;
  }

  private static boolean evaluateConditions(List<TargetCondition> conditions, EnvironmentInfo environmentInfo) {
    for (TargetCondition condition : conditions) {
      if (evaluateCondition(condition, environmentInfo)) {
        return true;
      }
    }
    return false;
  }

  private static boolean evaluateCondition(TargetCondition condition, EnvironmentInfo environmentInfo) {
    String key = condition.getKey();
    String subKey = condition.getSubKey();
    String operator = condition.getOperator();

    if (key == null || operator == null) {
      return false;
    }

    try {
      Object value = subKey != null
          ? environmentInfo.getValueBySubKey(key, subKey)
          : environmentInfo.getValueByKey(key);

      return evaluate(condition.getValue(), value, operator);
    } catch (Exception exception) {
      if (exception.getMessage() != null) {
        L.e("Evaluate exception : " + exception.getMessage());
      }

      return false;
    }
  }

  public static boolean evaluate(@Nullable String target, @Nullable Object value, String operator) throws Exception {
    switch (operator) {
      case EQUAL:
        return evaluateEqual(target, value);

      case NOT_EQUAL:
        return evaluateNotEqual(target, value);

      case IN:
        return evaluateIn(target, value);

      case NOT_IN:
        return evaluateNotIn(target, value);

      case PREFIX:
        return evaluatePrefix(target, value);

      case NOT_PREFIX:
        return evaluateNotPrefix(target, value);

      case REGEX:
        return evaluateRegex(target, value);

      case EXIST:
        return evaluateExist(value);

      case NOT_EXIST:
        return evaluateNotExist(value);

      case GT:
      case GTE:
      case LT:
      case LTE:
        return evaluateNumberTypeOperator(target, value, operator);

      default:
        return false;
    }
  }

  private static boolean evaluateEqual(@Nullable String target, @Nullable Object value) {
    return target != null && value != null && value.toString().equals(target);
  }

  private static boolean evaluateNotEqual(@Nullable String target, @Nullable Object value) {
    return target != null && value != null && !value.toString().equals(target);
  }

  private static boolean evaluateIn(@Nullable String target, @Nullable Object value) {
    return target != null && value != null && value.toString().contains(target);
  }

  private static boolean evaluateNotIn(@Nullable String target, @Nullable Object value) {
    return target != null && value != null && !value.toString().contains(target);
  }

  private static boolean evaluatePrefix(@Nullable String target, @Nullable Object value) {
    return target != null && value != null && value.toString().startsWith(target);
  }

  private static boolean evaluateNotPrefix(@Nullable String target, @Nullable Object value) {
    return target != null && value != null && !value.toString().startsWith(target);
  }

  private static boolean evaluateRegex(@Nullable String target, @Nullable Object value) {
    return target != null && value != null && Pattern.matches(target, value.toString());
  }

  private static boolean evaluateExist(@Nullable Object value) {
    return value != null;
  }

  private static boolean evaluateNotExist(@Nullable Object value) {
    return value == null;
  }

  private static boolean evaluateNumberTypeOperator(@Nullable String target, @Nullable Object value, String operator) throws Exception {
    if (target == null || value == null) {
      return false;
    }

    BigDecimal targetDecimal = new BigDecimal(target);
    BigDecimal valueDecimal = new BigDecimal(value.toString());

    switch (operator) {
      case GT:
        return evaluateGT(targetDecimal, valueDecimal);
      case GTE:
        return evaluateGTE(targetDecimal, valueDecimal);
      case LT:
        return evaluateLT(targetDecimal, valueDecimal);
      case LTE:
        return evaluateLTE(targetDecimal, valueDecimal);
      default:
        return false;
    }
  }

  private static boolean evaluateGT(BigDecimal target, BigDecimal value) {
    return value.compareTo(target) > 0;
  }

  private static boolean evaluateGTE(BigDecimal target, BigDecimal value) {
    return value.compareTo(target) >= 0;
  }

  private static boolean evaluateLT(BigDecimal target, BigDecimal value) {
    return value.compareTo(target) < 0;
  }

  private static boolean evaluateLTE(BigDecimal target, BigDecimal value) {
    return value.compareTo(target) <= 0;
  }

}
