package com.zoyi.channel.plugin.android.activity.chat;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.zoyi.channel.plugin.android.ChannelApiManager;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageContentClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnProfileUpdateListener;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.enumerate.ChatState;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.TranslationState;
import com.zoyi.channel.plugin.android.event.ChannelModelBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.ActionInput;
import com.zoyi.channel.plugin.android.model.TranslationInfo;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.entity.Person;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.*;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.selector.*;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.store.TranslationStore;
import com.zoyi.channel.plugin.android.store.UserChatStore;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.view.dialog.CountryCodeDialog;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2017. 2. 27..
 */

public class ChatPresenter
    implements
    ChatContract.Presenter,
    OnProfileUpdateListener,
    OnMessageContentClickListener,
    OnMessageSendListener {

  private ChatContract.View view;
  private ChatAdapterContract.View adapterView;
  private ChatAdapterContract.Model adapterModel;

  private Context context;

  @Nullable
  private UserChat userChat;
  @Nullable
  private Session session;

  @Nullable
  private String chatId;

  private ChatState state = ChatState.IDLE;

  private String backwardId = Const.MESSAGE_ID_MAX;
  private String forwardId = Const.MESSAGE_ID_MIN;
  private String forwardTempId = Const.MESSAGE_ID_MIN;
  private Long forwardTimestamp = Long.MIN_VALUE;

  private int fetchCounter = 0;
  private boolean backwardLoading = false;
  private boolean isNewSupportChat = false;

  @Nullable
  private Long lastReadAt;
  private long welcomedAt;

  private List<SendingMessageItem> tempQueue;
  @Nullable
  private SendingMessageItem supportTempItem;

  private final Object messageObject = new Object();

  public ChatPresenter(Context context) {
    this.context = context;

    TimeUtils.refreshOffset();

    tempQueue = new ArrayList<>();

    ChatManager.get().addListener(this);
  }

  @Override
  public void release() {
    ChatManager.get().removeListener(this);
  }

  @Initializer
  @Override
  public void setView(ChatContract.View view) {
    this.view = view;
  }

  @Nullable
  @Override
  public UserChat getUserChat() {
    return userChat;
  }

  @Nullable
  @Override
  public String getChatId() {
    return chatId;
  }

  @Override
  public void setChatId(@Nullable String chatId) {
    this.chatId = chatId;
  }

  @Nullable
  @Override
  public String getBackwardId() {
    return backwardId;
  }

  @Initializer
  @Override
  public void setAdapterView(ChatAdapterContract.View adapterView) {
    this.adapterView = adapterView;
    this.adapterView.setOnMessageContentClickListener(this);
    this.adapterView.setOnProfileUpdateListener(this);
  }

  @Initializer
  @Override
  public void setAdapterModel(ChatAdapterContract.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void init(@Nullable final String chatId) {
    if (!TextUtils.isEmpty(chatId)) {
      fetchUserChat(chatId);
    } else {
      updateInitMessage();

      view.onLoadUserChat(null);
      view.updateBackwardId(null);
    }
  }

  private void updateInitMessage() {
    if (SupportBotSelector.getSupportBotEntry() != null && adapterModel.getWelcomeMessageItem() == null) {
      view.setInputLayoutVisibility(Const.USER_CHAT_STATE_SUPPORTING);
      adapterModel.setSupportBotMessage(SupportBotSelector.getSupportBotEntry());
    } else {
      addWelcomeMessage();
    }
  }

  private void addWelcomeMessage() {
    welcomedAt = TimeUtils.getCurrentTime();
    adapterModel.setWelcomeMessage(PluginSelector.getWelcomeMessage(context));
  }

  private void fetchUserChat(String chatId) {
    this.chatId = chatId;

    adapterModel.addMessageItems(getChatManager().getSendingFailedItems(chatId));

    if (userChat == null) {
      view.showProgress(ResUtils.getString(context, "ch.loading_information"));
    }

    setState(ChatState.USER_CHAT_LOADING, false);

    ChannelApiManager.get()
        .getUserChat(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            view.hideProgress();
            handleUserChatFetchError(error);
          }

          @Override
          public void onNext(UserChatWrapper repo) {
            if (repo != null) {
              repo.update();
            }

            setUserChat(repo, false);
          }
        });
  }

  private void createUserChat() {
    view.showProgress(ResUtils.getString(context, "ch.loading_information"));

    setState(ChatState.USER_CHAT_LOADING, false);

    Plugin plugin = PluginSelector.getPlugin();

    if (plugin != null && plugin.getKey() != null) {
      ChannelApiManager.get()
          .createUserChat(plugin.getKey(), plugin.getId(), welcomedAt)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              view.hideProgress();
              handleUserChatFetchError(error);
            }

            @Override
            public void onNext(UserChatWrapper repo) {
              if (repo != null) {
                repo.update();
              }

              RxBus.post(new ChannelModelBus(repo.getUserChat(), true));
              RxBus.post(new ChannelModelBus(repo.getSession(), true));

              setUserChat(repo, true);
            }
          });
    } else {
      view.hideProgress();
    }
  }

  private void setUserChat(UserChatWrapper wrapper, boolean create) {
    view.hideProgress();

    if (wrapper != null) {
      bindUserChat(this.userChat, wrapper.getUserChat());
    }

    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      if (wrapper == null) {
        handleUserChatFetchError(new Exception("UserChatWrapper cannot be null"));
      } else {
        userChat = wrapper.getUserChat();
        chatId = userChat.getId();
        session = wrapper.getSession();

        view.onLoadUserChat(userChat);
        view.setReconnectVisibility(false);

        Store.getInstance(UserChatStore.class).add(userChat);

        if (lastReadAt == null) {
          lastReadAt = create || session == null ? Long.MAX_VALUE : session.getReadAt();
        }

        if (tempQueue.size() > 0) {
          for (SendingMessageItem item : tempQueue) {
            item.setChatId(chatId);
          }
          getChatManager().send(tempQueue);
          tempQueue.clear();
        }

        joinChat();
      }
    }
  }

  private void bindUserChat(@Nullable UserChat oldUserChat, UserChat newUserChat) {
    if (chatId == null || newUserChat == null || !chatId.equals(newUserChat.getId())) {
      return;
    }
    view.bindUserChat(newUserChat);
    if (oldUserChat != null) {
      if (CompareUtils.exists(oldUserChat.getState(), Const.USER_CHAT_STATE_READY, Const.USER_CHAT_STATE_SUPPORTING) &&
          Const.USER_CHAT_STATE_OPEN.equals(newUserChat.getState())) {
        requestProfileBot();
      }
    }
  }

  private void handleUserChatFetchError(Throwable throwable) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      setState(ChatState.USER_CHAT_NOT_LOADED, true);
    }
  }

  private void joinChat() {
    if (SocketManager.isReady()) {
      setState(ChatState.CHAT_JOINING, false);
      SocketManager.joinChat(chatId);
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  private boolean isAvailableRequest(int count) {
    return isStateEquals(ChatState.INIT_MESSAGES_LOADING) && fetchCounter == count;
  }

  private void fetchMessages() {
    if (SocketManager.isReady()) {
      fetchCounter++;
      setState(ChatState.INIT_MESSAGES_LOADING, false);
      if (!isMessageExists()) {
        fetchInitMessages(fetchCounter);
      } else {
        fetchForwardMessages(forwardId, fetchCounter);
      }
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  @Override
  public boolean isMessageExists() {
    return CompareUtils.compare(backwardId, forwardId) < 0;
  }

  private void fetchInitMessages(final int counter) {
    if (!TextUtils.isEmpty(chatId)) {
      ChannelApiManager.get()
          .getMessages(
              chatId,
              Const.MESSAGE_ID_MAX,
              Const.MESSAGE_FETCH_LIMIT,
              Const.DESC)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<MessagesWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              if (isAvailableRequest(counter)) {
                handleMessagesFetchError(error, counter);
              }
            }

            @Override
            public void onNext(MessagesWrapper repo) {
              if (isAvailableRequest(counter)) {
                if (repo != null) {
                  repo.update();
                }

                if (repo == null) {
                  handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
                } else {
                  backwardId = repo.getNext();
                  view.updateBackwardId(backwardId);

                  addMessages(repo.getMessages());
                  calculateForwardId(repo.getMessages());
                  calculateForwardId();

                  if (lastReadAt != null && lastReadAt >= forwardTimestamp) {
                    lastReadAt = Long.MAX_VALUE;
                  }

                  addNewMessageItem(repo.getMessages());

                  setState(ChatState.CHAT_READY, false);
                  readAll(repo.getMessages());

                  if (isNewSupportChat && supportTempItem != null) {
                    ChatManager.get().send(supportTempItem);
                    isNewSupportChat = false;
                  }

                  view.scrollToBottom(true);
                }
              }
            }
          });
    }
  }

  @Override
  public void fetchBackwardMessages() {
    if (backwardLoading || !isMessageExists() || chatId == null) {
      return;
    }
    backwardLoading = true;

    ChannelApiManager.get()
        .getMessages(
            chatId,
            backwardId,
            Const.MESSAGE_FETCH_LIMIT,
            Const.DESC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            backwardLoading = false;

            handleMessagesFetchError(error, fetchCounter);
          }

          @Override
          public void onNext(MessagesWrapper repo) {
            if (repo != null) {
              repo.update();
            }

            if (repo == null) {
              handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), fetchCounter);
            } else {
              backwardId = repo.getNext();
              view.updateBackwardId(backwardId);

              adapterModel.addMessages(repo.getMessages(), chatId);

              addNewMessageItem(repo.getMessages());
            }
            backwardLoading = false;
          }
        });
  }

  @Override
  public void fetchForwardMessages(String since, final int counter) {
    if (!TextUtils.isEmpty(chatId)) {
      ChannelApiManager.get()
          .getMessages(
              chatId,
              since,
              Const.MESSAGE_FETCH_LIMIT,
              Const.ASC)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<MessagesWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              if (isAvailableRequest(counter)) {
                handleMessagesFetchError(error, counter);
              }
            }

            @Override
            public void onNext(MessagesWrapper repo) {
              if (isAvailableRequest(counter)) {
                if (repo == null) {
                  handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
                } else {
                  repo.update();

                  addMessages(repo.getMessages());
                  calculateForwardId(repo.getMessages());

                  if (repo.getNext() == null) {
                    calculateForwardId();
                    setState(ChatState.CHAT_READY, false);
                    readAll(repo.getMessages());
                  } else {
                    fetchForwardMessages(forwardId, counter);
                  }
                }
              }
            }
          });
    }
  }

  private void handleMessagesFetchError(Throwable throwable, int counter) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.INIT_MESSAGES_LOADING) && fetchCounter == counter) {
      setState(ChatState.INIT_MESSAGES_NOT_LOADED, true);
    }
  }

  private void addNewMessageItem(List<Message> messages) {
    if (lastReadAt == null || userChat == null || userChat.isStateCompleted()) {
      return;
    }

    long min = Long.MAX_VALUE;
    long max = Long.MIN_VALUE;

    if (messages != null) {
      for (Message message : messages) {
        if (message.getLog() != null) {
          continue;
        }
        min = Math.min(min, message.getCreatedAt());
        max = Math.max(max, message.getCreatedAt());
      }
    }

    if (min <= lastReadAt && lastReadAt < max) {
      adapterModel.addMessageItem(new NewMessageItem(lastReadAt));
    }
  }

  private void addMessage(Message message) {
    if (!TextUtils.isEmpty(chatId) && CompareUtils.isSame(message.getChatId(), chatId)) {
      String oldMessageId = adapterModel.getLastMessageItemId();
      adapterModel.addMessage(message, chatId);
      String newMessageId = adapterModel.getLastMessageItemId();

      if (isStateEquals(ChatState.CHAT_READY)) {
        calculateForwardId(message);
      } else {
        forwardTempId = CompareUtils.max(forwardTempId, message.getId());
      }

      forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());

      view.scrollToBottom(view.isMovableToBottomPosition());

      if (!CompareUtils.isSame(oldMessageId, newMessageId)) {
        view.processNewMessage(
            ProfileSelector.getHostProfile(message.getPersonType(), message.getPersonId()),
            message);
      }
    }
  }

  private void addMessages(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());
        forwardId = CompareUtils.max(forwardId, message.getId());
      }
    }
    adapterModel.addMessages(messages, chatId);
  }

  private void calculateForwardId(Message message) {
    if (message != null) {
      forwardId = CompareUtils.max(forwardId, message.getId());
    }
  }

  private void calculateForwardId(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        calculateForwardId(message);
      }
    }
  }

  private void calculateForwardId() {
    forwardId = CompareUtils.max(forwardId, forwardTempId);
  }

  @Override
  public void sendTextMessage(String text) {
    if (!TextUtils.isEmpty(text)) {
      SendingMessageItem item = new SendingMessageItem(chatId);
      item.setText(text);

      adapterModel.addMessageItem(item);

      sendMessages(Collections.singletonList(item));
    }
  }

  @Override
  public void sendImageFiles(ArrayList<String> imagePaths) {
    List<SendingMessageItem> items = new ArrayList<>();
    if (imagePaths != null) {
      for (String imagePath : imagePaths) {
        SendingMessageItem item = new SendingMessageItem(chatId);
        item.setFilePath(imagePath);

        adapterModel.addMessageItem(item);
        items.add(item);
      }
    }
    sendMessages(items);
  }

  private void sendMessages(List<SendingMessageItem> items) {
    view.scrollToBottom(true);

    if (chatId == null) {
      tempQueue.addAll(items);
      createUserChat();
    } else {
      getChatManager().send(items);
    }
  }

  @Override
  public void receiveData(Entity entity, boolean upsert) {
    if (entity != null) {
      if (!TextUtils.isEmpty(chatId)) {
        synchronized (messageObject) {
          switch (entity.getClass().getSimpleName()) {
            case Message.CLASSNAME:
              Message message = (Message) entity;

              if (message.getBotOption() != null &&
                  message.getBotOption().isWelcome() &&
                  adapterModel.getWelcomeMessageItem() != null) {
                return;
              }

              handleMessageLog(message);
              addMessage(message);
              break;

            case Bot.CLASSNAME:
            case Manager.CLASSNAME:
              if (userChat != null && !userChat.isStateOpen()) {
                view.setLastResponseManager((Person) entity);
              }
              break;

            case UserChat.CLASSNAME:
              handleUserChatData((UserChat) entity, chatId);
              break;

            case Channel.CLASSNAME:
              view.setChannel((Channel) entity);
              break;
          }
        }
      }
    }
  }

  // Catch UserChat socket event
  private void handleUserChatData(UserChat userChat, String chatId) {
    if (userChat == null || !chatId.equals(userChat.getId())) {
      return;
    }

    bindUserChat(this.userChat, userChat);

    this.userChat = userChat;
    Store.getInstance(UserChatStore.class).add(userChat);

    if (userChat.isStateRemoved()) {
      view.onRemovedChat();
    } else if (userChat.isStateCompleted()) {
      view.scrollToBottom(false);
    }
  }

  private void handleMessageLog(Message message) {
    if (CompareUtils.isSame(message.getChatId(), chatId) && userChat != null && message.getLog() != null && message.getLog().isClose()) {
      userChat.setState(Const.USER_CHAT_STATE_CLOSED);
    }
  }

  private void requestProfileBot() {
    Plugin plugin = PluginSelector.getPlugin();

    if (plugin != null && !TextUtils.isEmpty(chatId)) {
      ChannelApiManager.get()
          .requestProfileBot(chatId, plugin.getId())
          .onBackpressureBuffer()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe();
    }
  }

  private void readAll(List<Message> messages) {
    if (ListUtils.hasItems(messages)) {
      Long lastMessageCreatedAt = Long.MIN_VALUE;

      for (Message message : messages) {
        if (message.getCreatedAt() > lastMessageCreatedAt) {
          lastMessageCreatedAt = message.getCreatedAt();
        }
      }

      if (lastMessageCreatedAt != Long.MIN_VALUE) {
        readAll(lastMessageCreatedAt);
      }
    }
  }

  private void readAll(Long lastMessageCreatedAt) {
    if (!TextUtils.isEmpty(chatId) && lastMessageCreatedAt != null) {
      ChannelApiManager.get()
          .read(chatId)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<Void>());
    }
  }

  @Override
  public void receiveCommand(Command command, @Nullable Object object) {
    switch (command) {
      case APP_STARTED:
        refresh();
        break;

      case SOCKET_DISCONNECTED:
        fetchCounter++;
        if (!isStateEquals(ChatState.IDLE)) {
          setState(ChatState.USER_CHAT_NOT_LOADED, true);
        }
        break;

      case SOCKET_ERROR:
        view.setReconnectVisibility(true);
        break;

      case READY:
        view.setReconnectVisibility(false);

        if (isStateEquals(ChatState.WAITING_SOCKET)) {
          joinChat();
        } else {
          refresh();
        }
        break;

      case LEAVED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_READY)) {
          setState(ChatState.NOT_JOINED, false);  // true?
        }
        break;

      case JOINED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_JOINING)) {

          view.setReconnectVisibility(false);

          fetchMessages();
        }
        break;
    }
  }

  @Override
  public void receiveTyping(Typing typing) {
    synchronized (messageObject) {
//      boolean isBottom = view.isBottomPosition();

      TypingItem item = new TypingItem();
      int position = adapterModel.getIndex(item);

      if (position >= 0) {
        item = (TypingItem) adapterModel.getItem(position);
      }
      if (item == null) {
        return;
      }

      switch (typing.getAction()) {
        case "start":
          item.addOrUpdateTyping(typing);
          break;

        case "stop":
          item.removeTyping(typing);
          break;
      }

      adapterModel.addOrUpdateMessageItem(position, item);
    }
  }

  @Override
  public void receiveLanguageEvent() {
    if (chatId == null) {
      updateInitMessage();
    }
  }

  @Override
  public void onResendButtonClick(SendingMessageItem item) {
    view.sendingMessageClicked(item);
  }

  @Override
  public void resend(SendingMessageItem item) {
    if (item != null) {
      removeFailedItem(item);

      SendingMessageItem newItem = SendingMessageItem.copyContentFrom(item);

      adapterModel.addMessageItem(newItem);
      getChatManager().send(newItem);
    }
  }

  @Override
  public void removeFailedItem(SendingMessageItem item) {
    if (item != null) {
      adapterModel.removeMessageItem(item);
      getChatManager().removeSendingFailedItem(item.getPrimaryKey());
    }
  }

  @Override
  public void onSendSuccess(Message message) {
    receiveData(message, true);
  }

  @Override
  public void onSendFail(SendingMessageItem item) {
    adapterModel.addMessageItem(item);
  }

  @Override
  public void refresh() {
    view.setRefreshVisibility(false);

    switch (state) {
      case USER_CHAT_NOT_LOADED:
        if (!TextUtils.isEmpty(chatId)) {
          fetchUserChat(chatId);
        }
        break;

      case WAITING_SOCKET:
        SocketManager.connect();
        break;

      case NOT_JOINED:
        setState(ChatState.CHAT_JOINING, true);
        SocketManager.joinChat(chatId);
        break;

      case INIT_MESSAGES_NOT_LOADED:
        fetchMessages();
        break;

      case IDLE:
        init(chatId);
        break;
    }
  }

  private void setState(ChatState state, boolean showRefresh) {
    this.state = state;
    view.setRefreshVisibility(showRefresh);
  }

  private boolean isStateEquals(ChatState targetState) {
    return targetState.equals(state);
  }

  @Override
  public void leaveChat() {
    SocketManager.leaveChat(chatId);
  }

  private ChatManager getChatManager() {
    return ChatManager.get();
  }

  @Override
  public void getCountries(final ProfileBotMessageItem item, final String key, final MobileNumber mobileNumber) {
    if (CountrySelector.get().isEmpty()) {
      view.showProgress(ResUtils.getString(context, "ch.loading_information"));

      ChannelApiManager.get()
          .getCountries()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<CountryWrapper>() {
            @Override
            public void onError(RetrofitException e) {
              view.hideProgress();
            }

            @Override
            public void onNext(CountryWrapper repo) {
              view.hideProgress();

              if (repo != null) {
                repo.set();
                showCountryDialog(item, key, mobileNumber);
              }
            }
          });
    } else {
      showCountryDialog(item, key, mobileNumber);
    }
  }

  private void showCountryDialog(final ProfileBotMessageItem item, final String key, final MobileNumber mobileNumber) {
    String countryName;

    switch (PrefSupervisor.getPluginLanguage(context)) {
      case "ko":
        countryName = "South Korea";
        break;
      case "ja":
        countryName = "Japan";
        break;
      default:
        countryName = "United States";
        break;
    }
    new CountryCodeDialog(context, countryName, new CountryCodeDialog.OnCountryCodeSelectListener() {
      @Override
      public void onCountryCodeSelected(int position, String countryCode) {
        mobileNumber.setCountryCode(Integer.valueOf(countryCode));
        item.setInputState(key, mobileNumber, false, null);
        adapterModel.addMessageItem(item);
      }
    }).show();
  }

  @Override
  public void onProfileUpdate(final ProfileBotMessageItem item, final String key, final Object value) {
    item.setInputState(key, item.getInputState(), true, null);
    adapterModel.addMessageItem(item);

    RequestBody requestBody = RequestUtils.form().set(key, value).create();
    ChannelApiManager.get()
        .updateProfileBot(item.getMessage().getId(), requestBody)
        .onBackpressureBuffer()
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessageWrapper>() {

          @Override
          public void onError(RetrofitException error) {
            item.setInputState(key, item.getInputState(), false, ResUtils.getString(context, "ch.profile_form.error"));
            adapterModel.addMessageItem(item);
            view.scrollToBottom(view.isMovableToBottomPosition());
          }

          @Override
          public void onNext(MessageWrapper repo) {
            if (repo != null) {
              repo.update();
            }

            item.setMessage(repo.getMessage());
            item.resetInputState();
            adapterModel.addMessageItem(item);
            view.scrollToBottom(view.isMovableToBottomPosition());
          }
        });
  }

  @Override
  public void onCompleteWriteProfile() {
    if (userChat != null && !userChat.isStateCompleted() && !userChat.isStateRemoved()) {
      view.requestFocusEditText();
    }
  }

  @Override
  public void onChangeProfileFocus(boolean hasFocus) {
    view.onChangeProfileMessageFocus(hasFocus);
  }

  @Override
  public void onFormClick(@Nullable String actionType, ActionInput actionInput) {
    // Support bot initialize
    if (Const.FORM_ACTION_TYPE_SUPPORT_BOT.equals(actionType) && chatId == null) {
      createSupportBotUserChat(actionInput);
    }

    // Reopen chat
    else if (CompareUtils.exists(actionType, Const.FORM_ACTION_TYPE_SOLVE, Const.FORM_ACTION_CLOSE) &&
        actionInput != null &&
        Const.FORM_ACTION_REOPEN.equals(actionInput.getKey())) {
      reopenChat(actionInput);
    }

    // Other
    else if (chatId != null) {
      SendingMessageItem item = new SendingMessageItem(chatId);
      item.setActionInput(actionType, actionInput);
      adapterModel.addMessageItem(item);
      ChatManager.get().sendImmediately(item);
    }
  }

  @Override
  public void onTranslationButtonClick(final Message message) {
    final String pluginLanguage = PrefSupervisor.getPluginLanguage(context);
    final String uniqueId = TranslationInfo.createId(message.getId(), pluginLanguage);

    final TranslationInfo translationInfo = TranslationSelector.get(uniqueId);

    if (translationInfo == null) {
      //For showing Progress view
      Store.getInstance(TranslationStore.class).add(TranslationInfo.createProgressStateInfo(uniqueId));
      adapterModel.addMessage(message, chatId);

      ChannelApiManager.get()
          .getTranslatedMessage(message.getId(), pluginLanguage)
          .onBackpressureBuffer()
          .subscribeOn(Schedulers.io())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<TranslationRepo>() {

            @Override
            public void onError(RetrofitException error) {
              Store.getInstance(TranslationStore.class).add(TranslationInfo.createOriginStateInfo(uniqueId));
              adapterModel.addMessage(message, chatId);
            }

            @Override
            public void onNext(TranslationRepo repo) {
              if (repo != null && repo.getTranslatedMessage() != null) {
                TranslationInfo info = new TranslationInfo(uniqueId, repo.getTranslatedMessage(), TranslationState.TRANSLATED);
                Store.getInstance(TranslationStore.class).add(info);
              } else {
                Store.getInstance(TranslationStore.class).add(TranslationInfo.createOriginStateInfo(uniqueId));
              }
              adapterModel.addMessage(message, chatId);
            }
          });
    } else {
      translationInfo.setState(TranslationState.ORIGIN.equals(translationInfo.getState())
          ? TranslationState.TRANSLATED
          : TranslationState.ORIGIN);

      Store.getInstance(TranslationStore.class).add(translationInfo);
      adapterModel.addMessage(message, chatId);
    }
  }

  @Override
  public void onWebPageClick(String url) {
    view.onWebDescriptionClick(url);
  }

  @Override
  public void onFileClick(File file) {
    view.onFileDownload(file);
  }

  @Override
  public void onImageClick(File file) {
    view.onShowPhotoAlbum(file);
  }

  private void reopenChat(ActionInput actionInput) {
    view.setInputLayoutVisibility(Const.USER_CHAT_STATE_CONTINUE_CHAT);

    Message message = actionInput.getMessage();
    message.clearForm();

    adapterModel.addMessageItem(new ChatMessageItem(message));
  }

  // Support bot logic

  private void createSupportBotUserChat(final ActionInput actionInput) {
    final SupportBotMessageItem supportBotItem = adapterModel.getSupportBotItem();
    final String supportBotId = supportBotItem != null ? supportBotItem.getSupportBotId() : null;

    if (actionInput != null && actionInput.getKey() != null && actionInput.getValue() != null && supportBotId != null) {
      setState(ChatState.USER_CHAT_LOADING, false);

      view.showProgress(ResUtils.getString(context, "ch.loading_information"));

      final SendingMessageItem sendingItem = new SendingMessageItem(chatId);
      sendingItem.setActionInput(Const.FORM_ACTION_TYPE_SUPPORT_BOT, actionInput);
      adapterModel.addMessageItem(sendingItem);

      ChannelApiManager.call(
          ChannelApiManager.get().createSupportBotUserChat(supportBotId),
          new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              L.e(error.getMessage());

              view.hideProgress();
              adapterModel.removeMessageItem(sendingItem);
            }

            @Override
            public void onSuccess(@NonNull UserChatWrapper repo) {
              view.hideProgress();

              repo.update();

              RxBus.post(new ChannelModelBus(repo.getUserChat(), true));
              RxBus.post(new ChannelModelBus(repo.getSession(), true));

              if (repo.getUserChat() != null) {
                sendingItem.setChatId(repo.getUserChat().getId());
              }

              if (repo.getMessage() != null) {
                actionInput.setMessage(repo.getMessage());
                sendingItem.setActionInput(Const.FORM_ACTION_TYPE_SUPPORT_BOT, actionInput);

                adapterModel.removeMessageItem(supportBotItem);
                adapterModel.addMessage(repo.getMessage(), repo.getUserChat().getId());
              }

              supportTempItem = sendingItem;
              isNewSupportChat = true;

              setUserChat(repo, true);
            }
          }
      );
    }
  }
}
