package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.util.LongSparseArray;

import com.zoyi.channel.plugin.android.ChannelApiManager;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat.model.MessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.type.ContentType;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.ActionInput;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.wrapper.MessageWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.okhttp3.MediaType;
import com.zoyi.okhttp3.MultipartBody;
import com.zoyi.okhttp3.RequestBody;

import java.util.*;

/**
 * Created by mika on 2017. 3. 2..
 */
public class ChatManager {

  @Nullable
  private static ChatManager chatManager;

  public static ChatManager get() {
    if (chatManager == null) {
      synchronized (ChatManager.class) {
        if (chatManager == null) {
          chatManager = new ChatManager();
        }
      }
    }
    return chatManager;
  }

  private boolean sending;
  private boolean released;

  private ArrayList<OnMessageSendListener> listeners;

  private Deque<SendingMessageItem> sendingItems;
  private LongSparseArray<SendingMessageItem> failedItems;

  private ChatManager() {
    released = false;
    sending = false;
    sendingItems = new LinkedList<>();
    failedItems = new LongSparseArray<>();
    listeners = new ArrayList<>();
  }

  public void addListener(OnMessageSendListener listener) {
    if (listeners != null && listener != null && listeners.indexOf(listener) < 0) {
      listeners.add(listener);
    }
  }

  public void removeListener(OnMessageSendListener listener) {
    if (listeners != null && listener != null) {
      listeners.remove(listener);
    }
  }

  public void send(SendingMessageItem item) {
    send(ListUtils.newArrayList(item));
  }

  public void send(List<SendingMessageItem> items) {
    if (!released) {
      sendingItems.addAll(items);
      sendWaitingMessage();
    }
  }

  public void sendImmediately(SendingMessageItem item) {
    if (!released) {
      sendingItems.addFirst(item);
      sendWaitingMessage();
    }
  }

  private void sendWaitingMessage() {
    if (!sending) {
      while (sendingItems.size() > 0) {
        SendingMessageItem item = sendingItems.poll();
        if (item != null && item.getContentType() != ContentType.NONE) {
          sending = true;
          routeItem(item);

          break;
        }
      }
    }
  }

  public void updateFailedMessages(List<SendingMessageItem> items) {
    for (SendingMessageItem item : items) {
      failedItems.put(item.getPrimaryKey(), item);
    }
  }

  public void read(@Nullable String chatId) {
    if (chatId != null) {
      ChannelApiManager.call(ChannelApiManager.get().read(chatId), new RestSubscriber<Void>());
    }
  }

  // Message item router

  private void routeItem(SendingMessageItem item) {
    switch (item.getContentType()) {
      case FILE:
        sendMediaFileMessage(item);
        break;

      case TEXT:
        sendTextMessage(item);
        break;

      case FORM:
        sendFormAction(item);
        break;
    }
  }

  private void sendFormAction(SendingMessageItem item) {
    String actionType = item.getActionType();
    ActionInput actionInput = item.getActionInput();

    if (actionType != null && actionInput != null) {
      switch (actionType) {
        case Const.FORM_ACTION_TYPE_SOLVE:
        case Const.FORM_ACTION_TYPE_CLOSE:
          switch (actionInput.getKey()) {
            case Const.FORM_ACTION_CLOSE:
              closeChat(item);
              break;

            case Const.FORM_ACTION_LIKE:
              reviewChat(item, Const.REVIEW_LIKE);
              break;

            case Const.FORM_ACTION_DISLIKE:
              reviewChat(item, Const.REVIEW_DISLIKE);
              break;

            default:
              sendFormMessage(item);
              break;
          }
          break;

        case Const.FORM_ACTION_TYPE_SUPPORT_BOT:
          sendSupportBotForm(item);
          break;

        default:
          sendFormMessage(item);
          break;
      }
    }
  }

  // Api functions

  private void sendTextMessage(SendingMessageItem item) {
    String chatId = item.getChatId();

    if (chatId != null) {
      RequestBody body = RequestUtils.form()
          .set("message", item.getText())
          .set("requestId", item.getRequestId())
          .create();

      ChannelApiManager.call(
          ChannelApiManager.get().sendMessage(chatId, body),
          createMessageSubscriber(item)
      );
    }
  }

  private void sendMediaFileMessage(SendingMessageItem item) {
    String chatId = item.getChatId();

    if (chatId != null) {
      MultipartBody.Part body = RequestUtils.makeMultipart(item.getFilePath());
      RequestBody timestamp = RequestBody.create(MediaType.parse("text/plane"), item.getRequestId());

      if (body == null) {
        onSendFail(new Exception("Not supported type"), item);
        return;
      }

      ChannelApiManager.call(
          ChannelApiManager.get().uploadFile(chatId, body, timestamp),
          createMessageSubscriber(item)
      );
    }
  }

  private void sendFormMessage(SendingMessageItem item) {
    String chatId = item.getChatId();
    ActionInput actionInput = item.getActionInput();

    if (chatId != null && actionInput != null) {
      Map<String, String> submit = new HashMap<>();
      submit.put("id", actionInput.getMessageId());
      submit.put("key", actionInput.getKey());

      RequestBody body = RequestUtils.form()
          .set("message", item.getText())
          .set("requestId", item.getRequestId())
          .set("submit", submit)
          .create();

      ChannelApiManager.call(
          ChannelApiManager.get().sendMessage(chatId, body),
          createMessageSubscriber(item)
      );
    }
  }

  private void closeChat(SendingMessageItem item) {
    String chatId = item.getChatId();
    ActionInput actionInput = item.getActionInput();

    if (chatId != null && actionInput != null) {
      ChannelApiManager.call(
          ChannelApiManager.get().closeUserChat(chatId, actionInput.getMessageId(), item.getRequestId()),
          createMessageSubscriber(item)
      );
    }
  }

  private void reviewChat(SendingMessageItem item, @NonNull String review) {
    String chatId = item.getChatId();
    ActionInput actionInput = item.getActionInput();

    if (chatId != null && actionInput != null) {
      ChannelApiManager.call(
          ChannelApiManager.get().reviewUserChat(chatId, actionInput.getMessageId(), review, item.getRequestId()),
          createMessageSubscriber(item)
      );
    }
  }

  private void sendSupportBotForm(SendingMessageItem item) {
    String chatId = item.getChatId();
    ActionInput actionInput = item.getActionInput();

    if (chatId != null && actionInput != null) {
      ChannelApiManager.call(
          ChannelApiManager.get().sendFirstAction(chatId, actionInput.getMessageId(), actionInput.getKey(), item.getRequestId()),
          createMessageSubscriber(item)
      );
    }
  }

  // Finish callbacks

  private void onSendSuccess(Message message) {
    if (!released) {
      if (listeners != null) {
        for (OnMessageSendListener listener : listeners) {
          listener.onSendSuccess(message);
        }
      }
      sending = false;
      sendWaitingMessage();
    }
  }

  private void onSendFail(Exception ex, SendingMessageItem item) {
    if (ex != null) {
      L.e(ex.getMessage());
    }

    if (!released) {
      item.setSending(false);
      failedItems.put(item.getPrimaryKey(), item);

      if (listeners != null) {
        for (OnMessageSendListener listener : listeners) {
          listener.onSendFail(item);
        }
      }

      sending = false;
      sendWaitingMessage();
    }
  }

  // Default subscriber callback generator

  private RestSubscriber<MessageWrapper> createMessageSubscriber(final SendingMessageItem item) {
    return new RestSubscriber<MessageWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        onSendFail(error, item);
      }

      @Override
      public void onSuccess(@NonNull MessageWrapper repo) {
        repo.update();
        onSendSuccess(repo.getMessage());
      }
    };
  }

  // Failed item logic

  public void removeSendingFailedItem(Long createdAt) {
    if (createdAt != null) {
      failedItems.remove(createdAt);
    }
  }

  public List<MessageItem> getSendingItems(@Nullable String chatId) {
    List<MessageItem> items = new ArrayList<>();

    for (SendingMessageItem item : sendingItems) {
      if (CompareUtils.isSame(chatId, item.getChatId())) {
        items.add(item);
      }
    }
    return items;
  }

  public List<MessageItem> getSendingFailedItems(@Nullable String chatId) {
    List<MessageItem> items = new ArrayList<>();

    for (int i = 0; i < failedItems.size(); i++) {
      SendingMessageItem item = failedItems.valueAt(i);

      if (CompareUtils.isSame(chatId, item.getChatId())) {
        items.add(item);
      }
    }
    return items;
  }

  // Release function

  public static void release() {
    if (chatManager != null) {
      chatManager.released = true;
      chatManager.sendingItems.clear();
      chatManager.failedItems.clear();

      chatManager = null;
    }
  }
}
